<?php

declare(strict_types=1);

namespace Drupal\commerce_irpaymentpack\Banks;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Handles Zibal Gateway transactions
 *
 */
class Zibal {

  use StringTranslationTrait;

  protected string $merchant;
  protected int $amount;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /** REST API Endpoints **/
  protected const API_REQUEST_URL = 'https://gateway.zibal.ir/v1/request';
  protected const API_VERIFY_URL = 'https://gateway.zibal.ir/v1/verify';
  protected const API_INQUIRY_URL = 'https://gateway.zibal.ir/v1/inquiry';
  protected const GATEWAY_URL = 'https://gateway.zibal.ir/start/';

  /**
   * Constructor.
   *
   * @param string $merchant
   *   The merchant key provided by Zibal.
   * @param int $amount
   *   The amount in Rials.
   * @param \Drupal\Core\Logger\LoggerChannelInterface|null $logger
   *   The logger channel.
   */
  public function __construct(string $merchant, int $amount, LoggerChannelInterface $logger = NULL) {
    $this->merchant = $merchant;
    $this->amount = $amount;
    $this->logger = $logger ?: \Drupal::logger('zibal_gateway');
  }

  /**
   * Sends a payment request to Zibal.
   *
   * @param string $callbackUrl
   *   The callback URL.
   * @param string $description
   *   Payment description.
   * @param string $orderId
   *   Optional order ID.
   * @param string $mobile
   *   Optional mobile number.
   *
   * @return string
   *   The track ID for the payment.
   *
   * @throws \Drupal\commerce_irpaymentpack\Banks\ZibalException
   */
  public function paymentRequest(string $callbackUrl, string $description = '', string $orderId = '', string $mobile = ''): string {
    $data = [
      'merchant' => $this->merchant,
      'amount' => $this->amount,
      'callbackUrl' => $callbackUrl,
    ];

    if (!empty($description)) {
      $data['description'] = $description;
    }

    if (!empty($orderId)) {
      $data['orderId'] = $orderId;
    }

    if (!empty($mobile)) {
      $data['mobile'] = $mobile;
    }

    $response = $this->sendRequest(static::API_REQUEST_URL, $data);

    if (isset($response->result) && $response->result == 100) {
      return (string) $response->trackId;
    }

    $error_code = $response->result ?? -1;
    $message = $this->getResultMessage($error_code);
    throw new ZibalException(
      $this->t('Payment request failed: @message', ['@message' => $message]),
      $error_code,
      ['response' => $response, 'request_data' => $data]
    );
  }

  /**
   * Verifies a payment transaction.
   *
   * @param string $trackId
   *   The track ID received from Zibal.
   *
   * @return object
   *   The verification response.
   *
   * @throws \Drupal\commerce_irpaymentpack\Banks\ZibalExceptionVerification
   */
  public function verifyTransaction(string $trackId): object {
    $data = [
      'merchant' => $this->merchant,
      'trackId' => $trackId,
    ];

    $response = $this->sendRequest(static::API_VERIFY_URL, $data);

    if (isset($response->result) && $response->result == 100) {
      return $response;
    }

    $error_code = $response->result ?? -1;
    $message = $this->getResultMessage($error_code);
    throw new ZibalExceptionVerification(
      $this->t('Transaction verification failed: @message', ['@message' => $message]),
      $error_code,
      ['response' => $response, 'track_id' => $trackId]
    );
  }

  /**
   * Inquires about a payment status.
   *
   * @param string $trackId
   *   The track ID.
   *
   * @return object
   *   The inquiry response.
   *
   * @throws \Drupal\commerce_irpaymentpack\Banks\ZibalException
   */
  public function inquiryTransaction(string $trackId): object {
    $data = [
      'merchant' => $this->merchant,
      'trackId' => $trackId,
    ];

    $response = $this->sendRequest(static::API_INQUIRY_URL, $data);

    if (isset($response->result)) {
      return $response;
    }

    throw new ZibalException($this->t('Inquiry request failed'));
  }

  /**
   * Builds the gateway URL for redirecting users.
   *
   * @param string $trackId
   *   The track ID.
   *
   * @return string
   *   The complete gateway URL.
   */
  public function buildGateURL(string $trackId): string {
    return static::GATEWAY_URL . $trackId;
  }

  /**
   * Sends a request to Zibal API.
   *
   * @param string $url
   *   The API endpoint URL.
   * @param array $data
   *   The data to send.
   *
   * @return object
   *   The decoded response.
   *
   * @throws \Drupal\commerce_irpaymentpack\Banks\ZibalException
   */
  private function sendRequest(string $url, array $data): object {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($response === false || !empty($curl_error)) {
      throw new ZibalException(
        $this->t('cURL Error: @error', ['@error' => $curl_error ?: 'Unknown error']),
        NULL,
        ['url' => $url, 'curl_error' => $curl_error]
      );
    }

    if ($http_code !== 200) {
      throw new ZibalException(
        $this->t('HTTP Error: @code', ['@code' => $http_code]),
        NULL,
        ['url' => $url, 'http_code' => $http_code, 'response' => $response]
      );
    }

    $decoded = json_decode($response);
    if ($decoded === null) {
      throw new ZibalException(
        $this->t('Invalid JSON response: @response', ['@response' => $response]),
        NULL,
        ['url' => $url, 'raw_response' => $response]
      );
    }

    // Log the API response for debugging
    $this->logger->info('Zibal API Response: @response', ['@response' => $response]);

    return $decoded;
  }

  /**
   * Gets a human-readable message for result codes.
   *
   * @param int $code
   *   The result code.
   *
   * @return string
   *   The message.
   */
  private function getResultMessage(int $code): string {
    switch ($code) {
      case 100:
        return $this->t('Success');

      case 102:
        return $this->t('Merchant not found');

      case 103:
        return $this->t('Merchant is inactive');

      case 104:
        return $this->t('Merchant is invalid');

      case 105:
        return $this->t('Amount must be greater than 1,000 Rials');

      case 106:
        return $this->t('Invalid callback URL (must start with http or https)');

      case 113:
        return $this->t('Amount exceeds transaction limit');

      case 201:
        return $this->t('Already verified');

      case 202:
        return $this->t('Order not paid or unsuccessful');

      case 203:
        return $this->t('Invalid track ID');

      default:
        return $this->t('Unknown error occurred');
    }
  }
}