<?php

namespace Drupal\commerce_irpaymentpack\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_irpaymentpack\Banks\Zarinpal;
use Drupal\commerce_irpaymentpack\Banks\ZarinpalException;
use Drupal\commerce_irpaymentpack\Banks\ZarinpalExceptionNok;
use Drupal\commerce_irpaymentpack\Banks\ZarinpalExceptionVerification;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Annotation\CommercePaymentGateway;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * The commerce payment gateway plugin for Zarinpal
 *
 * @CommercePaymentGateway(
 *   id = "commerce_irpaymentpack_zarinpal",
 *   label = @Translation("IRPaymentPack: Zarinpal"),
 *   display_label = @Translation("Zarinpal"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_irpaymentpack\PluginForm\OffsiteRedirect\ZarinpalRedirect",
 *   },
 *   payment_method_types = {"credit_card"},
 * )
 *
 * @author "Ahmad Hejazee" <mngafa@gmail.com>
 */
class ZarinpalGateway extends OffsitePaymentGatewayBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['zarinapl_merchant_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Code'),
      '#default_value' => $this->configuration['zarinapl_merchant_code'] ?? '',
      '#description' => $this->t('The merchant code is provided by Zarinpal. In the Test mode, any arbitrary code such as 123 can be used.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      // Save configuration
      $this->configuration['zarinapl_merchant_code'] = $values['zarinapl_merchant_code'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $merchant_code = $this->configuration['zarinapl_merchant_code'] ?? '';

    if (empty($merchant_code)) {
      $this->messenger()->addError($this->t('Zarinpal is not configured properly. Please contact site administrator.'));
      throw new InvalidRequestException($this->t("Zarinpal is not configured properly. Please contact site administrator."));
    }

    // Retrieve authority from request.
    $authority = $request->query->get('Authority');
    if (!$authority) {
      throw new InvalidRequestException($this->t('No Authority parameter received from Zarinpal.'));
    }

    $status = $request->query->get('Status');
    if ($status !== 'OK') {
      $this->messenger()->addError($this->t('Transaction not successful or cancelled by user.'));
      throw new PaymentGatewayException('Transaction not successful or cancelled by user.');
    }

    // Get amount in Toman.
    $amount_toman = (int) $order->getTotalPrice()->getNumber();
    if ($order->getTotalPrice()->getCurrencyCode() === 'IRR') {
      $amount_toman /= 10;
    }

    // Load payment entity.
    $payment_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment');
    $payments = $payment_storage->loadByProperties([
      'order_id' => $order->id(),
      'state' => 'authorization',
    ]);
    $payment = end($payments);

    if (!$payment) {
      throw new InvalidRequestException($this->t('Could not find the payment record.'));
    }

    try {
      $zarinpal = new Zarinpal($merchant_code, $amount_toman);
      $zarinpal->setSandbox($this->configuration['mode'] == 'test');
      $refId = $zarinpal->verifyTransaction($authority);

      $payment->setState('completed');
      $payment->setRemoteId($refId);
      $payment->save();

      $this->messenger()->addStatus($this->t('Payment was successful.'));
    } catch (ZarinpalException $e) {
      $this->messenger()->addError($this->t('No valid authority sent by Zarinpal. Payment failed.'));
      throw new PaymentGatewayException('No valid authority sent by Zarinpal. Payment failed.');
    } catch (ZarinpalExceptionNok $e) {
      $this->messenger()->addError($this->t('Transaction not successful or cancelled by user.'));
      throw new PaymentGatewayException('Transaction not successful or cancelled by user.');
    } catch (ZarinpalExceptionVerification $e) {
      \Drupal\Component\Utility\DeprecationHelper::backwardsCompatibleCall(\Drupal::VERSION, '10.1.0', fn() => \Drupal\Core\Utility\Error::logException(\Drupal::logger('commerce_irpaymentpack'), $e), fn() => watchdog_exception('commerce_irpaymentpack', $e));
      $this->messenger()->addError($e->getMessage());
      throw new PaymentGatewayException($e->getMessage());
    } catch (\Exception $e) {
      \Drupal\Component\Utility\DeprecationHelper::backwardsCompatibleCall(\Drupal::VERSION, '10.1.0', fn() => \Drupal\Core\Utility\Error::logException(\Drupal::logger('commerce_irpaymentpack'), $e), fn() => watchdog_exception('commerce_irpaymentpack', $e));
      $this->messenger()->addError($this->t('An unexpected error occurred.'));
      throw new PaymentGatewayException('An unexpected error occurred.');
    }
  }
}
