<?php

declare(strict_types=1);

namespace Drupal\commerce_irpaymentpack\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_irpaymentpack\Banks\Zibal;
use Drupal\commerce_irpaymentpack\Banks\ZibalException;
use Drupal\commerce_irpaymentpack\Banks\ZibalExceptionVerification;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Annotation\CommercePaymentGateway;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * The commerce payment gateway plugin for Zibal
 *
 * @CommercePaymentGateway(
 *   id = "commerce_irpaymentpack_zibal",
 *   label = @Translation("IRPaymentPack: Zibal"),
 *   display_label = @Translation("Zibal"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_irpaymentpack\PluginForm\OffsiteRedirect\ZibalRedirect",
 *   },
 *   payment_method_types = {"credit_card"},
 * )
 *
 */
class ZibalGateway extends OffsitePaymentGatewayBase implements ContainerFactoryPluginInterface {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->loggerFactory = $container->get('logger.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['zibal_merchant_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Key'),
      '#default_value' => $this->configuration['zibal_merchant_key'] ?? '',
      '#description' => $this->t('The merchant key is provided by Zibal. For testing, use "zibal" as the merchant key.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      // Save configuration
      $this->configuration['zibal_merchant_key'] = $values['zibal_merchant_key'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $merchant_key = $this->configuration['zibal_merchant_key'] ?? '';

    if (empty($merchant_key)) {
      $this->messenger()->addError($this->t('Zibal is not configured properly. Please contact site administrator.'));
      throw new InvalidRequestException($this->t("Zibal is not configured properly. Please contact site administrator."));
    }

    // Check if payment was successful
    $success = $request->query->get('success');
    if ($success != '1') {
      $this->messenger()->addError($this->t('Transaction not successful or cancelled by user.'));
      throw new PaymentGatewayException('Transaction not successful or cancelled by user.');
    }

    // Retrieve trackId from request
    $trackId = $request->query->get('trackId');
    if (!$trackId) {
      throw new InvalidRequestException($this->t('No trackId parameter received from Zibal.'));
    }

    // Get amount in Rials
    $amount_rials = (int) $order->getTotalPrice()->getNumber();
    if ($order->getTotalPrice()->getCurrencyCode() === 'TMN') {
      $amount_rials *= 10;
    }

    // Load payment entity
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payments = $payment_storage->loadByProperties([
      'order_id' => $order->id(),
      'state' => 'authorization',
    ]);
    $payment = end($payments);

    if (!$payment) {
      throw new InvalidRequestException($this->t('Could not find the payment record.'));
    }

    try {
      $zibal = new Zibal($merchant_key, $amount_rials, $this->loggerFactory->get('zibal_gateway'));
      $verify_result = $zibal->verifyTransaction($trackId);

      $payment->setState('completed');
      $payment->setRemoteId($verify_result->refNumber ?? $trackId);
      $payment->save();

      $this->messenger()->addStatus($this->t('Payment was successful.'));
    } catch (ZibalException $e) {
      $this->messenger()->addError($this->t('Payment request failed. @message', ['@message' => $e->getMessage()]));
      throw new PaymentGatewayException($e->getMessage());
    } catch (ZibalExceptionVerification $e) {
      \Drupal\Core\Utility\Error::logException($this->loggerFactory->get('commerce_irpaymentpack'), $e);
      $this->messenger()->addError($e->getMessage());
      throw new PaymentGatewayException($e->getMessage());
    } catch (\Exception $e) {
      \Drupal\Core\Utility\Error::logException($this->loggerFactory->get('commerce_irpaymentpack'), $e);
      $this->messenger()->addError($this->t('An unexpected error occurred.'));
      throw new PaymentGatewayException('An unexpected error occurred.');
    }
  }
}