<?php

declare(strict_types=1);

namespace Drupal\commerce_irpaymentpack\PluginForm\OffsiteRedirect;

use Drupal\commerce_irpaymentpack\Banks\Zibal;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * This class is used to build the offsite redirect form for Zibal gateway
 *
 */
class ZibalRedirect extends PaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $paymentStorage;

  /**
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $loggerFactory;

  /**
   * Constructor.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->paymentStorage = $entity_type_manager->getStorage('commerce_payment');
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory->get('zibal_gateway');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->getEntity();
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $payment->getOrder();

    // Get amount in Rials
    $amount_rials = (int) $payment->getAmount()->getNumber();
    // Convert Toman to Rials if needed
    if ($payment->getAmount()->getCurrencyCode() == 'TMN') {
      $amount_rials *= 10;
    }

    // Load gateway configuration
    $gateway_config = $payment->getPaymentGateway()->getPlugin()->getConfiguration();
    $merchant_key = $gateway_config['zibal_merchant_key'] ?? '';

    // If the configuration parameters are empty, display an error
    if (empty($merchant_key)) {
      $this->messenger->addError($this->t('Zibal is not configured properly. Please contact site administrator.'));
      return $form;
    }

    try {
      // Log debug information
      $this->loggerFactory->info('Initializing Zibal payment. Merchant: @merchant, Amount: @amount, Order: @order', [
        '@merchant' => $merchant_key,
        '@amount' => $amount_rials,
        '@order' => $order->id(),
      ]);

      $zibal = new Zibal($merchant_key, $amount_rials, $this->loggerFactory);

      // Prepare description
      $description = (string) $this->t('Payment for order @order_id from @store', [
        '@order_id' => $order->getOrderNumber(),
        '@store' => $order->getStore()->label(),
      ]);

      // Get customer mobile if available
      $mobile = '';
      if ($order->getBillingProfile() && $order->getBillingProfile()->hasField('field_phone')) {
        $phone_field = $order->getBillingProfile()->get('field_phone');
        if (!$phone_field->isEmpty()) {
          $mobile = $phone_field->value;
        }
      }

      $this->loggerFactory->info('Making payment request to Zibal. Callback URL: @url', [
        '@url' => $form['#return_url'],
      ]);

      $trackId = $zibal->paymentRequest(
        $form['#return_url'],
        $description,
        (string) $order->id(),
        $mobile
      );

      // Create a new payment but with state 'Authorization' not completed
      // On payment return, if everything is ok, the state of this new payment will be converted to 'Completed'
      $new_payment = $this->paymentStorage->create([
        'state' => 'authorization',
        'amount' => $order->getTotalPrice(),
        'payment_gateway' => $payment->getPaymentGatewayId(),
        'order_id' => $order->id(),
        'remote_state' => $trackId,
      ]);
      $new_payment->save();

      // Redirect to Zibal gateway
      $gateway_url = $zibal->buildGateURL($trackId);
      return $this->buildRedirectForm($form, $form_state, $gateway_url, [], parent::REDIRECT_POST);
    } catch (\Exception $e) {
      $this->messenger->addError($this->t('Cannot initialize the payment.'));
      $this->loggerFactory->warning('Cannot initialize payment. Order ID: @id. Error: @error', [
        '@id' => $order->id(),
        '@error' => $e->getMessage(),
      ]);

      return $form;
    }
  }
}