<?php

namespace Drupal\commerce_klarna_shipping\EventSubscriber;

use Drupal\commerce_klarna\Event\KlarnaShipmentsEvent;
use Drupal\commerce_shipping\ShipmentManagerInterface;
use Drupal\commerce_shipping\ShippingOrderManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Creates shipments within Klarna express checkout.
 */
class KlarnaExpressShipments implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(protected ShippingOrderManagerInterface $shippingOrderManager, protected ShipmentManagerInterface $shipmentManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_klarna.express_checkout_shipments' => ['expressCheckout', -100],
    ];
  }

  /**
   * Creates shipments within Klarna express checkout.
   *
   * @param \Drupal\commerce_klarna\Event\KlarnaShipmentsEvent $event
   *   The event.
   */
  public function expressCheckout(KlarnaShipmentsEvent $event): void {
    $order = $event->getOrder();
    $shipping_profile = $event->getShippingProfile();

    if ($order->get('shipments')->isEmpty()) {
      $shipments = $this->shippingOrderManager->pack($order);

      foreach ($shipments as $key => $shipment) {
        $shipment->setShippingProfile($shipping_profile);
        $rates = $this->shipmentManager->calculateRates($shipment);

        // There is no rates for shipping. "clear" the rate...
        if (empty($rates)) {
          $shipment->clearRate();
          continue;
        }
        $rate = $this->shipmentManager->selectDefaultRate($shipment, $rates);
        $this->shipmentManager->applyRate($shipment, $rate);
        $shipment->save();
        $shipments[$key] = $shipment;
      }

      $event->setShipments($shipments);
    }
  }

}
