<?php

namespace Drupal\commerce_klarna\EventSubscriber;

use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Cancel Klarna order upon order cancellation.
 */
class OrderCancelSubscriber implements EventSubscriberInterface {

  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_order.cancel.post_transition' => ['onOrderCancel', -100],
    ];
  }

  /**
   * Cancel Klarna order and void payments.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The event.
   */
  public function onOrderCancel(WorkflowTransitionEvent $event): void {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getEntity();

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    if ($payment_method = $order->get('payment_method')->entity) {
      $plugin = $payment_method->getPaymentGateway()->getPlugin();
      if ($plugin instanceof KlarnaPaymentsInterface) {
        /** @var \Drupal\commerce_payment\PaymentStorageInterface $payment_storage */
        $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

        $payments = $payment_storage->loadMultipleByOrder($order);
        foreach ($payments as $payment) {
          if ($payment->getState()->getId() !== 'authorization_voided') {
            $plugin->voidPayment($payment);
          }
        }
      }
    }
  }

}
