<?php

namespace Drupal\commerce_klarna;

use Drupal\address\AddressInterface;
use Drupal\commerce_klarna\Event\KlarnaEvents;
use Drupal\commerce_klarna\Event\KlarnaRequestEvent;
use Drupal\commerce_klarna\Exception\KlarnaException;
use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaMerchantCardInterface;
use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\commerce_order\AdjustmentTransformerInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_price\Calculator;
use Drupal\commerce_price\Entity\Currency;
use Drupal\commerce_price\MinorUnitsConverterInterface;
use Drupal\commerce_price\Price;
use Drupal\commerce_price\RounderInterface;
use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Url;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\EventDispatcher\EventDispatcherInterface;

/**
 * {@inheritdoc}
 */
class KlarnaManager implements KlarnaManagerInterface {

  use LoggerChannelTrait;

  /**
   * Klarna manager constructor.
   */
  public function __construct(protected Client $client, protected ModuleExtensionList $moduleExtensionList, protected MinorUnitsConverterInterface $minorUnitsConverter, protected EventDispatcherInterface $eventDispatcher, protected LanguageManagerInterface $languageManager, protected UuidInterface $uuid, protected AdjustmentTransformerInterface $adjustmentTransformer, protected RounderInterface $rounder) {}

  /**
   * {@inheritdoc}
   */
  public function paymentSession(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments): array {
    if ($session = $order->getData(KlarnaManagerInterface::KLARNA_ORDER_KEY)) {
      $this->updatePaymentSession($order, $session['session_id'], $klarna_payments);
      return $session;
    }
    return $this->createPaymentSession($order, $klarna_payments);
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentSession(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments): array {
    $payload = $this->getOrderPayload($order, KlarnaEvents::CREATE_PAYMENT_SESSION, $klarna_payments);
    return $this->apiRequest('/payments/v1/sessions', $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function updatePaymentSession(OrderInterface $order, string $session_id, KlarnaPaymentsInterface $klarna_payments): void {
    $payload = $this->getOrderPayload($order, KlarnaEvents::UPDATE_PAYMENT_SESSION, $klarna_payments);
    $this->apiRequest(sprintf('/payments/v1/sessions/%s', $session_id), $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentSession(OrderInterface $order, string $session_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    return $this->apiRequest(sprintf('/payments/v1/sessions/%s', $session_id), $klarna_payments);
  }

  /**
   * {@inheritdoc}
   */
  public function createOrder(OrderInterface $order, string $authorization_token): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    $payload = $this->getOrderPayload($order, KlarnaEvents::CREATE_ORDER_REQUEST, $klarna_payments);
    return $this->apiRequest(sprintf('/payments/v1/authorizations/%s/order', $authorization_token), $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function updateOrder(OrderInterface $order, string $klarna_order_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    $payload = $this->getOrderPayload($order, KlarnaEvents::CREATE_ORDER_REQUEST, $klarna_payments);
    return $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/authorization', $klarna_order_id), $klarna_payments, $payload, 'PATCH');
  }

  /**
   * {@inheritdoc}
   */
  public function getOrder(PaymentInterface $payment): array {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    return $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s', $payment->getRemoteId()), $klarna_payments);
  }

  /**
   * {@inheritdoc}
   */
  public function captureOrder(PaymentInterface $payment, Price $amount): array {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    $payload = [
      'captured_amount' => $this->minorUnitsConverter->toMinorUnits($amount),
    ];
    $event = new KlarnaRequestEvent($payment->getOrder(), $payload);
    $this->eventDispatcher->dispatch($event, KlarnaEvents::PAYMENT_CAPTURE_REQUEST);
    $payload = $event->getPayload();
    return $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/captures', $payment->getRemoteId()), $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function cancelOrder(PaymentInterface $payment): void {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/cancel', $payment->getRemoteId()), $klarna_payments, [], 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function refundOrder(PaymentInterface $payment, Price $amount): void {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    $order_id = $payment->getRemoteId();
    $payload = ['refunded_amount' => $this->minorUnitsConverter->toMinorUnits($amount)];
    $event = new KlarnaRequestEvent($payment->getOrder(), $payload);
    $this->eventDispatcher->dispatch($event, KlarnaEvents::PAYMENT_REFUND_REQUEST);
    $payload = $event->getPayload();
    $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/refunds', $order_id), $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function listOrderCaptures(PaymentInterface $payment): array {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    return $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/captures/', $payment->getRemoteId()), $klarna_payments, [], 'GET');
  }

  /**
   * {@inheritdoc}
   */
  public function addShippingInformation(PaymentInterface $payment, array $shipping_info): array {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaPaymentsInterface);
    return $this->apiRequest(sprintf('/ordermanagement/v1/orders/%s/shipping-info', $payment->getRemoteId()), $klarna_payments, $shipping_info, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function cancelAuthorization(KlarnaPaymentsInterface $klarna_payments, string $authorization_token): void {
    $this->apiRequest(sprintf('/payments/v1/authorizations/%s', $authorization_token), $klarna_payments, [], 'DELETE');
  }

  /**
   * {@inheritdoc}
   */
  public function createCardPromise(OrderInterface $order, string $klarna_order_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    $payload = [
      'order_id' => $klarna_order_id,
      'cards' => [
        [
          'amount' => $this->minorUnitsConverter->toMinorUnits($order->getTotalPrice()),
          'currency' => $order->getTotalPrice()->getCurrencyCode(),
          'reference' => $order->id(),
        ],
      ],
    ];

    $event = new KlarnaRequestEvent($order, $payload);
    $this->eventDispatcher->dispatch($event, KlarnaEvents::CREATE_CARD_PROMISE);
    $payload = $event->getPayload();
    return $this->apiRequest('/merchantcard/v3/promises', $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function getCardPromise(OrderInterface $order, string $promise_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    return $this->apiRequest(sprintf('/merchantcard/v3/promises/%s', $promise_id), $klarna_payments);
  }

  /**
   * {@inheritdoc}
   */
  public function createCardSettlement(OrderInterface $order, string $klarna_order_id, ?string $klarna_promise_id = NULL): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    $payment_method = $order->get('payment_method')->entity;
    $payload = [
      'order_id' => $klarna_order_id,
      'promise_id' => $klarna_promise_id ?: $payment_method->get('klarna_promise_id')->value,
      'key_id' => $payment_method->getPaymentGateway()->getPlugin()->getKeyId(),
    ];
    return $this->apiRequest('/merchantcard/v3/settlements', $klarna_payments, $payload, 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function getCardSettlement(OrderInterface $order, string $settlement_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    return $this->apiRequest(sprintf('/merchantcard/v3/settlements/%s', $settlement_id), $klarna_payments, [], 'GET', ['KeyId' => $klarna_payments->getKeyId()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCardSettlementByOrderId(OrderInterface $order, string $klarna_order_id): array {
    $klarna_payments = $order->get('payment_gateway')->entity->getPlugin();
    return $this->apiRequest(sprintf('/merchantcard/v3/settlements/order/%s', $klarna_order_id), $klarna_payments, [], 'GET', ['KeyId' => $klarna_payments->getKeyId()]);
  }

  /**
   * {@inheritdoc}
   */
  public function cancelMerchantCardOrder(PaymentInterface $payment): void {
    $klarna_payments = $payment->getPaymentGateway()->getPlugin();
    assert($klarna_payments instanceof KlarnaMerchantCardInterface);
    $this->apiRequest(sprintf('/merchantcard/v3/orders/%s/cancel-request', $payment->getRemoteId()), $klarna_payments, [], 'POST');
  }

  /**
   * {@inheritdoc}
   */
  public function getOrderPayload(OrderInterface $order, string $event_name, $klarna_payments = NULL): array {
    $payload = [
      'acquiring_channel' => 'ECOMMERCE',
      'merchant_reference1' => $order->id(),
      'order_amount' => $this->minorUnitsConverter->toMinorUnits($order->getTotalPrice()),
      'purchase_currency' => $order->getTotalPrice()->getCurrencyCode(),
      'intent' => 'buy',
    ];

    if ($payment_gateway = $order->get('payment_gateway')->entity) {
      $payload['merchant_urls']['notify'] = Url::fromRoute('commerce_payment.notify', ['commerce_payment_gateway' => $payment_gateway->id()])->toString();
    }

    foreach ($order->getItems() as $item) {
      // Skip malformed order items.
      if (!$item->getUnitPrice()) {
        continue;
      }

      // Fallback to the order item ID.
      $reference = $item->id();
      $purchased_entity = $item->getPurchasedEntity();

      // Send the "SKU" as the "reference" for product variations.
      if ($purchased_entity instanceof ProductVariationInterface) {
        $reference = $purchased_entity->getSku();
      }
      $name = trim($item->label());

      $order_line = [
        'reference' => mb_substr($reference, 0, 64),
        'name' => $name,
        'quantity' => (int) $item->getQuantity(),
        'total_amount' => $this->minorUnitsConverter->toMinorUnits($item->getAdjustedTotalPrice(['promotion'])),
        'unit_price' => $this->minorUnitsConverter->toMinorUnits($item->getUnitPrice()),
        'total_discount_amount' => 0,
        'total_tax_amount' => 0,
        'tax_rate' => 0,
      ];

      if ($klarna_payments) {
        $configuration = $klarna_payments->getConfiguration();
        if (!empty($configuration['image_field']) && $purchased_entity->hasField($configuration['image_field'])) {
          /** @var \Drupal\file\FileInterface $file */
          $file = $purchased_entity->get($configuration['image_field'])?->entity;
          if ($file) {
            $order_line['image_url'] = $file->createFileUrl(FALSE);
          }
        }
      }

      // Only pass included tax adjustments (i.e "VAT"), non-included tax
      // adjustments are passed separately (i.e "Sales tax").
      $tax_adjustments = $item->getAdjustments(['tax']);
      if ($tax_adjustments && $tax_adjustments[0]->isIncluded()) {
        $tax_rate = $tax_adjustments[0]->getPercentage();
        $order_line = array_merge($order_line, [
          'tax_rate' => (int) Calculator::multiply($tax_rate, '10000'),
          'total_tax_amount' => $this->minorUnitsConverter->toMinorUnits($tax_adjustments[0]->getAmount()),
        ]);
      }

      // Check if we have promotion adjustments.
      $promotion_adjustments = $item->getAdjustments(['promotion']);
      if ($promotion_adjustments  && ($promotions_total = $this->getAdjustmentsTotal($promotion_adjustments, [], TRUE))) {
        $order_line['total_discount_amount'] = $this->minorUnitsConverter->toMinorUnits($promotions_total->multiply('-1'));
      }
      $payload['order_lines'][] = $order_line;
    }

    $adjustment_types_mapping = [
      'tax' => 'sales_tax',
      'fee' => 'surcharge',
    ];
    // Shipping is handled separately.
    $exclude_adjustment_types = ['shipping', 'promotion', 'shipping_promotion'];
    $adjustments = $order->collectAdjustments();
    $adjustments = $adjustments ? $this->adjustmentTransformer->processAdjustments($adjustments) : [];
    foreach ($adjustments as $adjustment) {
      $adjustment_type = $adjustment->getType();
      // Skip included adjustments and the ones we don't handle.
      if ($adjustment->isIncluded() ||
        in_array($adjustment_type, $exclude_adjustment_types)) {
        continue;
      }
      $order_line = [
        // The reference is limited to 64 characters.
        'reference' => $adjustment->getSourceId() ? mb_substr($adjustment->getSourceId(), 0, 64) : '',
        'name' => $adjustment->getLabel(),
        'quantity' => 1,
        'tax_rate' => 0,
        'total_tax_amount' => 0,
        'unit_price' => $this->minorUnitsConverter->toMinorUnits($adjustment->getAmount()),
        'total_amount' => $this->minorUnitsConverter->toMinorUnits($adjustment->getAmount()),
      ];

      // Map the adjustment type to the type expected by Klarna.
      if (isset($adjustment_types_mapping[$adjustment_type])) {
        $order_line['type'] = $adjustment_types_mapping[$adjustment_type];
      }

      $payload['order_lines'][] = $order_line;
    }
    if ($order->hasField('shipments') && !$order->get('shipments')->isEmpty()) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface[] $shipments */
      $shipments = $order->get('shipments')->referencedEntities();
      foreach ($shipments as $shipment) {
        $shipment_amount = $shipment->getAmount();
        if (empty($shipment_amount)) {
          continue;
        }
        // Check if there are included tax adjustments.
        $tax_adjustments = array_filter($shipment->getAdjustments(['tax']), function ($adjustment) {
          // Skip "0" tax adjustments, there's no need to distribute the
          // shipment amount in this case.
          return $adjustment->isIncluded() && !$adjustment->getAmount()->isZero();
        });
        $shipping_line = [
          'name' => 'Shipping',
          'quantity' => 1,
          'type' => 'shipping_fee',
        ];
        // If there are no included tax adjustments, there's no need to split
        // the shipment amounts across multiple VAT rates.
        if (!$tax_adjustments) {
          $shipping_line += [
            'total_amount' => $this->minorUnitsConverter->toMinorUnits($shipment->getAdjustedAmount(['shipping_promotion'])),
            'tax_rate' => 0,
            'total_tax_amount' => 0,
          ];
          // The unit price shouldn't include discounts, we cannot use the
          // original amount as the "unit_price" because it won't reflect the
          // correct price if shipping rates were altered.
          $unit_price = $shipment->getAmount();
          $promotion_adjustments = $shipment->getAdjustments(['shipping_promotion']);
          if ($promotion_adjustments && ($promotions_total = $this->getAdjustmentsTotal($promotion_adjustments, [], TRUE))) {
            $shipping_line['total_discount_amount'] = $this->minorUnitsConverter->toMinorUnits($promotions_total->multiply('-1'));
          }
          $shipping_line['unit_price'] = $this->minorUnitsConverter->toMinorUnits($unit_price);
          $payload['order_lines'][] = $shipping_line;
        }
        else {
          $shipment_amounts = $this->splitShipping($order, $shipment);
          /** @var \Drupal\commerce_price\Price[] $amounts */
          foreach ($shipment_amounts as $amounts) {
            $payload['order_lines'][] = $shipping_line + [
              'unit_price' => $this->minorUnitsConverter->toMinorUnits($amounts['unit_amount']),
              'total_amount' => $this->minorUnitsConverter->toMinorUnits($amounts['adjusted_shipment_amount']),
              'tax_rate' => $amounts['tax_amount']->isZero() ? 0 : (int) Calculator::multiply($amounts['tax_rate'], '10000'),
              'total_tax_amount' => $this->minorUnitsConverter->toMinorUnits($amounts['tax_amount']),
              'total_discount_amount' => $this->minorUnitsConverter->toMinorUnits($amounts['promotions_amount']),
            ];
          }
        }
      }
    }

    // Default purchase country to store country.
    $store_address = $order->getStore()->getAddress();
    $purchase_country = $store_address->getCountryCode();

    $profiles = $order->collectProfiles();

    foreach ($profiles as $type => $profile) {
      if (!$profile->get('address')->isEmpty()) {
        $address = $profile->get('address')->first();
        switch ($type) {
          case 'billing':
            $payload['billing_address'] = $this->formatAddress($address, $order->getEmail());
            if (!empty($payload['billing_address']['country'])) {
              $purchase_country = $payload['billing_address']['country'];
            }
            break;

          case 'shipping':
            $payload['shipping_address'] = $this->formatAddress($address, $order->getEmail());
            break;
        }
      }
    }

    $payload['purchase_country'] = $purchase_country;
    if ($payment_gateway) {
      $payload['locale'] = $this->getLocale($purchase_country, $this->languageManager->getCurrentLanguage()->getId());
    }

    $event = new KlarnaRequestEvent($order, $payload);
    $this->eventDispatcher->dispatch($event, $event_name);

    return $event->getPayload();
  }

  /**
   * Split shipment amounts.
   */
  protected function splitShipping(OrderInterface $order, ShipmentInterface $shipment): array {
    // Group order items by tax percentage.
    $groups = [];
    foreach ($order->getItems() as $order_item) {
      $order_item_total = $order_item->getTotalPrice();
      $order_item_tax_adjustments = $order_item->getAdjustments(['tax']);
      if ($order_item_tax_adjustments) {
        $order_item_tax_adjustment = reset($order_item_tax_adjustments);
        $percentage = $order_item_tax_adjustment->getPercentage();
      }
      else {
        $percentage = '0';
      }
      if (!isset($groups[$percentage])) {
        $groups[$percentage] = $order_item_total;
      }
      else {
        $previous_total = $groups[$percentage];
        $groups[$percentage] = $previous_total->add($order_item_total);
      }
    }
    // Sort by percentage descending.
    krsort($groups, SORT_NUMERIC);
    // Calculate the ratio of each group.
    $subtotal = $order->getSubtotalPrice()->getNumber();
    $ratios = [];
    foreach ($groups as $percentage => $order_item_total) {
      // Cannot calculate the ratio if the order item total is 0 for this tax
      // percentage group.
      if ($order_item_total->isZero()) {
        $ratios[$percentage] = '0';
        continue;
      }
      $ratios[$percentage] = $order_item_total->divide($subtotal)->getNumber();
    }
    // We need to send the "unit_price" to Klarna (that includes tax and
    // excludes discount).
    // That is what the "original_amount" is supposed to be designed for except
    // that doesn't take into account the "altered" amounts.
    $unit_price = $shipment->getAmount();
    // Check if we have promotion adjustments.
    $promotion_adjustments = $shipment->getAdjustments(['shipping_promotion']);
    $promotion_total = new Price('0', $shipment->getAmount()->getCurrencyCode());
    /** @var \Drupal\commerce_order\Adjustment $adjustment */
    foreach ($promotion_adjustments as $adjustment) {
      $promotion_amount = $adjustment->getAmount()->multiply(-1);
      $promotion_total = $promotion_total->add($promotion_amount);
      // Add any included promotion adjustment to the unit price.
      if ($adjustment->isIncluded()) {
        $unit_price = $unit_price->add($promotion_amount);
      }
    }
    $promotion_amounts = [];
    if (!$promotion_total->isZero()) {
      $promotion_amounts = $this->allocate($promotion_total, $ratios);
    }

    $return = [];
    // The unit price should not include any discount.
    $unit_amounts = $this->allocate($unit_price, $ratios);
    $adjusted_shipment_amounts = $this->allocate($shipment->getAdjustedAmount(['shipping_promotion']), $ratios);
    $shipment_tax_adjustments = $shipment->getAdjustments(['tax']);
    // Re-key the tax adjustments by tax percentage for easier retrieval.
    $keyed_tax_adjustments = [];
    // Because there can be tax exempt items in the order affecting the ratios,
    // we need to recalculate the tax amounts, otherwise Klarna would complain
    // that the tax amount doesn't match the tax rate passed.
    foreach ($shipment_tax_adjustments as $adjustment) {
      $percentage = $adjustment->getPercentage();
      if (!isset($adjusted_shipment_amounts[$percentage])) {
        continue;
      }
      $tax_amount = $this->calculateTaxAmount($adjusted_shipment_amounts[$percentage], $percentage, $adjustment->isIncluded());
      $tax_amount = $this->rounder->round($tax_amount);
      $keyed_tax_adjustments[$percentage] = $tax_amount;
    }
    $zero_price = new Price('0', $shipment->getAmount()->getCurrencyCode());
    foreach ($ratios as $percentage => $ratio) {
      $return[$percentage] = [
        'promotions_amount' => $promotion_amounts[$percentage] ?? $zero_price,
        'unit_amount' => $unit_amounts[$percentage] ?? $zero_price,
        'adjusted_shipment_amount' => $adjusted_shipment_amounts[$percentage] ?? $zero_price,
        'tax_amount' => $keyed_tax_adjustments[$percentage] ?? $zero_price,
        'tax_rate' => $percentage,
      ];
    }

    return $return;
  }

  /**
   * Allocates the given amount according to a list of ratios.
   *
   * @param \Drupal\commerce_price\Price $amount
   *   The amount.
   * @param array $ratios
   *   An array of ratios, keyed by tax percentage.
   *
   * @return array
   *   An array of amounts keyed by tax percentage.
   */
  protected function allocate(Price $amount, array $ratios): array {
    $amounts = [];
    $amount_to_allocate = $amount;
    foreach ($ratios as $percentage => $ratio) {
      $individual_amount = $amount_to_allocate->multiply($ratio);
      $individual_amount = $this->rounder->round($individual_amount, PHP_ROUND_HALF_DOWN);
      // Due to rounding it is possible for the last calculated
      // per-order-item amount to be larger than the total remaining amount.
      if ($individual_amount->greaterThan($amount)) {
        $individual_amount = $amount;
      }
      $amounts[$percentage] = $individual_amount;
      $amount = $amount->subtract($individual_amount);
    }
    // The individual amounts don't add up to the full amount, distribute
    // the reminder among them.
    if (!$amount->isZero()) {
      /** @var \Drupal\commerce_price\Entity\CurrencyInterface $currency */
      $currency = Currency::load($amount->getCurrencyCode());
      $precision = $currency->getFractionDigits();
      // Use the smallest rounded currency amount (e.g. '0.01' for USD).
      $smallest_number = Calculator::divide('1', pow(10, $precision), $precision);
      $smallest_amount = new Price($smallest_number, $amount->getCurrencyCode());
      while (!$amount->isZero()) {
        foreach ($amounts as $percentage => $individual_amount) {
          $amounts[$percentage] = $individual_amount->add($smallest_amount);
          $amount = $amount->subtract($smallest_amount);
          if ($amount->isZero()) {
            break 2;
          }
        }
      }
    }

    return $amounts;
  }

  /**
   * Calculates the tax amount for the given shipping amount.
   *
   * This is copied from the commerce_shipping module.
   *
   * @param \Drupal\commerce_price\Price $amount
   *   The proportional shipping amount.
   * @param string $percentage
   *   The tax rate percentage.
   * @param bool $included
   *   Whether tax is already included in the price.
   *
   * @return \Drupal\commerce_price\Price
   *   The unrounded tax amount.
   */
  protected function calculateTaxAmount(Price $amount, string $percentage, bool $included = FALSE): Price {
    $tax_amount = $amount->multiply($percentage);
    if ($included) {
      $divisor = Calculator::add('1', $percentage);
      $tax_amount = $tax_amount->divide($divisor);
    }

    return $tax_amount;
  }

  /**
   * Calculates the total for the given adjustments.
   *
   * @param \Drupal\commerce_order\Adjustment[] $adjustments
   *   The adjustments.
   * @param string[] $adjustment_types
   *   The adjustment types to include in the calculation.
   *   Examples: fee, promotion, tax. Defaults to all adjustment types.
   * @param bool $skip_included
   *   Whether to skip included adjustments (Defaults to FALSE).
   *
   * @return \Drupal\commerce_price\Price|null
   *   The adjustments total, or NULL if no matching adjustments were found.
   */
  protected function getAdjustmentsTotal(array $adjustments, array $adjustment_types = [], $skip_included = FALSE): ?Price {
    $adjustments_total = NULL;
    $matching_adjustments = [];

    foreach ($adjustments as $adjustment) {
      if ($skip_included && $adjustment->isIncluded()) {
        continue;
      }
      if ($adjustment_types && !in_array($adjustment->getType(), $adjustment_types)) {
        continue;
      }
      $matching_adjustments[] = $adjustment;
    }
    if ($matching_adjustments) {
      $matching_adjustments = $this->adjustmentTransformer->processAdjustments($matching_adjustments);
      foreach ($matching_adjustments as $adjustment) {
        $adjustments_total = $adjustments_total ? $adjustments_total->add($adjustment->getAmount()) : $adjustment->getAmount();
      }
    }

    return $adjustments_total;
  }

  /**
   * Helper to format address.
   */
  protected function formatAddress(AddressInterface $address, ?string $email): array {
    $data = [
      'family_name' => $address->getFamilyName(),
      'given_name' => $address->getGivenName(),
      'city' => $address->getLocality(),
      'country' => $address->getCountryCode(),
      'postal_code' => $address->getPostalCode(),
      'street_address' => $address->getAddressLine1(),
      'street_address2' => $address->getAddressLine2(),
      'region' => $address->getAdministrativeArea(),
      'organization_name' => $address->getOrganization(),
    ];

    if (!empty($email)) {
      $data['email'] = $email;
    }

    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function getKlarnaSettings(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments, string $step, bool $express_cart = FALSE): array {
    $order_payload = $this->getOrderPayload($order, $express_cart ? KlarnaEvents::CREATE_EXPRESS_ORDER_REQUEST : KlarnaEvents::CREATE_PAYMENT_SESSION, $klarna_payments);
    $session_data = $order->getData(KlarnaManagerInterface::KLARNA_ORDER_KEY) ?? [];

    $checkout_type = KlarnaPaymentsInterface::KLARNA_PAYMENTS_CHECKOUT_MULTI_STEP;

    // If we are using cart express checkout, or we use finalize, but we don't
    // have session data, the klarna checkout flow is one step.
    if ($step === KlarnaPaymentsInterface::KLARNA_PAYMENTS_FINALIZE && empty($session_data)) {
      $checkout_type = KlarnaPaymentsInterface::KLARNA_PAYMENTS_CHECKOUT_ONE_STEP;
    }

    return [
      'orderPayload' => $order_payload,
      'client_token' => $session_data['client_token'] ?? $klarna_payments->getClientToken(),
      'style' => $klarna_payments->getStyle(),
      'auto_finalize' => $step === KlarnaPaymentsInterface::KLARNA_PAYMENTS_FINALIZE,
      'locale' => $order_payload['locale'] ?? 'en_US',
      'collect_shipping_address' => $express_cart && $klarna_payments->collectShippingAddress(),
      // Add endpoint url where needed.
      'endpoint' => NULL,
      'step' => $step,
      'checkout' => $checkout_type,
      'cart' => $express_cart,
    ];
  }

  /**
   * Handles all API calls towards Klarna.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  protected function apiRequest(string $endpoint, KlarnaPaymentsInterface $klarna_payments, array $payload = [], string $method = 'GET', array $headers = []): array {
    $commerce_core = $this->moduleExtensionList->getExtensionInfo('commerce');
    $values = [
      'headers' => [
        'Authorization' => sprintf("Basic %s", $klarna_payments->getPassword()),
        'Content-Type' => 'application/json',
        'Klarna-Idempotency-Key' => $this->uuid->generate(),
        'User-Agent' => sprintf('Drupal Commerce %s - https://www.drupal.org/project/commerce_klarna (Language=PHP/%s;)', $commerce_core[0]['version'] ?? '2', PHP_VERSION),
      ],
    ];

    if (!empty($headers)) {
      $values['headers'] += $headers;
    }

    if (!empty($payload)) {
      $values['json'] = $payload;
    }

    try {
      $request = $this->client->request($method, $klarna_payments->getApiUrl() . $endpoint, $values);
      $response = Json::decode($request->getBody());

      if ($klarna_payments->logRequests()) {
        $this->getLogger('commerce_klarna')->info($request->getBody());
      }
    }
    catch (GuzzleException $exception) {
      $data = Json::decode($exception->getResponse()->getBody()) ?? [];
      throw new KlarnaException($exception->getMessage(), $exception->getCode(), $data);
    }

    return $response ?? [];

  }

  /**
   * {@inheritdoc}
   */
  public function getLocale(string $purchase_country, string $language): string {
    $locales = self::KLARNA_LOCALE[$purchase_country] ?? [];
    if ($locales) {
      foreach ($locales as $locale) {
        if (sprintf('%s_%s', $language, $purchase_country) === $locale) {
          return $locale;
        }
      }

      return reset($locales);
    }

    return 'en-US';
  }

  /**
   * Generate Klarna messaging by order.
   */
  public function getOnsiteMessaging(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments, $locale = NULL): array {
    if ($locale === NULL) {
      $store_address = $order->getStore()->getAddress();
      $purchase_country = $store_address->getCountryCode();
      $locale = $this->getLocale($purchase_country, $this->languageManager->getCurrentLanguage()->getId());
    }
    return [
      '#theme' => 'klarna_onsite_messaging',
      '#locale' => $locale,
      '#environment' => $klarna_payments->getMode() === 'test' ? 'playground' : 'production',
      '#client_id' => $klarna_payments->getClientToken(),
      '#amount' => $this->minorUnitsConverter->toMinorUnits($order->getTotalPrice()),
    ];
  }

  /**
   * Generate Klarna messaging by purchase country.
   */
  public function onsiteMessagingByCountry(KlarnaPaymentsInterface $klarna_payments, Price $amount, $purchase_country): array {
    $locale = $this->getLocale($purchase_country, $this->languageManager->getCurrentLanguage()->getId());
    return [
      '#theme' => 'klarna_onsite_messaging',
      '#locale' => $locale,
      '#environment' => $klarna_payments->getMode() === 'test' ? 'playground' : 'production',
      '#client_id' => $klarna_payments->getClientToken(),
      '#amount' => $this->minorUnitsConverter->toMinorUnits($amount),
    ];
  }

}
