<?php

namespace Drupal\commerce_klarna;

use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_price\Price;

/**
 * Handling all Klarna related API calls.
 */
interface KlarnaManagerInterface {

  /**
   * Allowed list of Klarna payment methods.
   */
  public const KLARNA_PAYMENT_METHODS = [
    'direct_debit' => 'Pay now',
    'direct_bank_transfer' => 'Pay now',
    'pay_by_card' => 'Pay now',
    'invoice' => 'Pay later',
    'pay_later_by_card' => 'Pay later',
    'slice_it_by_card' => 'Pay over time',
    'fixed_sum_credit' => 'Pay over time',
  ];

  /**
   * Key to store Klarna session data.
   */
  public const KLARNA_ORDER_KEY = 'klarna_payment_session';

  /**
   * Countries per region.
   */
  public const KLARNA_COUNTRIES = [
    'AU' => 'oc',
    'AT' => 'eu',
    'BE' => 'eu',
    'CA' => 'na',
    'CZ' => 'eu',
    'DK' => 'eu',
    'FI' => 'eu',
    'FR' => 'eu',
    'DE' => 'eu',
    'GR' => 'eu',
    'HU' => 'eu',
    'IE' => 'eu',
    'IT' => 'eu',
    'MX' => 'eu',
    'NL' => 'eu',
    'NZ' => 'oc',
    'NO' => 'eu',
    'PL' => 'eu',
    'PT' => 'eu',
    'RO' => 'eu',
    'SK' => 'eu',
    'ES' => 'eu',
    'SE' => 'eu',
    'CH' => 'eu',
    'GB' => 'eu',
    'US' => 'na',
  ];

  /**
   * Klarna list of environments by regions.
   */
  public const KLARNA_REGIONS = [
    'eu' => [
      'live' => 'https://api.klarna.com',
      'test' => 'https://api.playground.klarna.com',
    ],
    'na' => [
      'live' => 'https://api-na.klarna.com',
      'test' => 'https://api-na.playground.klarna.com',
    ],
    'oc' => [
      'live' => 'https://api-oc.klarna.com',
      'test' => 'https://api-oc.playground.klarna.com',
    ],
  ];

  /**
   * Klarna locale list.
   */
  public const KLARNA_LOCALE = [
    'AU' => ['en-AU'],
    'AT' => ['en-AT', 'de-AT'],
    'BE' => ['nl-BE', 'fr-BE', 'en-BE'],
    'CA' => ['en-CA', 'fr-CA'],
    'CZ' => ['cs-CZ', 'en-CZ'],
    'DK' => ['da-DK', 'en-DK'],
    'FI' => ['fi-FI', 'sv-FI', 'en-FI'],
    'FR' => ['fr-FR', 'en-FR'],
    'DE' => ['de-DE', 'en-DE'],
    'GR' => ['el-GR', 'en-GR'],
    'HU' => ['hu-HU', 'en-HU'],
    'IE' => ['en-IE'],
    'IT' => ['it-IT', 'en-IT'],
    'MX' => ['en-MX', 'es-MX'],
    'NL' => ['nl-NL', 'en-NL'],
    'NZ' => ['en-NZ'],
    'NO' => ['nb-NO', 'en-NO'],
    'PL' => ['pl-PL', 'en-PL'],
    'PT' => ['pt-PT', 'en-PT'],
    'RO' => ['ro-RO', 'en-RO'],
    'SK' => ['sk-SK', 'en-SK'],
    'ES' => ['es-ES', 'en-ES'],
    'SE' => ['sv-SE', 'en-SE'],
    'CH' => ['de-CH', 'fr-CH', 'it-CH', 'en-CH'],
    'GB' => ['en-GB'],
    'US' => ['en-US', 'es-US'],
  ];

  /**
   * Manages order Klarna session.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function paymentSession(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments): array;

  /**
   * Creates Klarna session.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function createPaymentSession(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments): array;

  /**
   * Update existing payment session.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function updatePaymentSession(OrderInterface $order, string $session_id, KlarnaPaymentsInterface $klarna_payments): void;

  /**
   * Get session details.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function getPaymentSession(OrderInterface $order, string $session_id): array;

  /**
   * Create Klarna Order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function createOrder(OrderInterface $order, string $authorization_token): array;

  /**
   * Create Klarna Order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function updateOrder(OrderInterface $order, string $klarna_order_id): array;

  /**
   * Get a Klarna order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function getOrder(PaymentInterface $payment): array;

  /**
   * Capture an Klarna order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function captureOrder(PaymentInterface $payment, Price $amount): array;

  /**
   * Cancel Klarna order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function cancelOrder(PaymentInterface $payment): void;

  /**
   * Refund Klarna order.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function refundOrder(PaymentInterface $payment, Price $amount): void;

  /**
   * List order captures.
   */
  public function listOrderCaptures(PaymentInterface $payment): array;

  /**
   * Send shipping information.
   */
  public function addShippingInformation(PaymentInterface $payment, array $shipping_info): array;

  /**
   * Cancel authorization token.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function cancelAuthorization(KlarnaPaymentsInterface $klarna_payments, string $authorization_token): void;

  /**
   * Create Merchant Card service promise.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function createCardPromise(OrderInterface $order, string $klarna_order_id): array;

  /**
   * Retrieve Merchant Card service promise.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function getCardPromise(OrderInterface $order, string $promise_id): array;

  /**
   * Create card settlement.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function createCardSettlement(OrderInterface $order, string $klarna_order_id, ?string $klarna_promise_id = NULL): array;

  /**
   * Get card settlement by Klarna settlement_id.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function getCardSettlement(OrderInterface $order, string $settlement_id);

  /**
   * Get card settlement by Klarna order id.
   *
   * @throws \Drupal\commerce_klarna\Exception\KlarnaException
   */
  public function getCardSettlementByOrderId(OrderInterface $order, string $klarna_order_id): array;

  /**
   * Build order payload.
   */
  public function getOrderPayload(OrderInterface $order, string $event_name, ?KlarnaPaymentsInterface $klarna_payments): array;

  /**
   * Generate settings for use with Klarna JS SDK.
   */
  public function getKlarnaSettings(OrderInterface $order, KlarnaPaymentsInterface $klarna_payments, string $step, bool $express_cart = FALSE): array;

  /**
   * Generate locale based of purchase country and current language.
   */
  public function getLocale(string $purchase_country, string $language): string;

}
