<?php

namespace Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_klarna\Exception\KlarnaException;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides the Klarna payments payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "klarna_merchant_card",
 *   label = "Klarna Payments (Merchant Card Service)",
 *   display_label = "Klarna payments",
 *   forms = {
 *     "add-payment-method" =
 *   "Drupal\commerce_klarna\PluginForm\Klarna\PaymentMethodAddForm"
 *   },
 *   js_library = "commerce_klarna/payments",
 *   payment_method_types = {"klarna_merchant_card"},
 *   requires_billing_information = FALSE,
 * )
 */
class KlarnaMerchantCard extends KlarnaPaymentsBase implements KlarnaMerchantCardInterface {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mcs' => [
        'enabled' => FALSE,
        'private_key' => '',
        'key_id' => '',
        'acquirer' => '_none',
      ],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['mcs'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Merchant Card Service'),
      '#description' => $this->t('Settle orders with virtual credit cards issued by Klarna. MSC needs to be configured and enabled for a merchant. Please read documentation @link', ['@link' => 'https://docs.klarna.com/merchant-card-service/merchant-card-service/overview-of-merchant-card-service/']),
      '#tree' => TRUE,
    ];

    $form['mcs']['private_key'] = [
      '#title' => $this->t('Private key'),
      '#description' => $this->t('Private key for Klarna, see @link', ['@link' => 'https://docs.klarna.com/merchant-card-service/merchant-card-service/overview-of-merchant-card-service/']),
      '#type' => 'textarea',
      '#maxlength' => 8192,
      '#required' => TRUE,
      '#default_value' => $this->configuration['mcs']['private_key'],
    ];

    $form['mcs']['key_id'] = [
      '#title' => $this->t('Klarna key ID'),
      '#description' => $this->t('Unique identifier for the public key to be used for encryption of the card data. Provided by Klarna'),
      '#type' => 'textfield',
      '#maxlength' => 128,
      '#required' => TRUE,
      '#default_value' => $this->configuration['mcs']['key_id'],
    ];

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface[] $payment_gateways */
    $payment_gateways = $this->entityTypeManager->getStorage('commerce_payment_gateway')->loadByProperties(['status' => 1]);

    $options = [
      '_none' => $this->t('None'),
    ];

    foreach ($payment_gateways as $payment_gateway) {
      $payment_method_types = $payment_gateway->getPlugin()->getPaymentMethodTypes();
      if (isset($payment_method_types['credit_card'])) {
        $options[$payment_gateway->id()] = $payment_gateway->label();
      }
    }

    $form['mcs']['acquirer'] = [
      '#title' => $this->t('Merchants Acquirer'),
      '#description' => $this->t('Credit card processor to use virtual cards. If you select, the virtual cards are going to be used by that payment gateway.'),
      '#type' => 'select',
      '#options' => $options,
      '#default_value' => $this->configuration['mcs']['acquirer'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['mcs'] = $values['mcs'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    assert($payment_method instanceof PaymentMethodInterface);
    $order = $payment->getOrder();
    assert($order instanceof OrderInterface);
    $authorization_token = $payment_method->get('remote_id')->value;

    // In custom implementation of MCS we could already reach this point.
    // So we only need to create blank payment with order id.
    if ($payment_method->get('klarna_promise_id')->isEmpty()) {
      try {
        $response = $this->klarnaManager->createOrder($order, $authorization_token);
      }
      catch (KlarnaException $e) {
        throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
      }

      $klarna_order_id = $response['order_id'];

      try {
        $card_promise = $this->klarnaManager->createCardPromise($order, $klarna_order_id);
      }
      catch (KlarnaException $e) {
        throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
      }

      try {
        $card_settlement = $this->klarnaManager->createCardSettlement($order, $klarna_order_id);
      }
      catch (KlarnaException $e) {
        throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
      }
      $payment_method->set('klarna_payment_type', $response['authorized_payment_method']['type']);
      $payment_method->set('klarna_settlement_id', $card_settlement['settlement_id']);
      $payment_method->set('klarna_promise_id', $card_promise['promise_id']);
      $payment_method->set('klarna_cards', count($card_promise['cards']));
      $payment_method->setExpiresTime(strtotime($card_promise['expires_at']));
      $payment_method->save();
    }

    else {
      $promise = $payment_method->get('klarna_promise_id')->value;

      try {
        $card_promise = $this->klarnaManager->getCardPromise($order, $promise);
      }
      catch (KlarnaException $e) {
        throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
      }
      $klarna_order_id = $card_promise['order_id'];
    }

    // Mark MSC placeholder payment as completed.
    // Technically by creating card settlement and promise it is
    // from this point done.
    // All payments which are utilized from 3rd party card acquirer
    // need to be manually created.
    $payment->setState('completed');
    $payment->setRemoteId($klarna_order_id);
    $payment->setAmount(new Price('0', $payment->getAmount()->getCurrencyCode()));
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL) {
    // If we have external acquirer, and skip generic 0 payment.
    if (!$payment->getAmount()->isZero()) {
      if ($card_acquirer = $this->getCardAcquirer()) {
        $card_acquirer->getPlugin()->capturePayment($payment, $amount);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    // If we have external acquirer, and skip generic 0 payment.
    if (!$payment->getAmount()->isZero()) {
      if ($card_acquirer = $this->getCardAcquirer()) {
        $card_acquirer->getPlugin()->refundPayment($payment, $amount);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    // If we have external acquirer, and skip generic 0 payment.
    if (!$payment->getAmount()->isZero()) {
      if ($card_acquirer = $this->getCardAcquirer()) {
        $card_acquirer->getPlugin()->voidPayment($payment);
      }
    }
    else {
      try {
        $this->klarnaManager->cancelMerchantCardOrder($payment);
      }
      catch (KlarnaException $e) {
        throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
      }
      $payment->setState('authorization_voided');
      $payment->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCardAcquirer(): ?PaymentGatewayInterface {
    $card_acquirer = $this->configuration['mcs']['acquirer'];
    if ($card_acquirer === '_none') {
      return NULL;
    }
    /** @var \Drupal\commerce_payment\PaymentGatewayStorageInterface $payment_gateway_storage */
    $payment_gateway_storage = $this->entityTypeManager->getStorage('commerce_payment_gateway');

    return $payment_gateway_storage->load($card_acquirer);
  }

  /**
   * {@inheritdoc}
   */
  public function decryptCard($card_values): false|string {
    $aes_key = $card_values['aes_key'];
    $iv = $card_values['iv'];
    $pci_data = $card_values['pci_data'];
    openssl_private_decrypt(base64_decode($aes_key), $aes_key_decrypted, openssl_get_privatekey($this->getPrivateKey()), OPENSSL_PKCS1_PADDING);
    return openssl_decrypt(base64_decode($pci_data), 'AES-128-CTR', $aes_key_decrypted, OPENSSL_RAW_DATA, base64_decode($iv));
  }

  /**
   * {@inheritdoc}
   */
  public function getPrivateKey(): ?string {
    return $this->configuration['mcs']['private_key'];
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyId(): string {
    return $this->configuration['mcs']['key_id'];
  }

  /**
   * {@inheritdoc}
   */
  public function getExternalUrl(PaymentInterface $payment): ?string {
    if ($payment->getAmount()->isZero()) {
      return parent::getExternalUrl($payment);
    }
    return $this->getCardAcquirer()?->getPlugin()?->getExternalUrl($payment);
  }

}
