<?php

namespace Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_klarna\Exception\KlarnaException;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_price\Price;

/**
 * Provides the Klarna payments payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "klarna_payments",
 *   label = "Klarna payments",
 *   display_label = "Klarna payments",
 *   forms = {
 *     "add-payment-method" =
 *   "Drupal\commerce_klarna\PluginForm\Klarna\PaymentMethodAddForm"
 *   },
 *   js_library = "commerce_klarna/payments",
 *   payment_method_types = {"klarna"},
 *   requires_billing_information = FALSE,
 * )
 */
class KlarnaPayments extends KlarnaPaymentsBase {

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    assert($payment_method instanceof PaymentMethodInterface);
    $order = $payment->getOrder();
    assert($order instanceof OrderInterface);
    $authorization_token = $payment_method->get('remote_id')->value;

    try {
      $response = $this->klarnaManager->createOrder($order, $authorization_token);
    }
    catch (KlarnaException $e) {
      throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
    }

    $payment->setState('authorization');
    $payment->setRemoteId($response['order_id']);
    $payment->save();

    $payment_method->set('klarna_payment_type', $response['authorized_payment_method']['type']);
    $payment_method->save();
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['authorization']);
    // If not specified, capture the entire amount.
    $amount = $amount ?: $payment->getAmount();

    try {
      $this->klarnaManager->captureOrder($payment, $amount);
    }
    catch (KlarnaException $e) {
      throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
    }

    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    try {
      $this->klarnaManager->refundOrder($payment, $amount);
    }
    catch (KlarnaException $e) {
      throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    try {
      $this->klarnaManager->cancelOrder($payment);
    }
    catch (KlarnaException $e) {
      throw PaymentGatewayException::createForPayment($payment, $e->getKlarnaError());
    }
    $payment->setState('authorization_voided');
    $payment->save();
  }

}
