/**
 * @file
 * Defines behaviors for the Klarna payment method form.
 */

(function (Drupal, drupalSettings, Klarna) {
  /**
   * Attaches the commerceKlarna behavior.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the commerceKlarna behavior.
   */
  Drupal.behaviors.commerceKlarna = {
    attach: function () {

      if (!drupalSettings.commerceKlarna) {
        return;
      }

      // Create a markup for displaying errors.
      function errorHandling(errorMessage) {
        const error = document.createElement('div');
        error.setAttribute('role', 'alert');
        const message = document.createElement('div');
        message.classList.add(
          'payment-messages',
          'payment-messages--error',
          'messages',
          'messages--error'
        );
        message.innerText = errorMessage;
        error.append(message);
        return error;
      }

      // Handles one-step checkout flow, and the first step in multistep check.
      function klarnaOneStep(Klarna, klarnaSettings) {
        const checkoutForm = document.querySelector('.commerce-checkout-flow');
        if (klarnaSettings.cart === false && checkoutForm) {
          const submit = checkoutForm.querySelector('.button--primary');
          submit.setAttribute('disabled', 'disabled');
        }

        Klarna.Payments.Buttons.init({
          client_id: klarnaSettings.client_token,
        }).load(
          {
            container: '#klarna-payments-container',
            theme: klarnaSettings.style.theme,
            shape: klarnaSettings.style.shape,
            locale: klarnaSettings.locale,
            on_click: (authorize) => {
              authorize(
                {
                auto_finalize: klarnaSettings.auto_finalize,
                collect_shipping_address: klarnaSettings.collect_shipping_address,
                },
                klarnaSettings.orderPayload, (result) => {
                // If user closes Klarna popup, do nothing.
                if (result.approved === false) {
                  return;
                }

                if (!klarnaSettings.cart) {
                  document.querySelector(
                    '.klarna-session-data'
                  ).value = JSON.stringify(result);
                  checkoutForm.submit();
                } else {
                  fetch(drupalSettings.commerceKlarna.endpoint, {
                    method: 'POST',
                    headers: {
                      'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                      data: result,
                    }),
                  })
                    .then((response) => response.json())
                    .then((data) => {
                      if (data.redirect_url) {
                        window.location.replace(data.redirect_url);
                      } else {
                        document.querySelector(
                          '.cart-form'
                        ).prepend(errorHandling(data.error));
                      }
                    })
                    .catch((error) => {
                      console.log(error);
                    });
                }
              })
            },
          },
        )
      }

      // Handles multistep checkout flow.
      function klarnaMultistep(Klarna, klarnaSettings) {
        // If we are triggering multistep from cart,
        // we need first to get session id.
        // Later in checkout as with regular payment methods
        // we will have same logic.
        if (klarnaSettings.step === 'session') {
          klarnaOneStep(Klarna, klarnaSettings);
        } else {
          try {
            Klarna.Payments.init({
              client_token: klarnaSettings.client_token,
            });
          } catch (e) {
            // Do nothing.
          }
          Klarna.Payments.load(
            {
              container: '#klarna-payments-container',
              instance_id: 'klarna-widget',
            },
            {},
            function (res) {
              // Do nothing.
            },
          );

          // Endpoint is sent only when we are one last step in a multistep checkout.
          if (klarnaSettings.endpoint) {
            const checkoutForm = document.querySelector(
              '.commerce-checkout-flow'
            );
            const submit = checkoutForm.querySelector(
              '.button--primary'
            );
            submit.addEventListener('click', (event) => {
                event.preventDefault();
                Klarna.Payments.finalize(
                  {},
                  klarnaSettings.orderPayload, function (result) {
                  fetch(klarnaSettings.endpoint, {
                    method: 'POST',
                    headers: {
                      'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                      authorization_token: result.authorization_token,
                    }),
                  })
                    .then((response) => response.json())
                    .then((r) => {
                      if (result.error) {
                        const errorMessage = [];
                        Object.keys(result.error).forEach((key) => {
                          errorMessage.push(`${key}: ${result.error[key]}`);
                        });
                        checkoutForm.prepend(errorHandling(errorMessage.join('\n')));
                      } else if (r.error) {
                        checkoutForm.prepend(errorHandling(r.error));
                      } else {
                        checkoutForm.submit();
                      }
                    })
                    .catch((error) => {
                      checkoutForm.prepend(errorHandling(error));
                    });
                },
              );
            });
          }
        }
      }

      const container = document.getElementById('klarna-payments-container');
      if (!container) {
        return;
      }

      if (container.classList.contains('klarna-payments-init')) {
        return;
      }
      container.classList.add('klarna-payments-init');

      const waitForKlarnaSdk = setInterval(function () {
        if (typeof Klarna !== 'undefined') {
          const klarnaSettings = drupalSettings.commerceKlarna;

          const klarnaCheckout = drupalSettings.commerceKlarna.checkout;

          if (klarnaCheckout === 'one-step') {
            klarnaOneStep(Klarna, klarnaSettings);
          } else {
            klarnaMultistep(Klarna, klarnaSettings);
          }

          clearInterval(waitForKlarnaSdk);
        }
      }, 100);
    },

    detach: function (context, settings, trigger) {
      if (trigger !== 'unload') {
        return;
      }
      const checkoutForm = document.querySelector('.commerce-checkout-flow');
      if (!checkoutForm) {
        return;
      }
      const submit = checkoutForm.querySelector('.button--primary');
      submit.removeAttribute('disabled');
    },
  };
})(Drupal, drupalSettings, window.Klarna);
