<?php

namespace Drupal\commerce_klarna\Controller;

use Drupal\commerce_checkout\Resolver\ChainCheckoutFlowResolverInterface;
use Drupal\commerce_klarna\Event\KlarnaEvents;
use Drupal\commerce_klarna\Event\KlarnaShipmentsEvent;
use Drupal\commerce_klarna\KlarnaManagerInterface;
use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Url;
use Drupal\profile\Entity\ProfileInterface;
use GuzzleHttp\Exception\BadResponseException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Endpoints for Klarna single and multistep checkout.
 */
class KlarnaExpressCheckout extends ControllerBase {

  use LoggerChannelTrait;

  /**
   * The Klarna manager.
   */
  protected KlarnaManagerInterface $klarnaManager;

  /**
   * The checkout resolver.
   */
  protected ChainCheckoutFlowResolverInterface $checkoutFlowResolver;

  /**
   * The event dispatcher.
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->klarnaManager = $container->get('commerce_klarna.manager');
    $instance->checkoutFlowResolver = $container->get('commerce_checkout.chain_checkout_flow_resolver');
    $instance->eventDispatcher = $container->get('event_dispatcher');
    return $instance;
  }

  /**
   * Prepare Drupal for placing an express order from the cart.
   */
  public function onCreate(OrderInterface $commerce_order, PaymentGatewayInterface $commerce_payment_gateway, Request $request): JsonResponse {
    if (!$commerce_payment_gateway->getPlugin() instanceof KlarnaPaymentsInterface) {
      throw new AccessException('Invalid payment gateway provided.');
    }

    if (empty($request->getContent())) {
      throw new \InvalidArgumentException('There was a problem processing');
    }

    if ($commerce_order->getState()->getId() !== 'draft') {
      throw new AccessException('Invalid order state.');
    }

    $body = Json::decode($request->getContent());

    if (empty($body['data']['session_id'])) {
      $this->getLogger('commerce_klarna')->error('Missing session id or authorization token');
      $message = $this->t('Payment failed. Please review your information and try again.');
      return new JsonResponse(['message' => $message], Response::HTTP_BAD_REQUEST);
    }

    $commerce_order->set('payment_gateway', $commerce_payment_gateway);
    $commerce_order->setData(KlarnaManagerInterface::KLARNA_ORDER_KEY, $body['data']);
    $checkout_flow = $this->checkoutFlowResolver->resolve($commerce_order);
    $steps = $checkout_flow->getPlugin()->getSteps();
    // By default, take first step in checkout.
    $next_step = array_key_first($steps);
    $collect_shipping_address = $commerce_payment_gateway->getPlugin()->collectShippingAddress();
    $commerce_order->set('checkout_flow', $checkout_flow);
    $commerce_order->setRefreshState(OrderInterface::REFRESH_SKIP);
    $email = NULL;

    $shipping_address_collected = FALSE;
    try {
      $payment_session = $this->klarnaManager->getPaymentSession($commerce_order, $body['data']['session_id']);
      $shipping_address = $body['data']['collected_shipping_address'] ?? [];

      if ($collect_shipping_address && $shipping_address && $commerce_order->hasField('shipments')) {
        $email = $shipping_address['email'] ?? NULL;
        $shipping_profile = $this->createProfile($commerce_order, $shipping_address);
        $event = new KlarnaShipmentsEvent($commerce_order, $shipping_profile);
        $this->eventDispatcher->dispatch($event, KlarnaEvents::EXPRESS_CHECKOUT_SHIPMENTS);
        $shipments = $event->getShipments();
        $commerce_order->set('shipments', $shipments);
        $shipping_address_collected = TRUE;
      }

      $billing_information = $commerce_order->getBillingProfile();
      if (!$billing_information) {
        $billing = $payment_session['billing_address'] ?? NULL;
        if ($billing) {
          $email = $billing['email'] ?? NULL;
          $billing_information = $this->createProfile($commerce_order, $billing);
          $commerce_order->setBillingProfile($billing_information);
        }
      }

      $payment_method_storage = $this->entityTypeManager()->getStorage('commerce_payment_method');
      $payment_method = $payment_method_storage->create([
        'type' => $commerce_payment_gateway->getPlugin()->getDefaultPaymentMethodType()->getPluginId(),
        'uid' => $commerce_order->getCustomer()->id(),
        'payment_gateway' => $commerce_payment_gateway->id(),
        'payment_gateway_mode' => $commerce_payment_gateway->getPlugin()->getMode(),
        'remote_id' => $body['data']['session_id'],
        'reusable' => FALSE,
        'billing_profile' => $billing_information,
      ]);

      $payment_method->save();

      if ($email) {
        $commerce_order->setEmail($email);
      }

      if ($shipping_address_collected) {
        if (isset($steps['review'])) {
          $next_step = 'review';
        }
      }

      else {
        if (isset($steps['order_information'])) {
          $next_step = 'order_information';
        }
      }

      $commerce_order->set('payment_method', $payment_method->id());
      $commerce_order->set('checkout_step', $next_step);
      $commerce_order->setRefreshState(OrderInterface::REFRESH_ON_SAVE);
      $commerce_order->save();
      return new JsonResponse([
        'redirect_url' => Url::fromRoute('commerce_checkout.form', [
          'commerce_order' => $commerce_order->id(),
          'step' => $next_step,
        ])->toString(),
      ]);
    }
    catch (BadResponseException $exception) {
      $this->getLogger('commerce_klarna')->error($exception->getResponse()->getBody()->getContents());
      $message = $this->t('Payment failed. Please review your information and try again.');
      return new JsonResponse(['error' => $message], Response::HTTP_BAD_REQUEST);
    }
    catch (\Exception $exception) {
      $this->getLogger('commerce_klarna')->error($exception->getMessage());
      $message = $this->t('Payment failed. Please review your information and try again.');
      return new JsonResponse(['error' => $message], Response::HTTP_BAD_REQUEST);
    }
  }

  /**
   * Create profile.
   */
  protected function createProfile(OrderInterface $order, array $klarna_address): ProfileInterface {
    $profile_storage = $this->entityTypeManager()->getStorage('profile');
    $profile = $profile_storage->create([
      'type' => 'customer',
      'uid' => $order->getCustomer()->id(),
      'address' => [
        'country_code' => $klarna_address['country'],
        'locality' => $klarna_address['city'],
        'postal_code' => $klarna_address['postal_code'],
        'address_line1' => $klarna_address['street_address'],
        'address_line2' => $klarna_address['street_address2'] ?? NULL,
        'administrative_area' => $klarna_address['region'],
        'given_name' => $klarna_address['given_name'],
        'family_name' => $klarna_address['family_name'],
      ],
    ]);

    if ($profile->hasField('field_phone_number')) {
      $profile->set('field_phone_number', $klarna_address['phone'] ?? NULL);
    }

    $profile->save();
    return $profile;
  }

  /**
   * Finalize one step Klarna payment checkout.
   */
  public function onFinalizeMultiStep(OrderInterface $commerce_order, PaymentMethodInterface $commerce_payment_method, Request $request): JsonResponse {
    $commerce_payment_gateway = $commerce_payment_method->getPaymentGateway();
    if (!$commerce_payment_gateway->getPlugin() instanceof KlarnaPaymentsInterface) {
      throw new AccessException('Invalid payment gateway provided.');
    }
    $order_session = $commerce_order->getData(KlarnaManagerInterface::KLARNA_ORDER_KEY);

    if (empty($order_session['session_id'])) {
      throw new AccessException('Invalid payment gateway provided.');
    }
    if (empty($request->getContent())) {
      throw new \InvalidArgumentException('There was a problem processing');
    }

    if ($commerce_order->getState()->getId() !== 'draft') {
      throw new AccessException('Invalid order state.');
    }

    $body = Json::decode($request->getContent());
    $authorization_token = $body['authorization_token'] ?? NULL;

    if (!$authorization_token) {
      return new JsonResponse(['error' => 'Missing authorization token'], Response::HTTP_BAD_REQUEST);
    }

    $commerce_payment_method->setRemoteId($authorization_token);
    $commerce_payment_method->save();

    return new JsonResponse([]);

  }

}
