<?php

namespace Drupal\commerce_klarna\EventSubscriber;

use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Populate the billing profile from Klarna if we don't have one.
 */
class CheckoutCompletionSubscriber implements EventSubscriberInterface {

  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_checkout.completion' => ['onCheckoutCompletion', -100],
    ];
  }

  /**
   * During express checkout we cant get Klarna billing profile.
   *
   * Set profile from Klarna to payment method if there is none.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The event.
   */
  public function onCheckoutCompletion(OrderEvent $event): void {
    $order = $event->getOrder();

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    if ($payment_method = $order->get('payment_method')->entity) {
      $plugin = $payment_method->getPaymentGateway()->getPlugin();
      if ($plugin instanceof KlarnaPaymentsInterface && empty($payment_method->getBillingProfile())) {
        $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
        $payments = $payment_storage->loadMultipleByOrder($order);
        foreach ($payments as $payment) {
          $profile = $plugin->createBillingProfile($payment);
          if ($profile) {
            $payment_method->setBillingProfile($profile);
            $payment_method->save();

            if (empty($order->getBillingProfile())) {
              $profile_storage = $this->entityTypeManager->getStorage('profile');
              $billing_profile = $profile_storage->create(['type' => 'customer'])->populateFromProfile($profile);
              $billing_profile->save();
              $order->setBillingProfile($billing_profile);
            }
          }
        }

      }
    }
  }

}
