<?php

namespace Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_klarna\Exception\KlarnaException;
use Drupal\commerce_klarna\KlarnaManagerInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayBase;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\profile\Entity\Profile;
use Drupal\profile\Entity\ProfileInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Base class for all Klarna payment methods.
 */
abstract class KlarnaPaymentsBase extends OnsitePaymentGatewayBase implements KlarnaPaymentsInterface {

  /**
   * The Klarna Manager.
   */
  protected KlarnaManagerInterface $klarnaManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->klarnaManager = $container->get('commerce_klarna.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mode' => 'test',
      'username' => '',
      'password' => '',
      'client_token' => '',
      'store_id' => '',
      'region' => 'eu',
      'express_checkout' => [
        'enabled' => FALSE,
        'shipping' => FALSE,
      ],
      'style' => [
        'theme' => 'default',
        'shape' => 'default',
      ],
      'onsite_messaging' => [
        'add_to_cart' => FALSE,
        'cart' => FALSE,
        'checkout' => FALSE,
      ],
      'image_field' => NULL,
      'log_requests' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#description' => $this->t('Copy from @link', ['@link' => 'https://portal.klarna.com/settings/api-keys']),
      '#default_value' => $this->configuration['username'],
      '#required' => TRUE,
    ];

    $form['password'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('Password'),
      '#description' => $this->t('Copy from @link', ['@link' => 'https://portal.klarna.com/settings/api-keys']),
      '#default_value' => $this->configuration['password'],
      '#required' => TRUE,
    ];

    $form['client_token'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('Client Token'),
      '#description' => $this->t('Copy from @link', ['@link' => 'https://portal.klarna.com/settings/client-identifier']),
      '#default_value' => $this->configuration['client_token'],
      '#required' => TRUE,
    ];
    $form['store_id'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('Store ID'),
      '#description' => $this->t('Copy from @link', ['@link' => 'https://portal.klarna.com/settings/api-keys']),
      '#default_value' => $this->configuration['store_id'],
      '#required' => TRUE,
    ];

    $form['region'] = [
      '#title' => $this->t('Region'),
      '#type' => 'select',
      '#default_value' => $this->configuration['region'],
      '#options' => [
        'eu' => $this->t('Europe'),
        'na' => $this->t('North America'),
        'oc' => $this->t('Oceania'),
      ],
      '#required' => TRUE,
    ];

    $form['style'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Buy button style'),
      '#tree' => TRUE,
      '#required' => TRUE,
    ];

    $form['style']['theme'] = [
      '#title' => $this->t('Buy button theme'),
      '#type' => 'select',
      '#default_value' => $this->configuration['style']['theme'],
      '#options' => [
        'default' => $this->t('Default'),
        'light' => $this->t('Light'),
        'outlined' => $this->t('Outlined'),
      ],
      '#required' => TRUE,
    ];

    $form['style']['shape'] = [
      '#title' => $this->t('Buy button shape'),
      '#type' => 'select',
      '#default_value' => $this->configuration['style']['shape'],
      '#options' => [
        'default' => $this->t('Default'),
        'rect' => $this->t('Rect'),
        'pill' => $this->t('Pill'),
      ],
      '#required' => TRUE,
    ];

    $form['express_checkout'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Express checkout'),
      '#description' => $this->t('If enabled, provides Klarna button on the cart.'),
      '#tree' => TRUE,
    ];

    $form['express_checkout']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable express checkout'),
      '#default_value' => $this->configuration['express_checkout']['enabled'],
    ];

    $form['express_checkout']['shipping'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Collect shipping information trough Klarna'),
      '#default_value' => $this->configuration['express_checkout']['shipping'],
    ];

    $form['onsite_messaging'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Marketing blocks'),
      '#description' => $this->t('Specify placement for marketing blocks.  See documentation @link', ['@link' => 'https://docs.klarna.com/conversion-boosters/on-site-messaging/integrate-on-site-messaging/integrate-using-klarna-web-sdk/']),
      '#tree' => TRUE,
    ];

    $form['onsite_messaging']['add_to_cart'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable messaging on Add to Cart forms.'),
      '#default_value' => $this->configuration['onsite_messaging']['add_to_cart'],
    ];

    $form['onsite_messaging']['cart'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable messaging on cart page'),
      '#default_value' => $this->configuration['onsite_messaging']['cart'],
    ];

    $form['onsite_messaging']['checkout'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable messaging during checkout if Klarna is selected'),
      '#default_value' => $this->configuration['onsite_messaging']['cart'],
    ];

    $form['log_requests'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log API requests'),
      '#default_value' => $this->configuration['log_requests'],
    ];

    $form['image_field'] = [
      '#type' => 'textfield',
      '#maxlength' => 25,
      '#title' => $this->t('Image field'),
      '#description' => $this->t('Enter image machine field name from product variation entity, to be used on Klarna requests. Non required field.'),
      '#default_value' => $this->configuration['image_field'],
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['username'] = $values['username'];
      $this->configuration['password'] = $values['password'];
      $this->configuration['store_id'] = $values['store_id'];
      $this->configuration['client_token'] = $values['client_token'];
      $this->configuration['region'] = $values['region'];
      $this->configuration['express_checkout'] = $values['express_checkout'];
      $this->configuration['style'] = $values['style'];
      $this->configuration['onsite_messaging'] = $values['onsite_messaging'];
      $this->configuration['image_field'] = $values['image_field'];
      $this->configuration['log_requests'] = $values['log_requests'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details) {
    $required_keys = [
      'session',
    ];
    foreach ($required_keys as $required_key) {
      if (empty($payment_details[$required_key])) {
        throw InvalidRequestException::createForPayment($payment_method, sprintf('$payment_details must contain the %s key.', $required_key));
      }
    }

    $session_data = Json::decode($payment_details['session']);

    // Use authorization token if we have it.
    $remote_id = $session_data['authorization_token'] ?? $session_data['session_id'];

    $payment_method->set('remote_id', $remote_id);
    $payment_method->setReusable(FALSE);
    $payment_method->save();
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    // Do nothing for now.
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    // Do nothing.
  }

  /**
   * {@inheritdoc}
   */
  public function getUsername(): string {
    return $this->configuration['username'];
  }

  /**
   * {@inheritdoc}
   */
  public function getPassword(): string {
    return $this->configuration['password'];
  }

  /**
   * {@inheritdoc}
   */
  public function getClientToken(): string {
    return $this->configuration['client_token'];
  }

  /**
   * {@inheritdoc}
   */
  public function getStyle(): array {
    return $this->configuration['style'];
  }

  /**
   * {@inheritdoc}
   */
  public function getRegion(): string {
    return $this->configuration['region'];
  }

  /**
   * {@inheritdoc}
   */
  public function getApiUrl(): string {
    return KlarnaManagerInterface::KLARNA_REGIONS[$this->configuration['region']][$this->configuration['mode']];
  }

  /**
   * {@inheritdoc}
   */
  public function expressCheckoutEnabled(): bool {
    return (bool) $this->configuration['express_checkout']['enabled'];
  }

  /**
   * {@inheritdoc}
   */
  public function collectShippingAddress(): bool {
    return (bool) $this->configuration['express_checkout']['shipping'];
  }

  /**
   * {@inheritdoc}
   */
  public function addToCartMessaging(): bool {
    return (bool) $this->configuration['onsite_messaging']['add_to_cart'];
  }

  /**
   * {@inheritdoc}
   */
  public function cartPageMessaging(): bool {
    return (bool) $this->configuration['onsite_messaging']['cart'];
  }

  /**
   * {@inheritdoc}
   */
  public function checkoutPageMessaging(): bool {
    return (bool) $this->configuration['onsite_messaging']['checkout'];
  }

  /**
   * Determine if logging is enabled.
   */
  public function logRequests(): bool {
    return (bool) $this->configuration['log_requests'];
  }

  /**
   * {@inheritdoc}
   */
  public function getExternalUrl(PaymentInterface $payment): ?string {
    return sprintf('https://%s.klarna.com/orders/to-capture/merchants/%s/orders/%s', $this->getMode() === 'live' ? 'portal' : 'portal.playground', $this->configuration['store_id'], $payment->getRemoteId());
  }

  /**
   * Retrieve klarna billing data from Klarna order and create profile.
   */
  public function createBillingProfile(PaymentInterface $payment): ?ProfileInterface {
    try {
      $klarna_order = $this->klarnaManager->getOrder($payment);
    }
    catch (KlarnaException) {
      // Fail silent.
      $klarna_order = NULL;
    }

    if ($klarna_order && isset($klarna_order['billing_address'])) {
      $klarna_address = $klarna_order['billing_address'];
      $profile = Profile::create([
        'type' => 'customer',
        'uid' => $payment->getOrder()->getCustomer()->id(),
        'address' => [
          'country_code' => $klarna_address['country'],
          'locality' => $klarna_address['city'],
          'postal_code' => $klarna_address['postal_code'],
          'address_line1' => $klarna_address['street_address'],
          'address_line2' => $klarna_address['street_address2'] ?? NULL,
          'administrative_area' => $klarna_address['region'],
          'given_name' => $klarna_address['given_name'],
          'family_name' => $klarna_address['family_name'],
        ],
      ]);

      if ($profile->hasField('field_phone_number')) {
        $profile->set('field_phone_number', $klarna_address['phone'] ?? NULL);
      }

      $profile->save();
      return $profile;
    }

    return NULL;
  }

}
