<?php

namespace Drupal\commerce_klarna\PluginForm\Klarna;

use Drupal\commerce\InlineFormManager;
use Drupal\commerce_klarna\KlarnaManagerInterface;
use Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface;
use Drupal\commerce_payment\PluginForm\PaymentMethodAddForm as BasePaymentMethodAddForm;
use Drupal\commerce_store\CurrentStoreInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Klarna Payments payment method add form.
 */
class PaymentMethodAddForm extends BasePaymentMethodAddForm {

  use MessengerTrait;

  /**
   * The route match.
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The language manager.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The Klarna client.
   */
  protected KlarnaManagerInterface $klarnaManager;

  /**
   * Constructs a new PaymentMethodForm.
   */
  public function __construct(CurrentStoreInterface $current_store, EntityTypeManagerInterface $entity_type_manager, InlineFormManager $inline_form_manager, LoggerInterface $logger, RouteMatchInterface $route_match, LanguageManagerInterface $language_manager, KlarnaManagerInterface $klarna_manager) {
    parent::__construct($current_store, $entity_type_manager, $inline_form_manager, $logger);
    $this->routeMatch = $route_match;
    $this->languageManager = $language_manager;
    $this->klarnaManager = $klarna_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_store.current_store'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.commerce_inline_form'),
      $container->get('logger.channel.commerce_payment'),
      $container->get('current_route_match'),
      $container->get('language_manager'),
      $container->get('commerce_klarna.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['payment_details'] = $this->buildKlarnaPaymentsForm($form['payment_details'], $form_state);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildKlarnaPaymentsForm(array $element, FormStateInterface $form_state) {
    /** @var \Drupal\commerce_klarna\Plugin\Commerce\PaymentGateway\KlarnaPaymentsInterface $plugin */
    $plugin = $this->plugin;

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    if ($order = $this->routeMatch->getParameter('commerce_order')) {
      // In a multistep checkout with review pane, get payment session only.
      // Finalize Klarna payment at review pane.
      // @see \commerce_klarna_form_commerce_checkout_flow_alter.
      $klarna_step = KlarnaPaymentsInterface::KLARNA_PAYMENTS_SESSION;

      // If there is no review pane, finalize order upon submission of checkout
      // form.
      if ($checkout_flow = $order->get('checkout_flow')->entity->getPlugin()) {
        $current_step = $this->routeMatch->getParameter('step');
        $next_step = $checkout_flow->getNextStepId($current_step);

        if ($next_step === 'payment') {
          $klarna_step = KlarnaPaymentsInterface::KLARNA_PAYMENTS_FINALIZE;
        }
      }
      $plugin_definition = $plugin->getPluginDefinition();
      $element['#attached']['library'][] = $plugin_definition['js_library'];
      $klarna_js = $this->klarnaManager->getKlarnaSettings($order, $plugin, $klarna_step);
      $element['#attached']['drupalSettings']['commerceKlarna'] = $klarna_js;
      $element['#attributes']['class'][] = 'klarna-form';

      $element['klarna'] = [
        '#type' => 'container',
        '#id' => 'klarna-payments-container',
      ];

      $element['session'] = [
        '#type' => 'hidden',
        '#attributes' => [
          'class' => ['klarna-session-data'],
        ],
      ];
      // Put the Klarna widget below the billing address.
      $element['#weight'] = 50;

      if ($this->plugin->checkoutPageMessaging()) {
        $element['klarna_messaging'] = $this->klarnaManager->getOnsiteMessaging($order, $plugin, $klarna_js['locale']);
      }
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if ($order = $this->routeMatch->getParameter('commerce_order')) {
      $values = $form_state->getValue($form['payment_details']['#parents']);
      $session = Json::decode($values['session']);
      $order->setData(KlarnaManagerInterface::KLARNA_ORDER_KEY, $session);
    }
  }

}
