(function ($, Drupal, drupalSettings, cookies, navigator, screen) {
  Drupal.commerceMangopayApplePay = {}

  /**
   * Sets error on the form.
   *
   * @param form
   * @param formField
   * @param error
   */
  Drupal.commerceMangopayApplePay.setError = function (form, formField, error) {
    if (formField) {
      var parentElement = formField.parents('.js-form-item').first();
      parentElement.addClass('error').addClass('has-error');
    }

    if (error) {
      $('.js-messages', form).show();
      $('.js-messages ul', form).append($('<li/>', { text: error }));
    }
  };

  /**
   * Clears all errors on the form.
   *
   * @param form
   */
  Drupal.commerceMangopayApplePay.clearErrors = function (form) {
    $('.js-form-item.has-error', form).each(function () {
      $(this).removeClass('has-error').removeClass('error');
    });

    $('.js-messages', form).hide();
    $('.js-messages ul', form).empty();
  };


  /**
   * Validates required fields on the form.
   *
   * @param form
   * @returns {boolean}
   */
  Drupal.commerceMangopayApplePay.validateRequired = function (form) {
    var hasErrors = false;

    // Loop through all required fields and mark them as errorneous if no input provided.
    $('input[required], select[required]', form).each(function () {
      if (!$(this).val()) {
        Drupal.commerceMangopayApplePay.setError(form, $(this));
        hasErrors = true;
      }
    });

    if (hasErrors) {
      Drupal.commerceMangopayApplePay.setError(form, null, Drupal.t('Please fill required fields'));
    }

    return !hasErrors;
  };

  /**
   *
   * Helper function. Returns Billing information object for use in various ajax calls.
   *
   * @param form
   * @returns {{firstName: *, lastName: *, addressLine1: *, addressLine2: *, postalCode: *, city: *, country: *, email: string}}
   */
  Drupal.commerceMangopayApplePay.getBillingInformationInput = function (form) {
    return {
      firstName: $('input.given-name', form).val(),
      lastName: $('input.family-name', form).val(),
      addressLine1: $('input.address-line1', form).val(),
      addressLine2: $('input.address-line2', form).val(),
      postalCode: $('input.postal-code', form).val(),
      city: $('input.locality', form).val(),
      region: $('select.administrative-area', form).val(),
      country: $('select.country', form).val(),

      // For logged in users, we have a variable in drupalSettings. Anonymous users must provide it in the field.
      email: drupalSettings.commerceMangopay.email ? drupalSettings.commerceMangopay.email : $('input[data-drupal-selector="edit-contact-information-email"]', form).val()
    };
  };

  /**
   * Helper function. Returns Know Your Customer input.
   *
   * @param form
   * @returns {{cardholderName: *, cardholderCountry: *}}
   */
  Drupal.commerceMangopayApplePay.getKYCInput = function (form) {
    return {
      cardholderName: $('input.mangopay-dpi-cardholder-name', form).val(),
      cardholderCountry: $('select.mangopay-dpi-cardholder-country', form).val(),
    };
  };

  Drupal.commerceMangopayApplePay.onPayButtonClick = function (form) {
    console.log('starting');

    if (!ApplePaySession) {
      return;
    }

    // TODO: Validate entered data. Required fields must be
    // If there are any validation errors, scroll to the top and do not continue.

    var hasErrors = false;
    Drupal.commerceMangopayApplePay.clearErrors();
    if (!Drupal.commerceMangopayApplePay.validateRequired(form)) {
      hasErrors = true;
    }

    if (hasErrors) {
      $('html, body').animate({ scrollTop: 0 }, 200);
      return;
    }

    // Get user, wallet and card preregistration data.
    var billingInformationInput = Drupal.commerceMangopay.getBillingInformationInput(form);
    var kycInput = Drupal.commerceMangopay.getKYCInput(form);
    var names = kycInput.cardholderName.trim().split(' '); // Split by space and send to MANGO
    var firstName = names.shift();

    const request = {
      "countryCode": kycInput.cardholderCountry,
      "currencyCode": drupalSettings.commerceMangopay.currencyCode,
      "merchantCapabilities": [
        "supports3DS"
      ],
      "supportedNetworks": [
        "visa", "masterCard"
      ],
      "total": {
        "label": drupalSettings.commerceMangopay.transactionLabel,
        "type": "final",
        "amount": drupalSettings.commerceMangopay.amount,
      }
    }

    // Create ApplePaySession
    const session = new ApplePaySession(3, request);

    session.onvalidatemerchant = function (event) {
      $.ajax({
        method: "POST",
        url: "/commerce-mangopay-dpi/validate-apple-pay-merchant/" + drupalSettings.commerceMangopay.paymentGatewayId,
      })
        .done(function (merchantSession) {
          console.log('validated');
          session.completeMerchantValidation(merchantSession);
        })
        .fail(function (error) {
          console.log('NOT validated');
          console.error("Error fetching merchant session", error);
        });
    };

    session.onpaymentmethodselected = function (event) {
      console.log('onpaymentmethodselected');
      // Define ApplePayPaymentMethodUpdate based on the selected payment method.
      // No updates or errors are needed, pass an empty object.
      const update = {
        "newTotal": {
          "label": drupalSettings.commerceMangopay.transactionLabel,
          "type": "final",
          "amount": drupalSettings.commerceMangopay.amount,
        }
      };
      session.completePaymentMethodSelection(update);
    };

    session.onpaymentauthorized = function (event) {
      console.log('onpaymentauthorized');
      $.ajax({
        method: "POST",
        url: "/commerce-mangopay-dpi/preregister-apple-pay/" + drupalSettings.commerceMangopay.paymentGatewayId,
        data: {
          currency_code: drupalSettings.commerceMangopay.currencyCode,
          first_name: firstName,
          last_name: names.length > 0 ? names.join(' ') : 'UNDEFINED',
          email: billingInformationInput.email,
          address_line1: billingInformationInput.addressLine1,
          address_line2: billingInformationInput.addressLine2,
          postal_code: billingInformationInput.postalCode,
          city: billingInformationInput.city,
          region: billingInformationInput.region,
          country: kycInput.cardholderCountry
        }
      })
        .done(function (preregisterResponse) {

          // Save relevant data in the hidden fields on the form and pass the form over to Drupal for processing.
          $('input.mangopay-dpi-transaction-id', form).val(event.payment.token.transactionIdentifier);
          $('input.mangopay-dpi-network', form).val(event.payment.token.paymentMethod.network);
          $('input.mangopay-dpi-token-data', form).val(JSON.stringify(event.payment.token.paymentData));

          $('input.mangopay-dpi-user-id', form).val(preregisterResponse.userId);
          $('input.mangopay-dpi-wallet-id', form).val(preregisterResponse.walletId);

          console.log('Preregistered');

          // Call complete callback
          session.completePayment({
             "status": ApplePaySession.STATUS_SUCCESS
          });

          // Submit the whole and pass on control to actual Commerce checkout routines.
          form.submit();
        })
        .fail(function (error) {
          console.error("Error pre-registering the user data in MANGOPAY", error);
          Drupal.commerceMangopayApplePay.setError(form, null, Drupal.t('An unexpected error occurred while processing your payment. Please confirm the details you entered are correct or try a different card. If the problem persists, please contact us.'));
          $('html, body').animate({ scrollTop: 0 }, 200);
        });
    };

    session.oncancel = function (event) {
      // Payment cancelled by WebKit
    };

    session.begin();
  }

  Drupal.behaviors.commerceMangopayApplePay = {
    attach: function (context, settings) {
      var submitButton = $('[data-drupal-selector="edit-actions-next"], [data-drupal-selector="edit-actions-submit"]');
      var form = submitButton.parents('form').first();

      // Check if we actually have add payment method form opened (by looking for a specific field).
      // If not, make sure we've got our click event removed from the next button.
      // We have to do this in case this script is attached by the add new payment method
      // form, but then the form is closed when user selects existing payment method.
      if (!$('.mangopay-dpi-apple-pay-form', form).length) {
        submitButton.show();
        $('.mangopay-dpi-apple-pay-button').remove();
      }

      // TODO: If Apple Pay select, process default button by substituting it with Apple Pay button.

      $('.mangopay-dpi-apple-pay-form:not(".js-processed")', context).addClass('js-processed').each(function() {
        // Initially hide messages. Show only when there are js errors.
        Drupal.commerceMangopayApplePay.clearErrors(form);

        // If Apple Pay session enabled, please show the Apple Pay option and button.
        if (window.ApplePaySession) {
          submitButton.hide();
          submitButton.off('click');
          submitButton.removeClass('js-processed');

          var $button = $('<apple-pay-button buttonstyle="black" type="plain" locale="en" class="mangopay-dpi-apple-pay-button"></apple-pay-button>');
          $button.click(function() {
            Drupal.commerceMangopayApplePay.onPayButtonClick(form);
          })
          $button.insertBefore(submitButton);
        }
      })
    }
  };

})(jQuery, Drupal, drupalSettings, window.Cookies, navigator, screen);
