<?php

namespace Drupal\commerce_mangopay_dpi\Controller;

use Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface;
use Drupal\commerce_payment\PaymentStorageInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_payment\Exception\HardDeclineException;
use Drupal\commerce\Response\NeedsRedirectException;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Exception\ClientException;
use Drupal\Core\Url;

/**
 * MANGOPAY controller
 */
class MangopayController implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Payment storage
   *
   * @var PaymentStorageInterface
   */
  protected $paymentStorage;

  /**
   * Constructs a new MangopayController object.
   */
  public function __construct(MessengerInterface $messenger, PaymentStorageInterface $payment_storage) {
    $this->messenger = $messenger;
    $this->paymentStorage = $payment_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('entity_type.manager')->getStorage('commerce_payment'),
    );
  }

  /**
   * Callback method which pre registeres user's card
   * and creates user - wallet combo if it doesn't exist.
   *
   * @param PaymentGatewayInterface $commerce_payment_gateway
   * @param Request $request
   * @return void
   */
  public function preRegisterCard(PaymentGatewayInterface $commerce_payment_gateway, Request $request) {
    // Capture user details passed in the request
    $currency_code = $request->get('currency_code');
    if (empty($currency_code)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Currency code is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Currency code is required'], 400);
      $response->send();
      exit;
    }

    /** @var \Drupal\commerce_price\Entity\Currency $currency */
    $currency = \Drupal::entityTypeManager()->getStorage('commerce_currency')->load($currency_code);
    if (empty($currency)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Currency code is invalid');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Currency code is invalid'], 400);
      $response->send();
      exit;
    }

    $first_name = $request->get('first_name');
    if (empty($first_name)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('First name is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'First name is required'], 400);
      $response->send();
      exit;
    }

    $last_name = $request->get('last_name');
    if (empty($last_name)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Last name is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Last name is required'], 400);
      $response->send();
      exit;
    }

    $email = $request->get('email');
    if (empty($email)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Email is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Email is required'], 400);
      $response->send();
      exit;
    }

    $address_line1 = $request->get('address_line1');
    $address_line2 = $request->get('address_line2');
    $city = $request->get('city');
    $postal_code = $request->get('postal_code');
    $region = $request->get('region');
    $country = $request->get('country');
    if (empty($country)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Country is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Country is required'], 400);
      $response->send();
      exit;
    }

    $card_type = $request->get('card_type');
    if (empty($card_type)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Card type is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Card type is required'], 400);
      $response->send();
      exit;
    }

    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();

    // Load user if authenticated.
    $user = NULL;
    $account = \Drupal::currentUser();
    if ($account->isAuthenticated()) {
      $user = User::load($account->id());
    }

    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();

    $mangopay_user = $payment_gateway_plugin->resolveNaturalUser($first_name, $last_name, $email, $country, $address_line1, $address_line2, $city, $postal_code, $region, $payment_gateway_plugin->getTag(), $user);
    if (!$mangopay_user) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY user'
      ], 500);
      $response->send();
      exit;
    }

    $mangopay_wallet = $payment_gateway_plugin->resolveWallet($mangopay_user->Id, $currency_code);
    if (!$mangopay_wallet) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY wallet'
      ], 500);
      $response->send();
      exit;
    }

    // Initiate card registration
    try {
      $card_register = $payment_gateway_plugin->createCardRegistration($mangopay_user->Id, $currency_code, $card_type, $payment_gateway_plugin->getTag());
    } catch(\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to register card for user %s and wallet %s: %s: %s', $mangopay_user->Id, $mangopay_wallet->Id, $e->getCode(), $e->getMessage()));
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to register card'], 500);
      $response->send();
      exit;
    }

    // TODO: Handle errors better. Maybe some are recoverable? - https://docs.mangopay.com/guide/errors

    // Send response to the browser
    $response = new JsonResponse([
        'userId' => $mangopay_user->Id,
        'walletId' => $mangopay_wallet->Id,
        'cardRegistrationURL' => $card_register->CardRegistrationURL,
        'preregistrationData' => $card_register->PreregistrationData,
        'cardRegistrationId' => $card_register->Id,
        'cardType' => $card_register->CardType,
        'accessKey' => $card_register->AccessKey
      ]);
    $response->send();
    exit;
  }

  /**
   * Callback method for secure mode (3DS) results.
   *
   * @param Request $request
   * @param RouteMatchInterface $route_match
   * @return void
   */
  public function processSecureMode(Request $request, RouteMatchInterface $route_match) {

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $route_match->getParameter('commerce_order');
    $commerce_payment = $route_match->getParameter('commerce_payment');
    $return = $request->get('return');

    /** @var \Drupal\commerce_checkout\Entity\CheckoutFlow $checkout_flow */
    $checkout_flow = $order->get('checkout_flow')->entity;

    /** @var \Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowWithPanesInterface $checkout_flow_plugin */
    $checkout_flow_plugin = $checkout_flow->getPlugin();

    // Attempt for get step_id of payment information pane... If now step_id
    // If we're in checkout context, attempt to get step of payment_information pane.
    $current_step_id = $order->get('checkout_step')->getString();

    // If payment is already processed (by a hook, for example),
    // just redirect to the return URL, please.
    $state = $commerce_payment->getState()->value;
    if (in_array($state, ['completed'])) {
      if (!empty($return)) {
        \Drupal::messenger()->addMessage($this->t('Payment saved.'));
        throw new NeedsRedirectException($return);
      } else {
        $redirect_step_id = $checkout_flow_plugin->getNextStepId($current_step_id);
        $checkout_flow_plugin->redirectToStep($redirect_step_id);
      }
    }

    // Validate payment state. We allow only NEW payments here.
    $state = $commerce_payment->getState()->value;
    if (!in_array($state, ['new'])) {
      throw new \InvalidArgumentException(sprintf('The provided payment is in an invalid state ("%s").', $state));
    }

    // Validate payment method
    $payment_method = $commerce_payment->getPaymentMethod();
    if (empty($payment_method)) {
      throw new \InvalidArgumentException('The provided payment has no payment method referenced.');
    }
    if ($payment_method->isExpired()) {
      throw new HardDeclineException('The provided payment method has expired');
    }

    // Validate Remote ID exists
    if (empty($commerce_payment->getRemoteId())) {
      throw new \InvalidArgumentException('Payment missing remote Id');
    }

    $payment_gateway = $commerce_payment->getPaymentGateway();
    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $payment_gateway->getPlugin();
    $payment_gateway_configuration = $payment_gateway_plugin->getConfiguration();

    // Get Remote PayIn object and check its status.
    try {
      $payin = $payment_gateway_plugin->getPayIn($commerce_payment->getRemoteId());
    } catch(\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unknown critical error occurred while fetching pay in object %s', $commerce_payment->getRemoteId()));
      throw new \InvalidArgumentException('MANGOPAY Pay In object not found');
    }

    // Update payment object and redirect accordingly.
    switch($payin->Status) {
      case \MangoPay\PayInStatus::Succeeded:
        // Mark payment's state as completed.
        $commerce_payment->setState('completed');
        $commerce_payment->setCompletedTime(\Drupal::time()->getRequestTime());
        $commerce_payment->save();

        // Make sure to deregister card in Mangopay if it's a card payment and store_cards is disabled
        if (
          !$payment_gateway_configuration['store_cards']
          && $commerce_payment->getPaymentMethod()->bundle() == 'commerce_mangopay_dpi_credit_card'
        ) {
          $commerce_payment->getPaymentMethod()->setReusable(false)->save();
          try {
            $payment_gateway_plugin->deregisterCard($commerce_payment->getPaymentMethod()->getRemoteId());
          } catch (\Exception $e) {
            \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to deacrtivate card  %s:  %s', $commerce_payment->getPaymentMethod()->getRemoteId(), $e->getMessage()));
          }
        }

        $redirect_step_id = $checkout_flow_plugin->getNextStepId($current_step_id);
      break;

      case \MangoPay\PayInStatus::Failed:
        // TODO: Display more descriptive messages here for the transaction errors: https://docs.mangopay.com/guide/errors
        $commerce_payment->getPaymentMethod()->setReusable(false)->save();
        $commerce_payment->getOrder()->set('payment_method', NULL)->save();

        \Drupal::logger('commerce_mangopay_dpi')->warning(sprintf('Pay In Failure (3DS): %s: %s', $payin->ResultCode, $payin->ResultMessage));
        $message = $this->t('Your payment has either failed authentication or has been refused by the bank. Please verify your card details and try again.');
        $this->messenger->addError($message);
        $redirect_step_id = $current_step_id;
      break;

      default:
        $commerce_payment->getPaymentMethod()->setReusable(false)->save();
        $commerce_payment->getOrder()->set('payment_method', NULL)->save();

        // Always make sure to deregister card in Mangopay in case of non-standard failure
        if ($commerce_payment->getPaymentMethod()->bundle() == 'commerce_mangopay_dpi_credit_card') {
          try {
            $payment_gateway_plugin->deregisterCard($commerce_payment->getPaymentMethod()->getRemoteId());
          } catch (\Exception $e) {
            \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to deacrtivate card  %s:  %s', $commerce_payment->getPaymentMethod()->getRemoteId(), $e->getMessage()));
          }
        }

        \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Pay In Error (3DS): %s: %s', $payin->ResultCode, $payin->ResultMessage));
        $message = $this->t('Card authentication attempt has failed for an unknown reason. Please verify your card details and try again. If the problem persists, please contact us.');
        $this->messenger->addError($message);
        $redirect_step_id = $current_step_id;

      break;
    }

    // If return URL provided, return there instead of using checkout steps.
    if (!empty($return)) {
      \Drupal::messenger()->addMessage($this->t('Payment saved.'));
      throw new NeedsRedirectException($return);
    }

    $checkout_flow_plugin->redirectToStep($redirect_step_id);
  }

  /**
   * Validates Apple Pay merchant
   *
   * @param PaymentGatewayInterface $commerce_payment_gateway
   * @param Request $request
   * @return void
   */
  public function validateApplePayMerchant(PaymentGatewayInterface $commerce_payment_gateway) {
    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();
    $apple_pay_merchant_id = $payment_gateway_plugin->getApplePayMerchantId();

    /**
     * 1. Export cert from Keychain Access to .p12 format
     * 2. openssl pkcs12 -in applePayCert.p12 -out cert.pem -clcerts -nokeys
     * 3. openssl pkcs12 -in applePayCert.p12 -out key.pem -nocerts -nodes
     */
    $apple_pay_merchant_id_cert_path = $payment_gateway_plugin->getApplePayMerchantIdCertPath();
    $apple_pay_merchant_id_key_path = $payment_gateway_plugin->getApplePayMerchantIdKeyPath();

    $apple_pay_merchant_id_domain = $payment_gateway_plugin->getApplePayMerchantIdDomain();
    $site_name = \Drupal::config('system.site')->get('name');

    $http_client = \Drupal::httpClient();
    try {
      $apple_response = $http_client->post(
        'https://apple-pay-gateway.apple.com/paymentservices/paymentSession',
        [
          'json' => [
            'merchantIdentifier' => $apple_pay_merchant_id,
            'displayName' => $site_name,
            'initiative' => "web",
            'initiativeContext' => $apple_pay_merchant_id_domain
          ],
          'cert' => $apple_pay_merchant_id_cert_path,
          'ssl_key' => $apple_pay_merchant_id_key_path,
        ]
      );

      $response = new JsonResponse($apple_response->getBody()->getContents(), 200, [], true);
      return $response;
    } catch (ClientException $exception) {
      $apple_response = $exception->getResponse();
      $response = new JsonResponse($apple_response->getBody()->getContents(), 500, [], true);
      return $response;
    }
  }

  /**
   * Pre-registeres user and wallet for the purposes of Apple Pay
   *
   * @param PaymentGatewayInterface $commerce_payment_gateway
   * @param Request $request
   * @return void
   */
  public function preRegisterApplePay(PaymentGatewayInterface $commerce_payment_gateway, Request $request)
  {
    // Capture user details passed in the request
    $currency_code = $request->get('currency_code');
    if (empty($currency_code)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Currency code is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Currency code is required'
      ], 400);
      $response->send();
      exit;
    }

    /** @var \Drupal\commerce_price\Entity\Currency $currency */
    $currency = \Drupal::entityTypeManager()->getStorage('commerce_currency')->load($currency_code);
    if (empty($currency)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Currency code is invalid');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Currency code is invalid'
      ], 400);
      $response->send();
      exit;
    }

    $first_name = $request->get('first_name');
    if (empty($first_name)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('First name is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'First name is required'
      ], 400);
      $response->send();
      exit;
    }

    $last_name = $request->get('last_name');
    if (empty($last_name)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Last name is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Last name is required'
      ], 400);
      $response->send();
      exit;
    }

    $email = $request->get('email');
    if (empty($email)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Email is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Email is required'
      ], 400);
      $response->send();
      exit;
    }

    $address_line1 = $request->get('address_line1');
    $address_line2 = $request->get('address_line2');
    $city = $request->get('city');
    $postal_code = $request->get('postal_code');
    $region = $request->get('region');
    $country = $request->get('country');
    if (empty($country)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Country is missing');
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Country is required'
      ], 400);
      $response->send();
      exit;
    }

    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();

    // Load user if authenticated.
    $user = NULL;
    $account = \Drupal::currentUser();
    if ($account->isAuthenticated()) {
      $user = User::load($account->id());
    }

    /** @var \Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway\MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();

    $mangopay_user = $payment_gateway_plugin->resolveNaturalUser($first_name, $last_name, $email, $country, $address_line1, $address_line2, $city, $postal_code, $region, $payment_gateway_plugin->getTag(), $user);
    if (!$mangopay_user) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY user'
      ], 500);
      $response->send();
      exit;
    }

    $mangopay_wallet = $payment_gateway_plugin->resolveWallet($mangopay_user->Id, $currency_code);
    if (!$mangopay_wallet) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY wallet'
      ], 500);
      $response->send();
      exit;
    }

    // TODO: Handle errors better. Maybe some are recoverable? - https://docs.mangopay.com/guide/errors

    // Send response to the browser
    $response = new JsonResponse([
      'userId' => $mangopay_user->Id,
      'walletId' => $mangopay_wallet->Id
    ]);
    $response->send();
    exit;
  }

  /**
   * Preregisters a GooglePay
   */
  public function preregisterGooglePay(PaymentGatewayInterface $commerce_payment_gateway, Request $request): void
  {
    /** @var MangopayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();
    $billingDetails = $request->get('billing_details');

    $user = NULL;
    $account = \Drupal::currentUser();

    if ($account->isAuthenticated()) {
      $user = User::load($account->id());
    }

    $mangopay_user = $payment_gateway_plugin->resolveNaturalUser(
      $billingDetails['firstName'],
      $billingDetails['lastName'],
      $billingDetails['email'],
      $billingDetails['address']['country'],
      $billingDetails['address']['addressLine1'] ?? '',
      $billingDetails['address']['addressLine2'] ?? '',
      $billingDetails['address']['city'] ?? '',
      $billingDetails['address']['postalCode'] ?? '',
      $billingDetails['address']['region'] ?? '',
      $payment_gateway_plugin->getTag(),
      $user
    );

    if ($mangopay_user === NULL) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY user'
      ], 500);
      $response->send();
      exit;
    }

    $mangopay_wallet = $payment_gateway_plugin->resolveWallet($mangopay_user->Id,   $billingDetails['currency_code']);

    if (!$mangopay_wallet) {
      $response = new JsonResponse([
        'status' => 'Critical',
        'message' => 'Unable to create MANGOPAY wallet'
      ], 500);
      $response->send();
      exit;
    }

    $response = new JsonResponse([
      'userId' => $mangopay_user->Id,
      'walletId' => $mangopay_wallet->Id
    ]);

    $response->send();
    exit();
  }
}
