<?php

namespace Drupal\commerce_mangopay_dpi\Plugin\Commerce\PaymentGateway;

use Drupal\commerce\Response\NeedsRedirectException;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Exception\HardDeclineException;
use Drupal\commerce_payment\PaymentMethodTypeManager;
use Drupal\commerce_payment\PaymentTypeManager;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayBase;
use Drupal\commerce_price\Price;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\GeneratedUrl;
use Drupal\Core\Url;
use Drupal\Component\Serialization\Json;
use MangoPay\Address;
use MangoPay\Billing;
use MangoPay\BrowserInfo;
use MangoPay\Money;
use MangoPay\PayIn;
use MangoPay\PayInExecutionDetailsDirect;
use MangoPay\PayInPaymentDetailsGooglePay;
use MangoPay\PayInPaymentType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use MangoPay\TransactionStatus;

/**
 * Provides the Off-site Secure Mode (3DS) redirect
 *
 * @CommercePaymentGateway(
 *   id = "commerce_mangopay_dpi",
 *   label = "MANGOPAY Direct PayIn",
 *   display_label = "MANGOPAY Direct PayIn",
 *   requires_billing_information = FALSE,
 *   forms = {
 *     "add-payment-method" = "Drupal\commerce_mangopay_dpi\PluginForm\Onsite\PaymentMethodAddForm",
 *   },
 *   modes = {"sandbox" = "Sandbox", "production" = "Production"},
 *   payment_method_types = {
 *     "commerce_mangopay_dpi_credit_card", "commerce_mangopay_dpi_apple_pay", "commerce_mangopay_dpi_google_pay"
 *   },
 *   default_payment_method_type = "commerce_mangopay_dpi_credit_card",
 *   credit_card_types = {
 *     "dinersclub", "discover", "jcb", "maestro", "mastercard", "visa",
 *   },
 * )
 */
class Mangopay extends OnsitePaymentGatewayBase implements MangopayInterface {
  const STANDARD_TAG = 'drupal commerce';

  /**
   * @var \MangoPay\MangoPayApi
   */
  protected $api;

  /**
   *
   * @return \MangoPay\MangoPayApi
   */
  public function getApi() {
    return $this->api;
  }

  /**
   * {@inheritdoc}
   */
  public function getTag() {
    return $this->getConfiguration()['tag'];
  }

  /**
   * {@inheritdoc}
   */
  public function getApplePayMerchantId()
  {
    return $this->getConfiguration()['apple_pay_merchant_id'];
  }

  /**
   * {@inheritdoc}
   */
  public function getApplePayMerchantIdCertPath()
  {
    return $this->getConfiguration()['apple_pay_merchant_id_cert_path'];
  }

  /**
   * {@inheritdoc}
   */
  public function getApplePayMerchantIdKeyPath()
  {
    return $this->getConfiguration()['apple_pay_merchant_id_key_path'];
  }

  /**
   * {@inheritdoc}
   */
  public function getApplePayMerchantIdDomain()
  {
    return $this->getConfiguration()['apple_pay_merchant_id_domain'];
  }

  /**
   * @inheritdoc
   * @return mixed[]
   */
  public function getGooglePaySettings(): array
  {
    return $this->getConfiguration()['google_pay'];
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, PaymentTypeManager $payment_type_manager, PaymentMethodTypeManager $payment_method_type_manager, TimeInterface $time) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $payment_type_manager, $payment_method_type_manager, $time);

    // Construct MANGOPAY Api object
    $mode = $this->getConfiguration()['mode'];
    $client_id = $this->getConfiguration()['client_id'];
    $client_pass = $this->getConfiguration()['client_pass'];
    switch($mode) {
      case 'production':
        $base_url = 'https://api.mangopay.com';
        break;
      default:
        $base_url = 'https://api.sandbox.mangopay.com';
        break;
    }

    // Create instance of MangoPayApi SDK
    $this->api = new \MangoPay\MangoPayApi();
    $this->api->Config->BaseUrl = $base_url;
    $this->api->Config->ClientId = $client_id;
    $this->api->Config->ClientPassword = $client_pass;
    $this->api->Config->TemporaryFolder = \Drupal::service('file_system')->getTempDirectory();
    $this->api->Config->DebugMode = $this->getConfiguration()['debug'];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'client_id' => '',
      'client_pass' => '',
      'tag' => 'commerce_mangopay_dpi',
      'apple_pay_merchant_id' => '',
      'apple_pay_merchant_id_cert_path' => '',
      'apple_pay_merchant_id_key_path' => '',
      'apple_pay_merchant_id_domain' => '',
      '3ds_mode' => 'DEFAULT',
      'store_cards' => 0,
      'debug' => 0,
      'google_pay' => [
        'merchant_id' => '',
        'merchant_name' => '',
        'allowed_auth_methods' => [],
        'allowed_card_networks' => [],
      ]
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    // The PaymentInformation pane uses payment method labels
    // for on-site gateways, the display label is unused.
    $form['display_label']['#access'] = FALSE;

    $form['store_cards'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Store cards'),
      '#description' => $this->t('Allow customers to store cards for future use. WARNING: Make sure you have your users\' consent before enabling this option.'),
      '#default_value' => $this->configuration['store_cards'],
      '#required' => FALSE,
    ];

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Id'),
      '#description' => $this->t('Please enter your MANGOPAY client id applicable to the environment (mode) you\'re using.'),
      '#default_value' => $this->configuration['client_id'],
      '#required' => TRUE,
    ];

    $form['client_pass'] = [
      '#type' => 'password',
      '#title' => $this->t('Client Password'),
      '#description' => $this->t('Please enter your MANGOPAY client password applicable to the environment (mode) you\'re using.'),
      '#default_value' => $this->configuration['client_pass']
    ];

    $form['tag'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tag'),
      '#description' => $this->t('Standard tag to mark all MANGOPAY resources with. Used to identify resources \'owned\' by this payment gateway. Please note that once set, it\'s not recommended to change this value.'),
      '#default_value' => $this->configuration['tag'],
      '#required' => TRUE,
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug mode'),
      '#description' => $this->t('Enable debug mode to log all requests and responses to the MANGOPAY API. This is useful for troubleshooting.'),
      '#default_value' => $this->configuration['debug'],
      '#required' => FALSE,
    ];

    $form['apple_pay_merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Pay merchant Id'),
      '#description' => $this->t('This is will only be efective if your account has Apple Pay enabled and correctly configured.'),
      '#default_value' => $this->configuration['apple_pay_merchant_id'],
    ];

    $form['apple_pay_merchant_id_cert_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Pay merchant Id certificate path'),
      '#description' => $this->t('Filesystem path to your Apple Pay merchant Id certificate'),
      '#default_value' => $this->configuration['apple_pay_merchant_id_cert_path'],
    ];

    $form['apple_pay_merchant_id_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Pay merchant Id key path'),
      '#description' => $this->t('Filesystem path to your Apple Pay merchant Id key'),
      '#default_value' => $this->configuration['apple_pay_merchant_id_key_path'],
    ];

    $form['apple_pay_merchant_id_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Apple Pay merchant Id domain'),
      '#description' => $this->t('Domain name validated to make Apple Pay payments'),
      '#default_value' => $this->configuration['apple_pay_merchant_id_domain'],
    ];

    $form['3ds_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('3D Secure mode'),
      '#description' => $this->t('3D Secure mode to use for payments: DEFAULT, FORCE or NO_CHOICE'),
      '#options' => [
        'DEFAULT' => $this->t('DEFAULT'),
        'FORCE' => $this->t('FORCE'),
        'NO_CHOICE' => $this->t('NO_CHOICE'),
      ],
      '#default_value' => $this->configuration['3ds_mode'],
    ];

    $form['google_pay'] = [
      '#type' => 'details',
      '#title' => $this->t('Google Pay Settings'),
      '#open' => false,
    ];

    $form['google_pay']['merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant ID'),
      '#default_value' => $this->configuration['google_pay']['merchant_id'],
    ];

    $form['google_pay']['merchant_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant name'),
      '#default_value' => $this->configuration['google_pay']['merchant_name'],
    ];

    $form['google_pay']['allowed_auth_methods'] = [
      '#type' => 'checkboxes',
      '#options' => [
        'PAN_ONLY' => $this->t('PAN_ONLY'),
        'CRYPTOGRAM_3DS' => $this->t('CRYPTOGRAM_3DS'),
      ],
      '#title' => $this->t('Allowed auth methods'),
      '#description' => $this->t('The supported authentication methods'),
      '#default_value' => $this->configuration['google_pay']['allowed_auth_methods'] ?? [],
    ];

    $form['google_pay']['allowed_card_networks'] = [
      '#type' => 'checkboxes',
      '#options' => [
        'VISA' => $this->t('VISA'),
        'MASTERCARD' => $this->t('MASTERCARD'),
      ],
      '#title' => $this->t('Allowed card networks'),
      '#description' => $this->t('The card networks supported by Mangopay on Google Pay'),
      '#default_value' => $this->configuration['google_pay']['allowed_card_networks'] ?? [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    // Split 3ds_whitelist

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['client_id'] = $values['client_id'];
      if (!empty($values['client_pass'])) {
        $this->configuration['client_pass'] = $values['client_pass'];
      }
      $this->configuration['apple_pay_merchant_id'] = $values['apple_pay_merchant_id'];
      $this->configuration['apple_pay_merchant_id_cert_path'] = $values['apple_pay_merchant_id_cert_path'];
      $this->configuration['apple_pay_merchant_id_key_path'] = $values['apple_pay_merchant_id_key_path'];
      $this->configuration['apple_pay_merchant_id_domain'] = $values['apple_pay_merchant_id_domain'];
      $this->configuration['tag'] = $values['tag'];
      $this->configuration['store_cards'] = $values['store_cards'];
      $this->configuration['3ds_mode'] = $values['3ds_mode'];
      $this->configuration['debug'] = $values['debug'];
      $this->configuration['google_pay']['merchant_id'] = $values['google_pay']['merchant_id'];
      $this->configuration['google_pay']['merchant_name'] = $values['google_pay']['merchant_name'];
      $this->configuration['google_pay']['allowed_auth_methods'] = $values['google_pay']['allowed_auth_methods'];
      $this->configuration['google_pay']['allowed_card_networks'] = $values['google_pay']['allowed_card_networks'];
    }
  }

  /**
   * @param $values
   * @return string
   */
  private static function valuesString($values, $merge_tuples = FALSE) {
    if (!empty($values)) {
      if (!$merge_tuples) {
        return implode("\n", $values);
      }

      $tuples = [];
      foreach($values as $key => $value) {
        $tuples[] = $key . '|'  . $value;
      }
      return implode("\n", $tuples);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details) {
    $required_keys = [
      'user_id', 'wallet_id', 'cardholder_country', 'cardholder_name'
    ];
    foreach ($required_keys as $required_key) {
      if (empty($payment_details[$required_key])) {
        throw new \InvalidArgumentException(sprintf('$payment_details must contain the %s key.', $required_key));
      }
    }

    // Set remote User Id on the user object.
    $owner = $payment_method->getOwner();
    if ($owner && $owner->isAuthenticated()) {
      $this->setRemoteCustomerId($owner, $payment_details['user_id']);
      $owner->save();
    }

    // Set relevant details on payment method object.
    switch($payment_method->bundle()) {

      case 'commerce_mangopay_dpi_credit_card':
        $required_keys = [
          'card_type', 'card_alias', 'card_id', 'expiration'
        ];
        foreach ($required_keys as $required_key) {
          if (empty($payment_details[$required_key])) {
            throw new \InvalidArgumentException(sprintf('$payment_details must contain the %s key.', $required_key));
          }
        }

        $payment_method->user_id = $payment_details['user_id'];
        $payment_method->wallet_id = $payment_details['wallet_id'];
        $payment_method->card_type = $payment_details['card_type'];
        $payment_method->card_number = $payment_details['card_alias'];
        $payment_method->card_exp_month = $payment_details['expiration']['month'];
        $payment_method->card_exp_year = $payment_details['expiration']['year'];
        $payment_method->currency_code = $payment_details['currency_code'];
        $payment_method->name = $payment_details['cardholder_name'];
        $payment_method->country = $payment_details['cardholder_country'];
        $payment_method->nationality = '';
        $payment_method->dob = '';
        $payment_method->setRemoteId($payment_details['card_id']);
        $payment_method->setExpiresTime(CreditCard::calculateExpirationTimestamp($payment_details['expiration']['month'], $payment_details['expiration']['year']));
        $payment_method->save();
        break;
      case 'commerce_mangopay_dpi_apple_pay':

        $required_keys = [
          'transaction_id', 'network', 'token_data'
        ];
        foreach ($required_keys as $required_key) {
          if (empty($payment_details[$required_key])) {
            throw new \InvalidArgumentException(sprintf('$payment_details must contain the %s key.', $required_key));
          }
        }

        $payment_method->user_id = $payment_details['user_id'];
        $payment_method->wallet_id = $payment_details['wallet_id'];
        $payment_method->currency_code = $payment_details['currency_code'];
        $payment_method->name = $payment_details['cardholder_name'];
        $payment_method->country = $payment_details['cardholder_country'];
        $payment_method->nationality = '';
        $payment_method->dob = '';
        $payment_method->setRemoteId($payment_details['transaction_id']);
        $payment_method->setReusable(false);
        $payment_method->setExpiresTime(strtotime('now + 1 hour')); // Expires after 1 hour

        $payment_method->transaction_id = $payment_details['transaction_id'];
        $payment_method->network = $payment_details['network'];
        $payment_method->token_data = $payment_details['token_data'];

        $payment_method->save();
        break;

      case 'commerce_mangopay_dpi_google_pay':
        $required_keys = ['token_data'];

        foreach ($required_keys as $required_key) {
          if (empty($payment_details[$required_key])) {
            throw new \InvalidArgumentException(sprintf('$payment_details must contain the %s key.', $required_key));
          }
        }

        $payment_method->user_id = $payment_details['user_id'];
        $payment_method->wallet_id = $payment_details['wallet_id'];
        $payment_method->currency_code = $payment_details['currency_code'];
        $payment_method->name = $payment_details['cardholder_name'];
        $payment_method->country = $payment_details['cardholder_country'];
        $payment_method->token_data = $payment_details['token_data'];
        $payment_method->setReusable(false);
        $payment_method->setExpiresTime(strtotime('now + 1 hour'));
        $payment_method->save();
        break;
      default:
        throw new \InvalidArgumentException(sprintf('Unsupported payment method: %s', $payment_method->bundle()));

    }
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    // Delete the remote record here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // Delete the local entity.
    $payment_method->delete();

    // TODO: Instruct MANGOPAY API to remove the credit card? Is this possible?
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    // Payment must be saved on this step in order to generate unique ID
    $payment->save();

    // Validate payment state. We allow only NEW payments here.
    $state = $payment->getState()->value;
    if (!in_array($state, ['new'])) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Payment has been already processed');
      throw new PaymentGatewayException('Payment has been already processed.');
    }

    // Validate payment method
    $payment_method = $payment->getPaymentMethod();
    if (empty($payment_method)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('No payment method specified.');
      throw new PaymentGatewayException('No payment method specified.');
    }

    if ($payment_method->isExpired()) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('The provided payment method has expired.');
      throw new HardDeclineException('The provided payment method has expired.');
    }

    // Basic information
    $order = $payment->getOrder();
    $amount = doubleval($payment->getAmount()->getNumber()) * 100;
    $currency_code = $payment->getAmount()->getCurrencyCode();
    $user_id = $payment_method->user_id->value;
    $wallet_id = $payment_method->wallet_id->value;

    // Set relevant details on payment method object.
    switch ($payment_method->bundle()) {
      case 'commerce_mangopay_dpi_credit_card':
        $card_id = $payment_method->getRemoteId();

        // Determine if we should force secure mode based on the default threshold.
        $payment_gateway_configuration = $this->getConfiguration();
        $secure_mode = $this->configuration['3ds_mode'];

        // How to check if which flow we have to redirect to? Get current URL?
        $route_name = \Drupal::routeMatch()->getRouteName();
        $return = NULL;
        if ($route_name == 'entity.commerce_payment.add_form') {
          $return = Url::fromRoute('entity.commerce_payment.collection', ['commerce_order' => $order->id()])->toString();
        }

        $payin_arguments = [
          'user_id' => $user_id,
          'wallet_id' => $wallet_id,
          'card_id' => $card_id,
          'amount' => $amount,
          'currency_code' => $currency_code,
          'secure_mode' => $secure_mode,
          'secure_mode_return_url' => $this->getSecureModeReturnUrl($order, $payment),
          'ip_address' => $order->getIpAddress(),
          'browser_info' => $this->getBrowserInfo(),
          'statement_descriptor' => ''
        ];
        \Drupal::moduleHandler()->alter('commerce_mangopay_dpi_pay_in', $payin_arguments, $order, $payment);

        try {
          $payin = $this->createDirectPayIn(
            $payin_arguments['user_id'],
            $payin_arguments['wallet_id'],
            $payin_arguments['card_id'],
            $payin_arguments['amount'],
            $payin_arguments['currency_code'],
            $payin_arguments['ip_address'],
            $payin_arguments['browser_info'],
            $payin_arguments['secure_mode_return_url'],
            $payin_arguments['secure_mode'],
            $payin_arguments['statement_descriptor']
          );

          // Allow other modules to react to payin.
          \Drupal::moduleHandler()->invokeAll('commerce_mangopay_dpi_pay_in', [$payin, $payment]);
        } catch (\Exception $e) {
          $payment->getPaymentMethod()->setReusable(false)->save();
          $payment->getOrder()->set('payment_method', NULL)->save();

          \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to create Direct PayIn for card %s: %s: %s', $payin_arguments['card_id'], $e->getCode(), $e->getMessage()));
          throw new PaymentGatewayException('Payment has not been processed correctly. Please try again with a different card.');
        }
        break;

      case 'commerce_mangopay_dpi_apple_pay':
        $transaction_id = $payment_method->getRemoteId();
        $network = $payment_method->network->value;
        $token_data = $payment_method->token_data->value;

        $payin_arguments = [
          'user_id' => $user_id,
          'wallet_id' => $wallet_id,
          'amount' => $amount,
          'currency_code' => $currency_code,
          'transaction_id' => $transaction_id,
          'network' =>  $network,
          'token_data' => $token_data,
          'statement_descriptor' => '',
        ];
        \Drupal::moduleHandler()->alter('commerce_mangopay_dpi_pay_in', $payin_arguments);

        try {
          $payin = $this->createApplePayDirectPayIn(
            $payin_arguments['user_id'],
            $payin_arguments['wallet_id'],
            $payin_arguments['amount'],
            $payin_arguments['currency_code'],
            $payin_arguments['transaction_id'],
            $payin_arguments['network'],
            $payin_arguments['token_data'],
            $payin_arguments['statement_descriptor']
          );

          // Allow other modules to react to payin.
          \Drupal::moduleHandler()->invokeAll('commerce_mangopay_dpi_pay_in', [$payin, $payment]);
        } catch (\Exception $e) {
          $payment->getOrder()->set('payment_method', NULL)->save();

          \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to create Direct PayIn for transaction %s: %s: %s', $payin_arguments['transaction_id'], $e->getCode(), $e->getMessage()));
          throw new PaymentGatewayException('Payment has not been processed correctly. Please try again with a different card.');
        }
        break;

      case 'commerce_mangopay_dpi_google_pay':
        $user = $this->getUser($user_id);
        $order = $payment->getOrder();
        $billing_profile = $order?->getBillingProfile();
        $user_address = $billing_profile?->get('address')->first();

        $billing = new Billing();
        $billing->FirstName = $user->FirstName;
        $billing->LastName = $user->LastName;

        // Only create and attach the address when all required fields are present.
        if ($user_address) {
          $country = $user_address?->get('country_code')->getValue();
          $city = $user_address?->get('locality')->getValue();
          $region = $user_address?->get('administrative_area')->getValue();
          $postalCode = $user_address?->get('postal_code')->getValue();
          $line1 = $user_address?->get('address_line1')->getValue();
          $line2 = $user_address?->get('address_line2')->getValue();

          if (!empty($country) && !empty($city) && !empty($region) && !empty($postalCode) && !empty($line1)) {
            $address = new Address();
            $address->Country = $country;
            $address->City = $city;
            $address->Region = $region;
            $address->PostalCode = $postalCode;
            $address->AddressLine1 = $line1;
            // AddressLine2 is optional; only set if present.
            if (!empty($line2)) {
              $address->AddressLine2 = $line2;
            }

            // Assign the address to billing.
            $billing->Address = $address;
          }
        }

        $payin_arguments = [
          'user_id' => $user_id,
          'wallet_id' => $wallet_id,
          'transaction_id' => $payment_method->getRemoteId(),
          'amount' => $amount,
          'currency_code' => $currency_code,
          'ip_address' => $order->getIpAddress(),
          'statement_descriptor' => 'GooglePay',
          'token_data' => $payment_method->token_data->value,
          'browser_info' => $this->getBrowserInfo(),
          'billing' => $billing,
          'secure_mode_return_url' => $this->getSecureModeReturnUrl($order, $payment)
        ];
        \Drupal::moduleHandler()->alter('commerce_mangopay_dpi_pay_in', $payin_arguments);

        try {
          $payin = $this->createGooglePayDirectPayIn(
            $payin_arguments['user_id'],
            $payin_arguments['wallet_id'],
            $payin_arguments['amount'],
            $payin_arguments['currency_code'],
            $payin_arguments['token_data'],
            $payin_arguments['statement_descriptor'],
            $payin_arguments['ip_address'],
            $payin_arguments['browser_info'],
            $payin_arguments['billing'],
            $payin_arguments['secure_mode_return_url'],
          );

          // Allow other modules to react to payin.
          \Drupal::moduleHandler()->invokeAll('commerce_mangopay_dpi_pay_in', [$payin, $payment]);
        } catch (\Exception $e) {
          $payment_method = $payment->getPaymentMethod();
          $payment_method->delete();
          $payment->getOrder()->set('payment_method', NULL)->save();

          \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Uncable to create Direct PayIn for transaction %s: %s: %s', $payin_arguments['transaction_id'], $e->getCode(), $e->getMessage()));
          throw new PaymentGatewayException('Payment has not been processed correctly. Please try again with a different card.');
        }
        break;
      default:
        throw new PaymentGatewayException('Unrecognized payment method. Please try again with a different card.');
    }

    switch($payin->Status) {
      case \MangoPay\PayInStatus::Failed:
        $payment->getPaymentMethod()->setReusable(false)->save();
        $payment->getOrder()->set('payment_method', NULL)->save();
        $payment->setRemoteId($payin->Id);
        $payment->save();

        // TODO: Display more descriptive messages here for the transaction errors: https://docs.mangopay.com/guide/errors
        \Drupal::logger('commerce_mangopay_dpi')->warning(sprintf('Pay In Failure: %s: %s', $payin->ResultCode, $payin->ResultMessage));
        throw new HardDeclineException('Payment has not been processed correctly. Please try again with a different card.');

      // 3DS / Secure Mode, needs further processing.
      case \MangoPay\PayInStatus::Created:
        $payment->setRemoteId($payin->Id);
        $payment->save();

        if ($payin->ExecutionDetails->SecureModeNeeded && !empty($payin->ExecutionDetails->SecureModeRedirectURL)) {
          // Redirect to 3D Secure, please in order to handle customer auth.
          // TODO: Do we need to validate the URL?
          throw new NeedsRedirectException($payin->ExecutionDetails->SecureModeRedirectURL);
        }
        else {
          $payment->getPaymentMethod()->setReusable(false)->save();
          $payment->getOrder()->set('payment_method', NULL)->save();
          $payment->setRemoteId($payin->Id);
          $payment->save();

          // Always make sure to deregister card in Mangopay in case of failure
          if ($payment->getPaymentMethod()->bundle() == 'commerce_mangopay_dpi_credit_card') {
            try {
              $this->deregisterCard($payment->getPaymentMethod()->getRemoteId());
            } catch (\Exception $e) {
              \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to deactivate card %s: %s', $payment->getPaymentMethod()->getRemoteId(), $e->getMessage()));
            }
          }

          \Drupal::logger('commerce_mangopay_dpi')->warning(sprintf('No SecureModeRedirectURL provided for Created response: %s: %s', $payin->ResultCode, $payin->ResultMessage));
          throw new PaymentGatewayException('Payment has not been processed correctly. Please try again with a different card.');
        }
      break;

      // Success, mark payment as completed and continue
      case \MangoPay\PayInStatus::Succeeded:
        $payment->setState('completed');
        $payment->setCompletedTime(\Drupal::time()->getRequestTime());
        $payment->setRemoteId($payin->Id);
        $payment->save();

        // If cards are not stored as default, please deregister the card and set as non-reusable
        if (!$payment_gateway_configuration['store_cards']
        && $payment->getPaymentMethod()->bundle() == 'commerce_mangopay_dpi_credit_card') {
          $payment->getPaymentMethod()->setReusable(false)->save();
          try {
            $this->deregisterCard($payment->getPaymentMethod()->getRemoteId());
          } catch (\Exception $e) {
            \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to deactivate card  %s:  %s', $payment->getPaymentMethod()->getRemoteId(), $e->getMessage()));
          }
        }

      break;

      default:
        $payment->getPaymentMethod()->setReusable(false)->save();
        $payment->getOrder()->set('payment_method', NULL)->save();
        $payment->setRemoteId($payin->Id);
        $payment->save();

        // Always make sure to deregister card in Mangopay in case of unexpected failure.
        // We don't want to keep cards in Mangopay if the payment has failed.
        if ($payment->getPaymentMethod()->bundle() == 'commerce_mangopay_dpi_credit_card') {
          try {
            $this->deregisterCard($payment->getPaymentMethod()->getRemoteId());
          } catch (\Exception $e) {
            \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to deactivate card  %s:  %s', $payment->getPaymentMethod()->getRemoteId(), $e->getMessage()));
          }
        }

        \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Pay In Error: %s: %s', $payin->ResultCode, $payin->ResultMessage));
        throw new PaymentGatewayException('Payment has not been processed correctly. Please try again with a different card.');
      break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request)
  {
    // This is used to mark the pay-in as completed via the webhook.
    // This helps if for any reason the customer won't return to the site after
    // 3DS step. This can happen sometimes. In such case, the payment
    // wouldn't be completed and order wouldn't be placed.

    // Validate request
    if (empty($request->query->get('EventType'))) {
      \Drupal::logger('commerce_mangopay_dpi')->warning("Missing EventType");
      return new Response('Missing required parameter', 403);
    }

    if (empty($request->query->get('RessourceId'))) {
      \Drupal::logger('commerce_mangopay_dpi')->warning("Missing RessourceId");
      return new Response('Missing required parameter', 403);
    }

    if (empty($request->query->get('Date'))) {
      \Drupal::logger('commerce_mangopay_dpi')->warning("Missing Date");
      return new Response('Missing required parameter', 403);
    }

    if (!is_numeric($request->query->get('Date'))) {
      \Drupal::logger('commerce_mangopay_dpi')->warning(
        "Incorrect Date type for pay-in success notification (non-numeric)"
      );
      return new Response('Invalid parameter', 403);
    }

    if ($request->query->get('EventType') != 'PAYIN_NORMAL_SUCCEEDED') {
      \Drupal::logger('commerce_mangopay_dpi')->warning(
        "Incorrect EventType for pay-in success notification"
      );
      return new Response('Invalid parameter', 403);
    }

    // Get payment by externalID and validate it against MANGOPAY

    /** @var \Drupal\commerce_payment\PaymentStorageInterface $payment_storage **/
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $payment_storage->loadByRemoteId($request->query->get('RessourceId'));

    if (empty($payment)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning(
        "Payment with requested Resource Id doesn't exist."
      );
      return new Response('Payment not found', 404);
    }

    // If payment is not completed already, mark it as so
    if (!in_array($payment->getState()->getId(), ['new'])) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Payment has been already processed');
      return new Response('Payment has been already processed', 200);
    }

    try {
      $existing_pay_in = $this->getPayIn($request->query->get('RessourceId'));
    } catch (\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->warning(
        sprintf('Unable to fetch existing MANGOPAY pay-in: %s: %s', $e->getCode(), $e->getMessage())
      );
      return new Response('Mangopay Pay-in not found', 404);
    }

    if ($existing_pay_in->Status != TransactionStatus::Succeeded) {
      \Drupal::logger('commerce_mangopay_dpi')->warning(
        'Status of MANGOPAY Pay-in is not SUCCEEDED'
      );
      return new Response('MANGOPAY Pay-in is in invalid state', 403);
    }

    $payment->setState('completed');
    $payment->setCompletedTime(\Drupal::time()->getRequestTime());
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    // Perform the refund request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    $remote_id = $payment->getRemoteId();

    // Call MANGOPAY api to refund payment
    try {
      $result = $this->createRefund($remote_id, $amount);
      if ($result->Status != 'SUCCEEDED') {
        // Put on deferred queue. This will be retried the next day.
        \Drupal::logger('commerce_mangopay_dpi')->warning(
          sprintf('Refund for payment %s deferred: %s', $payment->id(), $result->ResultMessage));
        throw new PaymentGatewayException($result->ResultMessage);
      }
    } catch (\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to refund MANGOPAY pay-in: %s: %s', $e->getCode(), $e->getMessage()));
      throw new PaymentGatewayException($e->getMessage());
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  public function createRefund($pay_in_id, Price $refund_amount) {
    // Get user Id of the payin
    $pay_in = $this->api->PayIns->Get($pay_in_id);
    $user_id = $pay_in->AuthorId;

    // Create refund
    $refund = new \MangoPay\Refund();
    $refund->AuthorId = $user_id;

    $refund->DebitedFunds = new \MangoPay\Money();
    $refund->DebitedFunds->Currency = $refund_amount->getCurrencyCode();
    $refund->DebitedFunds->Amount = $refund_amount->getNumber() * 100;

    $refund->Fees = new \MangoPay\Money();
    $refund->Fees->Currency = $refund_amount->getCurrencyCode();
    $refund->Fees->Amount = 0;

    return $this->api->PayIns->CreateRefund($pay_in_id, $refund);
  }

  /**
   * {@inheritdoc}
   */
  public function createNaturalUser($first_name, $last_name, $email, $country, $address_line1 = '', $address_line2 = '', $city = '', $postal_code = '', $region = '', $tag = '') {
    $user = new \MangoPay\UserNaturalSca();
    $user->FirstName = $first_name;
    $user->LastName = $last_name;
    $user->Email = $email;
    $user->CountryOfResidence = $country;
    $user->UserCategory = "PAYER";
    $user->TermsAndConditionsAccepted = true;

    if (!empty($address_line1)
    && !empty($city)
    && !empty($postal_code)) {
      $user->Address = new \MangoPay\Address();
      $user->Address->AddressLine1 = $address_line1;
      $user->Address->AddressLine2 = $address_line2;
      $user->Address->City = $city;
      $user->Address->PostalCode = $postal_code;
      $user->Address->Region = $region;
      $user->Address->Country = $country;
    }

    $user->Tag = $tag;
    return $this->api->Users->Create($user);
  }

  /**
   * {@inheritdoc}
   */
  public function getUser($user_id) {
    return $this->api->Users->GetSca($user_id);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveNaturalUser($first_name, $last_name, $email, $country, $address_line1 = '', $address_line2 = '', $city = '', $postal_code = '', $region = '', $tag = '', $user = NULL) {
    $mangopay_user = NULL;

    // Check if the currently logged in user has already Remote Id set.
    // If yes, try to fetch the MANGOPAY user from the API.
    if ($user) {
      /** @var \Drupal\commerce\Plugin\Field\FieldType\RemoteIdFieldItemListInterface $remote_ids */
      $remote_ids = $user->get('commerce_remote_id');
      $mangopay_remote_id = $remote_ids->getByProvider($this->parentEntity->id() . '|' . $this->getMode());
      if (!empty($mangopay_remote_id)) {
        try {
          $mangopay_user = $this->getUser($mangopay_remote_id);
        } catch (\Exception $e) {
          \Drupal::logger('commerce_mangopay_dpi')->notice(sprintf('Unable to retrieve MANGOPAY user %s while registering a card: %s: %s', $mangopay_remote_id, $e->getCode(), $e->getMessage()));
        }
      }
    }

    // IF no MANGOPAY user retrieved, try to find it by exisiting orders.
    // If there is an order with given email, retrieve the payment object
    // and attempt MANGOPAY wallet retrieval using that.

    // This piece of code is used for returning anonymous users, so that we don't
    // have an accumulation of user in MANGOPAY with the same email address.
    if (!$mangopay_user) {
      $order_query = \Drupal::entityQuery('commerce_order');
      $order_query->accessCheck(FALSE);
      $order_query->condition('mail', $email);
      $order_query->range(0, 10); // Try for payment object on max 10 last orders
      $order_query->sort('created', 'DESC');
      $order_query_result = $order_query->execute();

      if (!empty($order_query_result)) {
        $payment_query = \Drupal::entityQuery('commerce_payment');
        $payment_query->accessCheck(FALSE);
        $payment_query->condition('order_id', array_keys($order_query_result), 'IN');
        $payment_query->condition('payment_gateway', $this->parentEntity->id());
        $payment_query->condition('state', 'completed');
        $payment_query->range(0, 1); # Any latest successful payment is enough.
        $payment_query_result = $payment_query->execute();

        if (!empty($payment_query_result)) {
          /** @var \Drupal\commerce_payment\PaymentStorageInterface $payment_storage **/
          $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

          /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
          $payment = $payment_storage->load(reset($payment_query_result));
          $remote_id = $payment->getRemoteId();
          try {
            $exisiting_pay_in = $this->getPayIn($remote_id);
            $exisiting_wallet = $this->getWallet($exisiting_pay_in->CreditedWalletId);

            foreach ($exisiting_wallet->Owners as $owner_id) {
              $exisiting_user = $this->getUser($owner_id);
              if ($exisiting_user->Email == $email) {
                // User found. Break out from the loop.
                $mangopay_user = $exisiting_user;
                break;
              }
            }
          } catch (\Exception $e) {
            \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to determine exisiting MANGOPAY user: %s: %s', $e->getCode(), $e->getMessage()));
          }
        }
      }
    }

    // IF no MANGOPAY user retrieved, try to create it.
    if (!$mangopay_user) {
      // Create user for payment if there is no Remote Id already stored in Drupal.
      try {
        $mangopay_user = $this->createNaturalUser($first_name, $last_name, $email, $country, $address_line1, $address_line2, $city, $postal_code, $region, $tag);

        // Set MANGOPAY User Id on the user object if the account is logged in.
        if ($user) {
          /** @var \Drupal\commerce\Plugin\Field\FieldType\RemoteIdFieldItemListInterface $remote_ids */
          $remote_ids = $user->get('commerce_remote_id');
          $remote_ids->setByProvider($this->parentEntity->id() . '|' . $this->getMode(), $mangopay_user->Id);
          $user->save();
        }
      } catch (\Exception $e) {
        \Drupal::logger('commerce_mangopay_dpi')->error(sprintf('Unable to create MANGOPAY user while registering a card: %s: %s', $e->getCode(), $e->getMessage()));
      }
    }

    return $mangopay_user;
  }

  /**
   * {@inheritdoc}
   */
  public function createWallet($user_id, $currency_code, $description, $tag = '') {
    $wallet = new \MangoPay\Wallet();
    $wallet->Owners = [$user_id];
    $wallet->Description = $description;
    $wallet->Currency = $currency_code;
    $wallet->Tag = $tag;
    return $this->api->Wallets->Create($wallet);
  }

  /**
   * {@inheritdoc}
   */
  public function getWallets($user_id) {
    return $this->api->Users->GetWallets($user_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getWallet($wallet_id) {
    return $this->api->Wallets->Get($wallet_id);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveWallet($user_id, $currency_code, $tag = '')
  {
    /** @var \Drupal\commerce_price\Entity\Currency $currency */
    $currency = \Drupal::entityTypeManager()->getStorage('commerce_currency')->load($currency_code);
    if (empty($currency)) {
      \Drupal::logger('commerce_mangopay_dpi')->warning('Currency code is invalid');
      return FALSE;
    }

    // Check if user already has an active wallet for Drupal Commerce with specified currency.
    // If yes, use it. Otherwise, create a new one.
    $mangopay_wallet = NULL;
    try {
      $wallets = $this->getWallets($user_id);
      foreach ($wallets as $wallet) {
        if (
          $wallet->Tag == $tag
          && $wallet->Currency == $currency_code
        ) {
          return $wallet;
        }
      }
    } catch (\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->notice(sprintf('Unable to retrieve MANGOPAY wallets for user %s', $user_id));
    }

    // If wallet not returned, please create a new one.
    try {
      $mangopay_wallet = $this->createWallet($user_id, $currency_code, sprintf('%s wallet', $currency->getName()), $tag);
      return $mangopay_wallet;
    } catch (\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')
      ->error(sprintf('Unable to create MANGOPAY wallet for user %s while registering a card: %s: %s', $user_id, $e->getCode(), $e->getMessage()));
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createCardRegistration($user_id, $currency_code, $card_type, $tag = '') {
    $cardRegister = new \MangoPay\CardRegistration();
    $cardRegister->UserId = $user_id;
    $cardRegister->Currency = $currency_code;
    $cardRegister->CardType = $card_type;
    $cardRegister->Tag = $tag;
    return $this->api->CardRegistrations->Create($cardRegister);
  }

  /**
   * {@inheritdoc}
   */
  public function deregisterCard($card_id) {
    $card = new \MangoPay\Card();

    $card->Id = $card_id;
    $card->Active = false;

    return $this->api->Cards->Update($card);
  }

  /**
   * {@inheritdoc}
   */
  public function createDirectPayIn($user_id, $wallet_id, $card_id, $amount, $currency_code, $ip_address, $browser_info, $secure_mode_return_url, $secure_mode = 'DEFAULT', $statement_descriptor = '') {

    // Create pay-in CARD DIRECT
    $pay_in = new \MangoPay\PayIn();
    $pay_in->CreditedWalletId = $wallet_id;
    $pay_in->AuthorId = $user_id;
    $pay_in->DebitedFunds = new \MangoPay\Money();
    $pay_in->DebitedFunds->Amount = $amount;
    $pay_in->DebitedFunds->Currency = $currency_code;
    $pay_in->Fees = new \MangoPay\Money();
    $pay_in->Fees->Amount = 0;
    $pay_in->Fees->Currency = $currency_code;
    $pay_in->StatementDescriptor = $statement_descriptor;

    // Payment type as CARD
    // TODO: Do we have to make a call here? Why not storing this?
    // TODO: Shall we validate in case card no longer exists or is expired?
    $card = $this->api->Cards->Get($card_id);

    $pay_in->PaymentDetails = new \MangoPay\PayInPaymentDetailsCard();
    $pay_in->PaymentDetails->CardType = $card->CardType;
    $pay_in->PaymentDetails->CardId = $card->Id;
    $pay_in->PaymentDetails->IpAddress = $ip_address;
    $pay_in->PaymentDetails->BrowserInfo = $browser_info;

    // Execution type as DIRECT
    $pay_in->ExecutionDetails = new \MangoPay\PayInExecutionDetailsDirect();

    // 3D Secure configuration
    $pay_in->ExecutionDetails->SecureMode = $secure_mode;
    $pay_in->ExecutionDetails->SecureModeReturnURL = $secure_mode_return_url;

    return $this->api->PayIns->Create($pay_in);
  }


  /**
   * {@inheritdoc}
   */
  public function createApplePayDirectPayIn($user_id, $wallet_id, $amount, $currency_code, $transaction_id, $network, $token_data, $statement_descriptor = '')
  {

    // Create pay-in
    $pay_in = new \MangoPay\PayIn();
    $pay_in->CreditedWalletId = $wallet_id;
    $pay_in->AuthorId = $user_id;
    $pay_in->DebitedFunds = new \MangoPay\Money();
    $pay_in->DebitedFunds->Amount = $amount;
    $pay_in->DebitedFunds->Currency = $currency_code;
    $pay_in->Fees = new \MangoPay\Money();
    $pay_in->Fees->Amount = 0;
    $pay_in->Fees->Currency = $currency_code;
    $pay_in->ExecutionDetails = new \MangoPay\PayInExecutionDetailsDirect();

    $pay_in->PaymentDetails = new \MangoPay\PayInPaymentDetailsApplePay();
    $pay_in->PaymentDetails->PaymentData = new \MangoPay\PaymentData();
    $pay_in->PaymentDetails->PaymentData->TransactionId = $transaction_id; //'061EB32181A2D9CA42AD16031B476EEBAA62A9A095AD660E2759FBA52B51A61';
    $pay_in->PaymentDetails->PaymentData->Network = $network; // 'VISA'
    $pay_in->PaymentDetails->PaymentData->TokenData = $token_data ; // "{\"version\":\"EC_v1\",\"data\":\"w4HMBVqNC9ghPP4zncTA\/0oQAsduERfsx78oxgniynNjZLANTL6+0koEtkQnW\/K38Zew8qV1GLp+fLHo+qCBpiKCIwlz3eoFBTbZU+8pYcjaeIYBX9SOxcwxXsNGrGLk+kBUqnpiSIPaAG1E+WPT8R1kjOCnGvtdombvricwRTQkGjtovPfzZo8LzD3ZQJnHMsWJ8QYDLyr\/ZN9gtLAtsBAMvwManwiaG3pOIWpyeOQOb01YcEVO16EZBjaY4x4C\/oyFLWDuKGvhbJwZqWh1d1o9JT29QVmvy3Oq2JEjq3c3NutYut4rwDEP4owqI40Nb7mP2ebmdNgnYyWfPmkRfDCRHIWtbMC35IPg5313B1dgXZ2BmyZRXD5p+mr67vAk7iFfjEpu3GieFqwZrTl3\/pI5V8Sxe3SIYKgT5Hr7ow==\",\"signature\":\"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\",\"header\":{\"ephemeralPublicKey\":\"MFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAE0rs3wRpirXjPbFDQfPRdfEzRIZDWm0qn7Y0HB0PNzV1DDKfpYrnhRb4GEhBF\/oEXBOe452PxbCnN1qAlqcSUWw==\",\"publicKeyHash\":\"saPRAqS7TZ4bAYwzBj8ezDDC55ZolyH1FL+Xc8fd93o=\",\"transactionId\":\"b061eb32181a2d9ca42ad16031b476eebaa62a9a095ad660e2759fba52b51a61\"}}";
    $pay_in->PaymentDetails->StatementDescriptor = $statement_descriptor;

    return $this->api->PayIns->Create($pay_in);
  }

  /**
   * {@inheritdoc}
   */
  public function getPayIn($payin_id) {
    return $this->api->PayIns->Get($payin_id);
  }

  /**
   * Creates a basic PayIn object.
   */
  public function createPayIn($wallet_id, $user_id, $amount, $currency_code): PayIn
  {
    $pay_in = new PayIn();
    $pay_in->CreditedWalletId = $wallet_id;
    $pay_in->AuthorId = $user_id;
    $pay_in->DebitedFunds = new Money();
    $pay_in->DebitedFunds->Amount = $amount;
    $pay_in->DebitedFunds->Currency = $currency_code;
    $pay_in->Fees = new Money();
    $pay_in->Fees->Amount = 0;
    $pay_in->Fees->Currency = $currency_code;

    return $pay_in;
  }

  /**
   * @inheritdoc
   */
  public function createGooglePayDirectPayIn(
    string $user_id,
    string $wallet_id,
    int $amount,
    string $currency_code,
    string $payment_data,
    string $statement_descriptor,
    string $ip_address,
    BrowserInfo $browser_info,
    Billing $billing,
    string $secure_mode_return_url
  ): PayIn {
    $pay_in = $this->createPayIn($wallet_id, $user_id, $amount, $currency_code);

    $pay_in->ExecutionDetails = new PayInExecutionDetailsDirect();
    $pay_in->PaymentType = PayInPaymentType::GooglePayV2;
    $pay_in->CreditedUserId = $user_id;
    $pay_in->Tag = 'Create an Google card direct Payin';

    $pay_in->ExecutionDetails->SecureModeReturnURL = $secure_mode_return_url;
    $pay_in->ExecutionDetails->SecureMode = 'DEFAULT';
    $pay_in->ExecutionDetails->Billing = $billing;

    $payment_details = new PayInPaymentDetailsGooglePay();
    $payment_details->PaymentData = $payment_data;
    $payment_details->StatementDescriptor = $statement_descriptor;
    $payment_details->IpAddress = $ip_address;
    $payment_details->BrowserInfo = $browser_info;

    $pay_in->PaymentDetails = $payment_details;

    return $this->api->PayIns->CreateGooglePay($pay_in);
  }

  /**
   * Returns browser information from the request.
   */
  public function getBrowserInfo(): BrowserInfo
  {
    $client_cookie = \Drupal::request()->cookies->get('commerce_mangopay_dpi_client');
    if (empty($client_cookie)) {
      \Drupal::logger('commerce_mangopay_dpi')->error('Client cookie not supplied');
      throw new PaymentGatewayException('Payment has not been processed correctly.');
    }

    $client = Json::decode($client_cookie);
    if (empty($client)) {
      \Drupal::logger('commerce_mangopay_dpi')->error('Client cookie has not been decoded successfuly: ' . $client_cookie);
      throw new PaymentGatewayException('Payment has not been processed correctly.');
    }

    $client_cookie_keys = [
      'userAgent', 'language', 'javaEnabled', 'colorDepth',
      'screenHeight', 'screenWidth', 'timezoneOffset', 'javaScriptEnabled'
    ];
    foreach ($client_cookie_keys as $key) {
      if (!array_key_exists(
        $key,
        $client
      )) {
        \Drupal::logger('commerce_mangopay_dpi')->error('Client cookie is missing mandatory information');
        throw new PaymentGatewayException('Payment has not been processed correctly.');
      }
    }

    $accept_header = \Drupal::request()->headers->get('Accept');
    if (empty($accept_header)) {
      \Drupal::logger('commerce_mangopay_dpi')->error('Accept header not supplied');
      throw new PaymentGatewayException('Payment has not been processed correctly.');
    }

    // Allow other modules to alter to payin before it is executed
    $browser_info = new BrowserInfo();
    $browser_info->AcceptHeader = $accept_header;
    $browser_info->UserAgent = $client['userAgent'];
    $browser_info->Language = $client['language'];
    $browser_info->JavaEnabled = $client['javaEnabled'];
    $browser_info->ColorDepth = $client['colorDepth'];
    $browser_info->ScreenHeight = $client['screenHeight'];
    $browser_info->ScreenWidth = $client['screenWidth'];
    $browser_info->TimeZoneOffset = $client['timezoneOffset'];
    $browser_info->JavascriptEnabled = $client['javaScriptEnabled'];

    return $browser_info;
  }

  /**
   * Gets return URL based on the current route.
   */
  public function getReturnUrl(?OrderInterface $order): string|null|GeneratedUrl
  {
    $route_name = \Drupal::routeMatch()->getRouteName();
    $return = NULL;

    if ($route_name === 'entity.commerce_payment.add_form') {
      $return = Url::fromRoute('entity.commerce_payment.collection', ['commerce_order' => $order->id()])->toString();
    }

    return $return;
  }

  /**
   * Gets Secure mode return URL for 3D Secure processing.
   */
  public function getSecureModeReturnUrl(?OrderInterface $order, PaymentInterface $payment): string|GeneratedUrl
  {
    return Url::fromRoute('commerce_mangopay_dpi.process_secure_mode', ['commerce_order' => $order->id(), 'commerce_payment' => $payment->id()], ['absolute' => TRUE, 'query' => ['return' => $this->getReturnUrl($order)]])->toString();
  }
}
