<?php

namespace Drupal\commerce_mangopay_dpi\PluginForm\Onsite;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Exception\DeclineException;
use Drupal\commerce_store\StoreStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Locale\CountryManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\commerce\InlineFormManager;
use Drupal\commerce_store\CurrentStoreInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Psr\Log\LoggerInterface;
use Drupal\commerce_payment\PluginForm\PaymentMethodFormBase;

class PaymentMethodAddForm extends PaymentMethodFormBase {

  use StringTranslationTrait;

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

    /**
   * Constructs a new PaymentMethodFormBase.
   *
   * @param \Drupal\commerce_store\CurrentStoreInterface $current_store
   *   The current store.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\commerce\InlineFormManager $inline_form_manager
   *   The inline form manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(CurrentStoreInterface $current_store, EntityTypeManagerInterface $entity_type_manager, InlineFormManager $inline_form_manager, LoggerInterface $logger, RouteMatchInterface $routeMatch) {
    $this->currentStore = $current_store;
    $this->entityTypeManager = $entity_type_manager;
    $this->inlineFormManager = $inline_form_manager;
    $this->logger = $logger;
    $this->routeMatch = $routeMatch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_store.current_store'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.commerce_inline_form'),
      $container->get('logger.channel.commerce_payment'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorElement(array $form, FormStateInterface $form_state) {
    return $form['payment_details'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $this->entity;

    $payment_gateway_id = $payment_method->getPaymentGatewayId();

    /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $payment_method->getPaymentGateway()->getPlugin();
    $mode = $payment_gateway_plugin->getConfiguration()['mode'];
    $client_id = $payment_gateway_plugin->getConfiguration()['client_id'];
    $apple_pay_merchant_id = $payment_gateway_plugin->getConfiguration()['apple_pay_merchant_id'];
    $card_type = 'CB_VISA_MASTERCARD'; // TODO: Are those types only ones supported at the moment?
    switch($mode) {
      case 'production':
        $base_url = 'https://api.mangopay.com';
        break;
      default:
        $base_url = 'https://api.sandbox.mangopay.com';
        break;
    }

    $email = NULL;

    /** @var OrderInterface $order */
    if ($order = $this->routeMatch->getParameter('commerce_order')) {
      $currency_code = $order->getTotalPrice()->getCurrencyCode();
      $email = $order->getEmail();
    }
    else {
      /** @var StoreStorageInterface $store_storage */
      $store_storage = \Drupal::entityTypeManager()->getStorage('commerce_store');
      $store = $store_storage->loadDefault();
      $currency_code = $store->getDefaultCurrencyCode();
    }

    $user = \Drupal::currentUser();
    if (empty($email) && $user->isAuthenticated()) {
      $email = $user->getEmail();
    }

    // Attach container for displaying status and error messages with javascript.
    // Waiting patiently for: https://www.drupal.org/node/77245 but in the meantime...
    $form['status'] = [
      '#theme' => 'status_messages',
      '#display' => 'error',

      // We take an opportunity here to warn the user that this form requires
      // the JavaScript to be enabled (sorry, no progressive enhancement at the mo).
      // If a user has JS enabled, this message will disappear.
      // Doing this also renders a full messages block with actual list items
      // usable from JS level to display errors.
      '#message_list' => ['error' => [
        $this->t('This form requires JavaScript.'),
        $this->t('Please make sure your browser is up to date and JavaScript is not disabled.')]],
      '#status_headings' => ['error' => $this->t('Error message')]];

    $form['#tree'] = TRUE;
    $form['payment_details'] = [
      '#parents' => array_merge($form['#parents'], ['payment_details']),
      '#type' => 'container',
      '#payment_method_type' => $payment_method->bundle()
    ];

    // Build a month select list that shows months with a leading zero.
    $months = [0 => $this->t('MM')];
    for ($i = 1; $i < 13; $i++) {
      $month = str_pad($i, 2, '0', STR_PAD_LEFT);
      $months[$month] = $month;
    }
    // Build a year select list that uses a 4 digit key with a 2 digit value.
    $current_year = date('y');
    $years = [0 => $this->t('YY')];
    for ($i = 0; $i < 10; $i++) {
      $years[$current_year + $i] = $current_year + $i;
    }

    // Attach JS script and related settings.
    $form['#attached']['library'][] = 'commerce_payment/payment_method_form';
    $form['#attached']['library'][] = 'commerce_mangopay_dpi/payment_method_add_form';
    $form['#attached']['drupalSettings']['commerceMangopay'] = [
      'mode' => $mode,
      'baseUrl' => $base_url,
      'clientId' => $client_id,
      'email' => $email,
      'cardType' => $card_type,
      'currencyCode' => $currency_code,
      'paymentGatewayId' => $payment_gateway_id,
      'applePayMerchantID' => $apple_pay_merchant_id,
      'amount' => $order->getTotalPrice()->getNumber() - $order->getTotalPaid()->getNumber(),
      'googlePaySettings' => [
        'merchantId' => $payment_gateway_plugin->getGooglePaySettings()['merchant_id'],
        'merchantName' => $payment_gateway_plugin->getGooglePaySettings()['merchant_name'],
        'allowedCardNetworks' => array_values($payment_gateway_plugin->getGooglePaySettings()['allowed_card_networks']),
        'allowedAuthMethods' => array_values($payment_gateway_plugin->getGooglePaySettings()['allowed_auth_methods']),
      ]
    ];

    // Attach Apple Pay SDK
    switch($payment_method->bundle()) {
      case 'commerce_mangopay_dpi_credit_card':
        $form['#attached']['library'][] = 'commerce_mangopay_dpi/register_card';

        $form['payment_details']['#attributes']['class'][] = 'credit-card-form';
        $form['payment_details']['#attributes']['class'][] = 'mangopay-dpi-credit-card-form';

        $form['payment_details']['number'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Card number'),
          '#attributes' => ['autocomplete' => 'off'],
          '#maxlength' => 19,
          '#size' => 20,
          '#required' => FALSE, // From the perspective of FAPI this field is not required. We only use it in JavaScript.
          '#attributes' => [
            'class' => ['mangopay-dpi-card-number'],
          ],
          /**
           * Mark as sensitive - Can only be transferred to MANGOPAY directly
           * @see commerce_mangopay_dpi_preprocess_input
           * @see commerce_mangopay_dpi_preprocess_form_element
           */
          '#commerce_mangopay_dpi_sensitive' => TRUE
        ];

        $form['payment_details']['expiration'] = [
          '#type' => 'item',
          '#title' => $this->t('Expiry date'),
          '#required' => TRUE,
          '#attributes' => [
            'class' => ['credit-card-form__expiration'],
          ],
        ];

        $form['payment_details']['expiration']['month'] = [
          '#type' => 'select',
          '#options' => $months,
          '#default_value' => 0,
          '#required' => TRUE,
          '#attributes' => [
            'class' => ['mangopay-dpi-expiration-month'],
          ],
        ];


        $form['payment_details']['expiration']['year'] = [
          '#type' => 'select',
          '#options' => $years,
          '#default_value' => 0,
          '#required' => TRUE,
          '#attributes' => [
            'class' => ['mangopay-dpi-expiration-year'],
          ],
        ];

        $form['payment_details']['security_code'] = [
          '#type' => 'textfield',
          '#title' => $this->t('CVV'),
          '#attributes' => ['autocomplete' => 'off'],
          '#maxlength' => 4,
          '#size' => 4,
          '#placeholder' => $this->t('Three digit number on the back of your card'),
          '#description' => $this->t('Three digit number on the back of your card'),
          '#required' => FALSE, // From the perspective of FAPI this field is not required. We only use it in JavaScript.,
          '#attributes' => [
            'class' => ['mangopay-dpi-card-security-code'],
          ],
          /**
           * Mark as sensitive - Can only be transferred to MANGOPAY directly
           * @see commerce_mangopay_dpi_preprocess_input
           * @see commerce_mangopay_dpi_preprocess_form_element
           */
          '#commerce_mangopay_dpi_sensitive' => TRUE
        ];

        $form['payment_details']['card_type'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-card-type'],
          ]
        ];

        $form['payment_details']['card_alias'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-card-alias'],
          ]
        ];

        $form['payment_details']['card_id'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-card-id'],
          ]
        ];
        break;

      case 'commerce_mangopay_dpi_apple_pay':
        if (!$order) {
          // TODO: throw the right exception
          return;
        }

        $form['#attached']['drupalSettings']['commerceMangopay']['amount'] = $order->getTotalPrice()->getNumber();
        $form['#attached']['drupalSettings']['commerceMangopay']['transactionLabel'] = \Drupal::config('system.site')->get('name');


        $form['#attached']['library'][] = 'commerce_mangopay_dpi/apple_pay';

        $form['payment_details']['#attributes']['class'][] = 'apple-pay-form';
        $form['payment_details']['#attributes']['class'][] = 'mangopay-dpi-apple-pay-form';


        $form['payment_details']['transaction_id'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-transaction-id'],
          ]
        ];

        $form['payment_details']['network'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-network'],
          ]
        ];

        $form['payment_details']['token_data'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-token-data'],
          ]
        ];

        break;

      case 'commerce_mangopay_dpi_google_pay':
        if (!$order) {
          // TODO: throw the right exception
          return;
        }

        $form['payment_details']['#attributes']['class'][] = 'google-pay-form';
        $form['payment_details']['#attributes']['class'][] = 'mangopay-dpi-google-pay-form';
        $form['#attached']['library'][] = 'commerce_mangopay_dpi/google_pay';
        $form['#attached']['drupalSettings']['commerceMangopay']['transactionLabel'] = \Drupal::config('system.site')->get('name');
        $form['#attached']['drupalSettings']['commerceMangopay']['orderId'] = $order->id();

        $form['payment_details']['transaction_id'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-transaction-id'],
          ]
        ];

        $form['payment_details']['network'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-network'],
          ]
        ];

        $form['payment_details']['token_data'] = [
          '#type' => 'hidden',
          '#attributes' => [
            'class' => ['mangopay-dpi-token-data'],
          ]
        ];

        break;
    }

    $form['payment_details']['currency_code'] = [
      '#type' => 'hidden',
      '#attributes' => [
        'class' => ['mangopay-dpi-currency-code'],
      ],
      '#default_value' => $currency_code
    ];

    $form['payment_details']['user_id'] = [
      '#type' => 'hidden',
      '#attributes' => [
        'class' => ['mangopay-dpi-user-id'],
      ]
    ];

    $form['payment_details']['wallet_id'] = [
      '#type' => 'hidden',
      '#attributes' => [
        'class' => ['mangopay-dpi-wallet-id'],
      ]
    ];

    $form['kyc_details'] = [
      '#parents' => array_merge($form['#parents'], ['kyc_details']),
      '#type' => 'container',
      '#payment_method_type' => $payment_method->bundle()
    ];

    $form['kyc_details']['cardholder_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cardholder name'),
      '#attributes' => ['autocomplete' => 'off'],
      '#required' => TRUE,
      '#attributes' => [
        'class' => ['mangopay-dpi-cardholder-name'],
      ]
    ];

    // Allow other modules to alter both list of countries and default country.
    $countries = CountryManager::getStandardList();
    $default_country = key($countries);
    \Drupal::moduleHandler()->alter('commerce_mangopay_dpi_countries', $countries);
    \Drupal::moduleHandler()->alter('commerce_mangopay_dpi_default_country', $default_country);

    $form['kyc_details']['cardholder_country'] = [
      '#type' => 'select',
      '#title' => $this->t('Country of residence'),
      '#required' => TRUE,
      '#options' => $countries,
      '#default_value' => $default_country,
      '#attributes' => [
        'class' => ['mangopay-dpi-cardholder-country'],
      ]
    ];

    // Move the billing information below the payment details.
    if (isset($form['billing_information'])) {
      $form['billing_information']['#weight'] = 10;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    $values = $form_state->getValue($form['#parents']);
    $payment_details = $values['payment_details'];

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $this->entity;

    // TODO: Handle Apple Pay
    // Set relevant details on payment method object.
    switch ($payment_method->bundle()) {
      case 'commerce_mangopay_dpi_credit_card':
        // Validate if we have required data to correctly register the card.
        if (
          empty($payment_details['card_type']) || empty($payment_details['card_alias'])
          || empty($payment_details['card_id']) || empty($payment_details['user_id'])
          || empty($payment_details['wallet_id'])
        ) {
          $form_state->setError($form,  $this->t('Card. The credit card form has not been processed correctly. Please make sure your browser is up to date and supports JavaScript.' . print_r($payment_details, TRUE)));
        }
        break;
      case 'commerce_mangopay_dpi_apple_pay':
        if (
          empty($payment_details['transaction_id']) || empty($payment_details['network'])
          || empty($payment_details['token_data']) || empty($payment_details['user_id'])
          || empty($payment_details['wallet_id'])
        ) {
          $form_state->setError($form,  $this->t('Apple. The credit card form has not been processed correctly. Please make sure your browser is up to date and supports JavaScript.') . print_r($payment_details, TRUE));
        }
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $this->plugin;

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $this->entity;

    $values = $form_state->getValue($form['#parents']);

    try {
      $payment_gateway_plugin->createPaymentMethod($payment_method, (array)$values['payment_details'] + (array)$values['kyc_details']);
    }
    catch (\Exception $e) {
      \Drupal::logger('commerce_mangopay_dpi')->error($e->getMessage());
      throw new DeclineException('We encountered an error processing your payment method. Please verify your details and try again.');
    }
  }
}
