<?php

namespace Drupal\commerce_mautic_connect\EventSubscriber;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\commerce_cart\Event\CartEntityAddEvent;
use Drupal\commerce_cart\Event\CartEvents;
use Drupal\commerce_cart\Event\CartOrderItemRemoveEvent;
use Drupal\commerce_cart\Event\CartOrderItemUpdateEvent;
use Drupal\commerce_mautic_connect\Service\CartRecoveryService;
use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Event subscriber for cart updates to sync with Mautic.
 */
class CartUpdateSubscriber implements EventSubscriberInterface {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The Mautic API wrapper service.
   *
   * @var \Drupal\advanced_mautic_integration\MauticApiWrapperInterface
   */
  protected $mauticApi;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The cart recovery service.
   *
   * @var \Drupal\commerce_mautic_connect\Service\CartRecoveryService
   */
  protected $recoveryService;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a new CartUpdateSubscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\advanced_mautic_integration\MauticApiWrapperInterface $mautic_api
   *   The Mautic API wrapper service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\commerce_mautic_connect\Service\CartRecoveryService $recovery_service
   *   The cart recovery service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    MauticApiWrapperInterface $mautic_api,
    RequestStack $request_stack,
    LoggerChannelInterface $logger,
    RendererInterface $renderer,
    CartRecoveryService $recovery_service,
    LanguageManagerInterface $language_manager
  ) {
    $this->configFactory = $config_factory;
    $this->mauticApi = $mautic_api;
    $this->requestStack = $request_stack;
    $this->logger = $logger;
    $this->renderer = $renderer;
    $this->recoveryService = $recovery_service;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events = [
      CartEvents::CART_ENTITY_ADD => 'onCartUpdate',
      CartEvents::CART_ORDER_ITEM_UPDATE => 'onCartUpdate',
      CartEvents::CART_ORDER_ITEM_REMOVE => 'onCartUpdate',
      OrderEvents::ORDER_UPDATE => 'onOrderUpdate',
      'commerce_order.place.post_transition' => 'onOrderPlaced',
    ];
    return $events;
  }

  /**
   * Handles cart update events.
   *
   * @param \Drupal\commerce_cart\Event\CartEntityAddEvent|\Drupal\commerce_cart\Event\CartOrderItemUpdateEvent|\Drupal\commerce_cart\Event\CartOrderItemRemoveEvent $event
   *   The cart event.
   */
  public function onCartUpdate($event) {
    // Get the cart (order) from the event.
    if ($event instanceof CartEntityAddEvent) {
      $cart = $event->getCart();
    }
    elseif ($event instanceof CartOrderItemUpdateEvent) {
      $cart = $event->getCart();
    }
    elseif ($event instanceof CartOrderItemRemoveEvent) {
      $cart = $event->getCart();
    }
    else {
      return;
    }

    // Identity Check 1: Check if cart has an email.
    $email = $cart->getEmail();

    // Identity Check 2: Check for Mautic tracking cookie.
    $request = $this->requestStack->getCurrentRequest();
    $mtc_id = $request ? $request->cookies->get('mtc_id') : NULL;

    // Abort if neither identifier is available.
    if (empty($email) && empty($mtc_id)) {
      $this->logger->debug('Cart update: No email or mtc_id found. Skipping Mautic sync.');
      return;
    }

    // Get the Mautic field alias from configuration.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    $field_alias = $config->get('field_cart_html') ?: 'commerce_cart_items_html';

    // Check if cart is empty.
    if (empty($cart->getItems())) {
      $this->logger->debug('Cart is now empty. Clearing abandoned cart field in Mautic.');

      // Clear the cart field in Mautic.
      $this->clearCartField($email, $mtc_id, $field_alias);
      return;
    }

    // Generate HTML cart contents.
    $html = $this->generateCartHtml($cart);

    // Generate recovery URL.
    $token = $this->recoveryService->generateToken($cart);
    $recovery_url = Url::fromRoute('commerce_mautic_connect.restore', [
      'order' => $cart->id(),
      'token' => $token,
    ], ['absolute' => TRUE])->toString();

    // Get the cart language using Commerce's approach:
    // Use customer's preferred language if available, otherwise current interface language.
    // This matches how Commerce sends order receipt emails.
    $order_language = $this->languageManager->getCurrentLanguage()->getId();
    $customer = $cart->getCustomer();
    if ($customer && !$customer->isAnonymous()) {
      $preferred_langcode = $customer->get('preferred_langcode')->value;
      if (!empty($preferred_langcode)) {
        $order_language = $preferred_langcode;
      }
    }

    // Get field aliases.
    $field_cart_updated  = $config->get('field_cart_updated') ?: 'commerce_cart_updated';
    $field_cart_language = $config->get('field_cart_language') ?: 'commerce_cart_language';

    try {
      // Get the Mautic contacts API.
      $api = $this->mauticApi->getApi('contacts');

      if (!$api) {
        $this->logger->error('Failed to retrieve Mautic contacts API.');
        return;
      }

      // Prepare cart data with cart's last changed timestamp and language.
      $data = [
        $field_alias => $html,
        'cart_recovery_url' => $recovery_url,
        $field_cart_updated => date('c', $cart->getChangedTime()), // ISO 8601 format (e.g., 2024-11-21T17:30:00+00:00)
        $field_cart_language => $order_language, // Order language code (e.g., pt, en, es)
      ];

      // Scenario A: Have email - use search-first logic.
      if (!empty($email)) {
        // Search for existing contact by email to prevent 422 errors.
        $search = $api->getList('email:' . $email, 0, 1);

        if (!empty($search['contacts'])) {
          // Email found: Update existing contact.
          $contact = reset($search['contacts']);
          $target_id = $contact['id'];
          $result = $api->edit($target_id, $data);

          if (isset($result['contact'])) {
            $this->logger->info('Synced cart to Mautic contact @id for email: @email', [
              '@id' => $target_id,
              '@email' => $email,
            ]);
          }
        }
        else {
          // Email not found: Create new contact or update via create API.
          $data['email'] = $email;
          $result = $api->create($data);

          if (isset($result['contact'])) {
            $this->logger->info('Synced cart to Mautic for email: @email', ['@email' => $email]);
          }
          else {
            $this->logger->warning('Unexpected response from Mautic API for email: @email', ['@email' => $email]);
          }
        }
      }
      // Scenario B: Have mtc_id only - update anonymous contact.
      elseif (!empty($mtc_id)) {
        $result = $api->edit($mtc_id, $data);

        if (isset($result['contact'])) {
          $this->logger->info('Synced cart to Mautic for mtc_id: @mtc_id', ['@mtc_id' => $mtc_id]);
        }
        else {
          $this->logger->warning('Unexpected response from Mautic API for mtc_id: @mtc_id', ['@mtc_id' => $mtc_id]);
        }
      }
    }
    catch (\Exception $e) {
      // Catch all exceptions to prevent checkout errors.
      $this->logger->error('Error syncing cart to Mautic: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Handles order update events to sync email and cart when email is added.
   *
   * This event fires when an order is updated, such as when a customer
   * provides their email address during checkout.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The order event.
   */
  public function onOrderUpdate(OrderEvent $event) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getOrder();

    // Only process cart orders (draft state).
    if ($order->getState()->getId() !== 'draft') {
      return;
    }

    // Check if order has an email.
    $email = $order->getEmail();
    if (empty($email)) {
      return;
    }

    // Check if the email was just added (original order had no email).
    $original = $order->original ?? NULL;
    if ($original && $original->getEmail() === $email) {
      // Email hasn't changed, skip to avoid redundant API calls.
      return;
    }

    $this->logger->debug('Order update: Email detected (@email). Syncing to Mautic.', ['@email' => $email]);

    // Generate HTML cart contents.
    $html = $this->generateCartHtml($order);

    // Get the Mautic field alias from configuration.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    $field_alias = $config->get('field_cart_html') ?: 'commerce_cart_items_html';

    // Check for Mautic tracking cookie to identify existing contact.
    $request = $this->requestStack->getCurrentRequest();
    $mtc_id = $request ? $request->cookies->get('mtc_id') : NULL;

    try {
      // Get the Mautic contacts API.
      $api = $this->mauticApi->getApi('contacts');

      if (!$api) {
        $this->logger->error('Failed to retrieve Mautic contacts API.');
        return;
      }

      // Prepare the cart data to update.
      $data = [
        $field_alias => $html,
      ];

      $target_id = NULL;

      // SEARCH FIRST: Check if the email already exists in Mautic.
      // This prevents 422 errors when trying to assign an email to a contact
      // when that email already belongs to a different contact.
      $search = $api->getList('email:' . $email, 0, 1);

      if (!empty($search['contacts'])) {
        // Email found: Use the existing contact ID.
        $contact = reset($search['contacts']);
        $target_id = $contact['id'];

        $this->logger->debug('Found existing Mautic contact @id for email: @email', [
          '@id' => $target_id,
          '@email' => $email,
        ]);

        // Update the existing contact with cart data.
        // We don't need to include email since it already has it.
        $result = $api->edit($target_id, $data);

        if (isset($result['contact'])) {
          $this->logger->info('Updated existing Mautic contact @id with cart data for email: @email', [
            '@id' => $target_id,
            '@email' => $email,
          ]);
        }
      }
      elseif (!empty($mtc_id)) {
        // Email not found, but we have a cookie ID.
        // Update the anonymous contact with the email.
        $data['email'] = $email;
        $result = $api->edit($mtc_id, $data);

        if (isset($result['contact'])) {
          $this->logger->info('Updated anonymous Mautic contact @mtc_id with email: @email', [
            '@mtc_id' => $mtc_id,
            '@email' => $email,
          ]);
        }
        else {
          $this->logger->warning('Unexpected response from Mautic API when updating mtc_id: @mtc_id', ['@mtc_id' => $mtc_id]);
        }
      }
      else {
        // No existing contact and no cookie ID: Create new contact.
        $data['email'] = $email;
        $result = $api->create($data);

        if (isset($result['contact'])) {
          $this->logger->info('Created new Mautic contact for email: @email', ['@email' => $email]);
        }
        else {
          $this->logger->warning('Unexpected response from Mautic API for email: @email', ['@email' => $email]);
        }
      }
    }
    catch (\Exception $e) {
      // Catch all exceptions to prevent checkout errors.
      $this->logger->error('Error syncing order email to Mautic: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Handles order placed event to clear the cart field in Mautic.
   *
   * Fires when an order completes checkout (transition to 'placed' state),
   * regardless of payment status.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The workflow transition event.
   */
  public function onOrderPlaced(WorkflowTransitionEvent $event) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getEntity();
    $email = $order->getEmail();

    if (empty($email)) {
      return;
    }

    // Get the Mautic field alias from configuration.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    $field_alias = $config->get('field_cart_html') ?: 'commerce_cart_items_html';

    try {
      // Get the Mautic contacts API.
      $api = $this->mauticApi->getApi('contacts');

      if (!$api) {
        $this->logger->error('Failed to retrieve Mautic contacts API.');
        return;
      }

      // Clear the cart field to remove from abandoned cart segment.
      $data = [
        $field_alias => '',
        // Critical: By default Mautic API ignores empty values to protect data.
        // We must explicitly tell it to overwrite with blank.
        'overwriteWithBlank' => TRUE,
      ];

      // SEARCH FIRST: Find the contact by email to ensure we update the correct record.
      $search = $api->getList('email:' . $email, 0, 1);

      if (!empty($search['contacts'])) {
        // Contact found: Update specific ID.
        $contact = reset($search['contacts']);
        $target_id = $contact['id'];

        $result = $api->edit($target_id, $data);

        if (isset($result['contact'])) {
          $this->logger->info('Cleared abandoned cart field for Mautic contact @id (email: @email)', [
            '@id' => $target_id,
            '@email' => $email,
          ]);
        }
        else {
          $this->logger->warning('Failed to clear cart field for contact @id. Response: @response', [
            '@id' => $target_id,
            '@response' => json_encode($result),
          ]);
        }
      }
      else {
        // Fallback: Try to update by email if search failed.
        $data['email'] = $email;
        $result = $api->create($data);

        if (isset($result['contact'])) {
          $this->logger->info('Cleared abandoned cart field for email: @email (via create/update)', ['@email' => $email]);
        }
        else {
          $this->logger->warning('Failed to clear cart field for email @email. Response: @response', [
            '@email' => $email,
            '@response' => json_encode($result),
          ]);
        }
      }
    }
    catch (\Exception $e) {
      // Catch all exceptions to prevent checkout errors.
      $this->logger->error('Error clearing cart field in Mautic: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Clears the abandoned cart field in Mautic.
   *
   * @param string|null $email
   *   The customer email address, if available.
   * @param string|null $mtc_id
   *   The Mautic tracking cookie ID, if available.
   * @param string $field_alias
   *   The Mautic field alias for cart HTML.
   */
  protected function clearCartField($email, $mtc_id, $field_alias): void {
    try {
      $api = $this->mauticApi->getApi('contacts');
      if (!$api) {
        $this->logger->error('Failed to retrieve Mautic contacts API.');
        return;
      }

      // Get field aliases.
      $config = $this->configFactory->get('commerce_mautic_connect.settings');
      $field_cart_updated  = $config->get('field_cart_updated') ?: 'commerce_cart_updated';
      $field_cart_language = $config->get('field_cart_language') ?: 'commerce_cart_language';

      // Prepare clear data.
      $data = [
        $field_alias => '',
        'cart_recovery_url' => '',
        $field_cart_updated => '',
        $field_cart_language => '',
        'overwriteWithBlank' => TRUE,
      ];

      // Priority 1: Clear by email.
      if (!empty($email)) {
        $search = $api->getList('email:' . $email, 0, 1);

        if (!empty($search['contacts'])) {
          $contact = reset($search['contacts']);
          $contact_id = $contact['id'];
          $result = $api->edit($contact_id, $data);

          if (isset($result['contact'])) {
            $this->logger->info('Cleared abandoned cart field for Mautic contact @id (email: @email)', [
              '@id' => $contact_id,
              '@email' => $email,
            ]);
          }
        }
      }
      // Priority 2: Clear by mtc_id.
      elseif (!empty($mtc_id)) {
        $result = $api->edit($mtc_id, $data);

        if (isset($result['contact'])) {
          $this->logger->info('Cleared abandoned cart field for Mautic contact (mtc_id: @mtc_id)', [
            '@mtc_id' => $mtc_id,
          ]);
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error clearing cart field in Mautic: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Generates HTML representation of cart contents.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order entity.
   *
   * @return string
   *   HTML string representing cart items.
   */
  protected function generateCartHtml($cart) {
    $items = $cart->getItems();

    if (empty($items)) {
      return '';
    }

    // Get the cart language using Commerce's approach:
    // Use customer's preferred language if available, otherwise current interface language.
    // This matches how Commerce sends order receipt emails.
    $order_language = $this->languageManager->getCurrentLanguage()->getId();
    $customer = $cart->getCustomer();
    if ($customer && !$customer->isAnonymous()) {
      $preferred_langcode = $customer->get('preferred_langcode')->value;
      if (!empty($preferred_langcode)) {
        $order_language = $preferred_langcode;
      }
    }
    $current_language = $this->languageManager->getCurrentLanguage()->getId();

    // Switch to order's language if different.
    $original_language = NULL;
    if ($order_language !== $current_language) {
      $original_language = $this->languageManager->getCurrentLanguage();
      $this->languageManager->setConfigOverrideLanguage($this->languageManager->getLanguage($order_language));
    }

    // Prepare items array for template.
    $template_items = [];
    foreach ($items as $item) {
      $purchased_entity = $item->getPurchasedEntity();
      $product_url = NULL;

      // Try to get product URL in the correct language.
      if ($purchased_entity && $purchased_entity->hasLinkTemplate('canonical')) {
        try {
          // Get translated entity if available.
          if ($purchased_entity->hasTranslation($order_language)) {
            $purchased_entity = $purchased_entity->getTranslation($order_language);
          }

          $product_url = $purchased_entity->toUrl('canonical', [
            'absolute' => TRUE,
            'language' => $this->languageManager->getLanguage($order_language),
          ])->toString();
        }
        catch (\Exception $e) {
          // URL generation failed, continue without URL.
        }
      }

      $unit_price = $item->getUnitPrice();
      $total_price = $item->getTotalPrice();

      $template_items[] = [
        'title' => $item->getTitle(),
        'quantity' => $item->getQuantity(),
        'unit_price' => $unit_price ? $unit_price->__toString() : '',
        'total_price' => $total_price ? $total_price->__toString() : '',
        'product_url' => $product_url,
        'entity' => $purchased_entity, // Pass full entity for advanced theming (e.g. images).
      ];
    }

    // Get cart URL in the correct language.
    $cart_url = NULL;
    try {
      $cart_url = Url::fromRoute('commerce_cart.page', [], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      // Cart URL generation failed.
    }

    // Generate Recovery URL (Magic Link) in the correct language.
    $recovery_url = NULL;
    try {
      $token = $this->recoveryService->generateToken($cart);
      $recovery_url = Url::fromRoute('commerce_mautic_connect.restore', [
        'order' => $cart->id(),
        'token' => $token,
      ], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to generate recovery URL: @message', ['@message' => $e->getMessage()]);
    }

    // Get cart total.
    $cart_total = $cart->getTotalPrice();
    $cart_total_formatted = $cart_total ? $cart_total->__toString() : '';

    // Get base URL for email compatibility (absolute URLs for images, links).
    $base_url_value = \Drupal::request()->getSchemeAndHttpHost();

    // Render template in the order's language.
    $build = [
      '#theme' => 'commerce_mautic_connect_cart_email',
      '#items' => $template_items,
      '#cart_total' => $cart_total_formatted,
      '#cart_url' => $recovery_url ?: $cart_url, // Prefer recovery URL.
      '#base_url' => $base_url_value,
    ];

    $html = $this->renderer->renderPlain($build);

    // Restore original language.
    if ($original_language) {
      $this->languageManager->setConfigOverrideLanguage($original_language);
    }

    return $html;
  }

}

