<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Form;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\commerce_mautic_connect\MauticFeaturePluginManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Commerce Mautic Connect.
 *
 * This form serves as a hub for MauticFeature plugins, which provide
 * individual settings tabs. New features can be added by creating plugins
 * that implement MauticFeaturePluginInterface.
 *
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginManager
 */
class MauticConnectForm extends ConfigFormBase {

  /**
   * The Mautic API wrapper service.
   *
   * @var \Drupal\advanced_mautic_integration\MauticApiWrapperInterface
   */
  protected MauticApiWrapperInterface $mauticApi;

  /**
   * The MauticFeature plugin manager.
   *
   * @var \Drupal\commerce_mautic_connect\MauticFeaturePluginManager
   */
  protected MauticFeaturePluginManager $mauticFeatureManager;

  /**
   * Cached plugin instances.
   *
   * @var \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface[]
   */
  protected array $pluginInstances = [];

  /**
   * Constructs a new MauticConnectForm.
   *
   * @param \Drupal\advanced_mautic_integration\MauticApiWrapperInterface $mautic_api
   *   The Mautic API wrapper service.
   * @param \Drupal\commerce_mautic_connect\MauticFeaturePluginManager $mautic_feature_manager
   *   The MauticFeature plugin manager.
   */
  public function __construct(
    MauticApiWrapperInterface $mautic_api,
    MauticFeaturePluginManager $mautic_feature_manager,
  ) {
    $this->mauticApi            = $mautic_api;
    $this->mauticFeatureManager = $mautic_feature_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('advanced_mautic_integration.api'),
      $container->get('plugin.manager.commerce_mautic_connect.mautic_feature'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'commerce_mautic_connect_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['commerce_mautic_connect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Display Mautic connection status at the top.
    $connection_status = $this->checkMauticConnection();
    if ($connection_status['connected']) {
      $version_info = '';
      if (!empty($connection_status['version'])) {
        $version_info = ' | <strong>' . $this->t('Version:') . '</strong> ' . $connection_status['version'];
      }

      $form['mautic_status'] = [
        '#type'   => 'markup',
        '#markup' => '<div class="messages messages--status">' .
          $this->t('✓ <strong>Mautic Connection:</strong> Successful') . $version_info .
          '</div>',
      ];
    }
    else {
      $error_message = $connection_status['error'] ?? $this->t('Unknown connection error');

      // Generate link to Advanced Mautic Integration settings.
      $mautic_settings_link = Link::createFromRoute(
        $this->t('Configure API credentials'),
        'advanced_mautic_integration.admin_settings_form'
      )->toString();

      $form['mautic_status'] = [
        '#type'   => 'markup',
        '#markup' => '<div class="messages messages--error">' .
          $this->t('✗ <strong>Mautic Connection:</strong> Failed') .
          '<br><br><strong>' . $this->t('Error:') . '</strong> ' . $error_message .
          '<br><br><strong>' . $this->t('Troubleshooting:') . '</strong>' .
          '<br>• ' . $this->t('Check if your Mautic instance is online and accessible') .
          '<br>• ' . $this->t('Verify that API access is enabled in Mautic: <em>Settings → Configuration → API Settings → Enable API</em>') .
          '<br>• ' . $this->t('Verify your Mautic API user and password: ') . $mautic_settings_link .
          '<br><br><small>' . $this->t('You can still configure settings here, but field creation and syncing will not work until the connection is restored.') . '</small>' .
          '</div>',
      ];
    }

    // Vertical Tabs Container.
    $form['tabs'] = [
      '#type'        => 'vertical_tabs',
      '#default_tab' => 'edit-abandoned-cart',
    ];

    // Load all MauticFeature plugins sorted by weight.
    foreach ($this->mauticFeatureManager->getSortedDefinitions() as $plugin_id => $definition) {
      /** @var \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface $plugin */
      $plugin                          = $this->mauticFeatureManager->createInstance($plugin_id);
      $this->pluginInstances[$plugin_id] = $plugin;
      $form[$plugin_id]                = $plugin->buildForm($form, $form_state, $this->configFactory());
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Delegate validation to each plugin.
    foreach ($this->getPluginInstances() as $plugin) {
      $plugin->validateForm($form, $form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Delegate submission to each plugin.
    foreach ($this->getPluginInstances() as $plugin) {
      $plugin->submitForm($form, $form_state, $this->configFactory());
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets cached plugin instances.
   *
   * @return \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface[]
   *   Array of plugin instances.
   */
  protected function getPluginInstances(): array {
    // If instances aren't cached (e.g., during validation/submission),
    // reload them.
    if (empty($this->pluginInstances)) {
      foreach ($this->mauticFeatureManager->getSortedDefinitions() as $plugin_id => $definition) {
        $this->pluginInstances[$plugin_id] = $this->mauticFeatureManager->createInstance($plugin_id);
      }
    }

    return $this->pluginInstances;
  }

  /**
   * Checks the Mautic API connection.
   *
   * Handles all possible failure scenarios gracefully including:
   * - Mautic API server being down/unreachable
   * - Invalid credentials
   * - Network timeouts
   * - Malformed API responses
   *
   * @return array
   *   Connection status array with keys:
   *   - connected: (bool) Whether connection succeeded
   *   - version: (string|null) Mautic version if available
   *   - error: (string) Error message if connection failed
   */
  protected function checkMauticConnection(): array {
    try {
      // Attempt to initialize the API client.
      $api = $this->mauticApi->getApi('contacts');
      if (!$api) {
        return [
          'connected' => FALSE,
          'error'     => $this->t('Failed to initialize Mautic API client. Check your Mautic API user and password.'),
        ];
      }

      // Test the connection with a minimal API call.
      $result = $api->getList('', 0, 1);

      // Check if we got a valid response.
      if (!is_array($result)) {
        return [
          'connected' => FALSE,
          'error'     => $this->t('Mautic API returned an invalid response. The server may be down or misconfigured.'),
        ];
      }

      // Check if the API returned errors.
      if (isset($result['errors'])) {
        // Handle nested error arrays properly.
        $error_details = $this->flattenErrorArray($result['errors']);
        return [
          'connected' => FALSE,
          'error'     => $this->t('Mautic API error: @error', ['@error' => $error_details]),
        ];
      }

      // Check if the response contains the expected 'contacts' key.
      if (!isset($result['contacts'])) {
        return [
          'connected' => FALSE,
          'error'     => $this->t('Mautic API returned unexpected response format. The server may be misconfigured or unreachable.'),
        ];
      }

      // Connection successful - try to get version.
      $version = NULL;
      try {
        $version = $api->getMauticVersion();
      }
      catch (\Exception $e) {
        // Version retrieval failed, but connection is still valid.
      }

      return [
        'connected' => TRUE,
        'version'   => $version,
      ];
    }
    catch (\Exception $e) {
      // Catch all exceptions: network errors, timeouts, auth failures, etc.
      $error_message = $e->getMessage();

      // Provide user-friendly error messages for common scenarios.
      if (stripos($error_message, 'could not resolve host') !== FALSE ||
          stripos($error_message, 'connection refused') !== FALSE ||
          stripos($error_message, 'connection timed out') !== FALSE) {
        $error_message = $this->t('Mautic server is unreachable. Please check that the server is online and the URL is correct.');
      }
      elseif (stripos($error_message, '401') !== FALSE || stripos($error_message, 'unauthorized') !== FALSE) {
        $error_message = $this->t('Authentication failed. Please check your Mautic API user and password.');
      }
      elseif (stripos($error_message, '403') !== FALSE || stripos($error_message, 'forbidden') !== FALSE) {
        $error_message = $this->t('Access denied. Please check your Mautic user permissions.');
      }

      return [
        'connected' => FALSE,
        'error'     => $error_message,
      ];
    }
  }

  /**
   * Flattens a nested error array into a readable string.
   *
   * @param mixed $errors
   *   The error data (can be string, array, or nested array).
   *
   * @return string
   *   A readable error message.
   */
  protected function flattenErrorArray(mixed $errors): string {
    if (is_string($errors)) {
      return $errors;
    }

    if (!is_array($errors)) {
      return (string) $errors;
    }

    $messages = [];
    foreach ($errors as $key => $value) {
      if (is_array($value)) {
        $messages[] = $this->flattenErrorArray($value);
      }
      else {
        $messages[] = (string) $value;
      }
    }

    return implode('; ', array_filter($messages));
  }

}

