<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect;

use Drupal\commerce_mautic_connect\Attribute\MauticFeature;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Manages discovery and instantiation of MauticFeature plugins.
 *
 * MauticFeature plugins provide configuration tabs for the Commerce Mautic
 * Connect settings form. This manager discovers plugins using PHP 8 attributes
 * in the Plugin/MauticFeature namespace.
 *
 * Example plugin discovery:
 * @code
 * // In any module's src/Plugin/MauticFeature directory:
 * #[MauticFeature(
 *   id: 'my_feature',
 *   label: new TranslatableMarkup('My Feature'),
 *   weight: 20,
 * )]
 * class MyFeature extends MauticFeaturePluginBase { ... }
 * @endcode
 *
 * @see \Drupal\commerce_mautic_connect\Attribute\MauticFeature
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginBase
 */
class MauticFeaturePluginManager extends DefaultPluginManager {

  /**
   * Constructs a MauticFeaturePluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      'Plugin/MauticFeature',
      $namespaces,
      $module_handler,
      MauticFeaturePluginInterface::class,
      MauticFeature::class,
    );

    $this->alterInfo('mautic_feature_info');
    $this->setCacheBackend($cache_backend, 'mautic_feature_plugins');
  }

  /**
   * Gets plugin definitions sorted by weight.
   *
   * @return array
   *   An array of plugin definitions sorted by weight (ascending).
   */
  public function getSortedDefinitions(): array {
    $definitions = $this->getDefinitions();

    uasort($definitions, function ($a, $b) {
      $weight_a = $a['weight'] ?? 0;
      $weight_b = $b['weight'] ?? 0;
      return $weight_a <=> $weight_b;
    });

    return $definitions;
  }

}

