<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Plugin\MauticFeature;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\commerce_mautic_connect\Attribute\MauticFeature;
use Drupal\commerce_mautic_connect\MauticFeaturePluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\state_machine\WorkflowManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Customer Metrics feature settings.
 *
 * Enables synchronization of customer metrics (RFM - Recency, Frequency,
 * Monetary) to Mautic for advanced customer segmentation and marketing
 * automation.
 */
#[MauticFeature(
  id: 'customer_metrics',
  label: new TranslatableMarkup('Customer Metrics'),
  weight: 10,
)]
class CustomerMetrics extends MauticFeaturePluginBase {

  /**
   * The workflow manager.
   *
   * @var \Drupal\state_machine\WorkflowManagerInterface
   */
  protected WorkflowManagerInterface $workflowManager;

  /**
   * Constructs a CustomerMetrics instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\advanced_mautic_integration\MauticApiWrapperInterface $mautic_api
   *   The Mautic API wrapper service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\state_machine\WorkflowManagerInterface $workflow_manager
   *   The workflow manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    MauticApiWrapperInterface $mautic_api,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    WorkflowManagerInterface $workflow_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $mautic_api, $logger_factory, $entity_type_manager);
    $this->workflowManager = $workflow_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('advanced_mautic_integration.api'),
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.workflow'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): array {
    $config = $config_factory->get('commerce_mautic_connect.settings');

    $element = [
      '#type'  => 'details',
      '#title' => $this->t('Customer Metrics'),
      '#group' => 'tabs',
    ];

    $element['enable_customer_metrics'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Customer Metrics Sync'),
      '#description'   => $this->t('If enabled, customer Lifetime Value and order stats will be synced to Mautic on order completion.'),
      '#default_value' => $config->get('enable_customer_metrics') ?? FALSE,
    ];

    $currency_options            = $this->getCurrencyOptions();
    $element['base_currency'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Base Currency'),
      '#description'   => $this->t('The currency used for calculating and displaying customer metrics (CLV, AOV). All order totals will be converted to this currency.'),
      '#options'       => $currency_options,
      '#default_value' => $config->get('base_currency') ?: $this->getDefaultCurrency(),
      '#required'      => TRUE,
    ];

    // Show currency conversion note if multiple currencies exist.
    if (count($currency_options) > 1) {
      $module_handler = \Drupal::moduleHandler();
      if ($module_handler->moduleExists('commerce_exchanger')) {
        // Module is installed - show success message.
        $element['currency_conversion_note'] = [
          '#type'   => 'markup',
          '#markup' => '<div class="messages messages--status">' .
            $this->t('<strong>Currency Conversion:</strong> Using the <a href="@url" target="_blank">Commerce Exchanger</a> module for automatic currency conversion. All CLV and AOV values will be converted to the selected base currency.', [
              '@url' => 'https://www.drupal.org/project/commerce_exchanger',
            ]) .
            '</div>',
        ];
      }
      else {
        // Module is not installed - show warning.
        $element['currency_conversion_note'] = [
          '#type'   => 'markup',
          '#markup' => '<div class="messages messages--warning">' .
            $this->t('<strong>Note:</strong> For automatic currency conversion, install the <a href="@url" target="_blank">Commerce Exchanger</a> module. Without it, orders in different currencies will be calculated at face value (e.g., $100 USD = €100 EUR).', [
              '@url' => 'https://www.drupal.org/project/commerce_exchanger',
            ]) .
            '</div>',
        ];
      }
    }

    $element['field_total_spent'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Total Spent Field Alias'),
      '#description'   => $this->t('Mautic custom field for customer lifetime value (Monetary).'),
      '#default_value' => $config->get('field_total_spent') ?: 'commerce_total_spent',
    ];

    $element['field_total_orders'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Total Orders (Frequency) Field Alias'),
      '#description'   => $this->t('Mautic custom field for total number of orders (Frequency).'),
      '#default_value' => $config->get('field_total_orders') ?: 'commerce_total_orders',
    ];

    $element['field_last_order_date'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Last Order Date (Recency) Field Alias'),
      '#description'   => $this->t('Mautic custom field for the date of the most recent order (Recency).'),
      '#default_value' => $config->get('field_last_order_date') ?: 'commerce_last_order_date',
    ];

    $element['field_first_order_date'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('First Order Date Field Alias'),
      '#description'   => $this->t('Mautic custom field for the date of the first order ever placed.'),
      '#default_value' => $config->get('field_first_order_date') ?: 'commerce_first_order_date',
    ];

    $element['field_average_order_value'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Average Order Value (AOV) Field Alias'),
      '#description'   => $this->t('Mautic custom field for the average order value.'),
      '#default_value' => $config->get('field_average_order_value') ?: 'commerce_aov',
    ];

    $element['customer_metrics_order_states'] = [
      '#type'          => 'checkboxes',
      '#title'         => $this->t('Included Order States'),
      '#description'   => $this->t('Select which order states should count towards customer metrics calculations.'),
      '#options'       => $this->getOrderStateOptions(),
      '#default_value' => $config->get('customer_metrics_order_states') ?: ['completed', 'fulfillment'],
    ];

    $element['create_fields'] = [
      '#type'  => 'details',
      '#title' => $this->t('Operations'),
      '#open'  => TRUE,
    ];

    $element['create_fields']['create_customer_metrics_fields'] = [
      '#type'            => 'submit',
      '#value'           => $this->t('Create Customer Metrics Fields in Mautic'),
      '#submit'          => [[static::class, 'createCustomerMetricsFieldsSubmit']],
      '#button_type'     => 'secondary',
      '#plugin_instance' => $this,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Skip validation if user is clicking "Create Fields" button.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#submit']) &&
        in_array([static::class, 'createCustomerMetricsFieldsSubmit'], $triggering_element['#submit'])) {
      return;
    }

    // Validate Customer Metrics fields if being enabled.
    if ($form_state->getValue('enable_customer_metrics')) {
      $missing_fields = [];
      $rfm_fields     = [
        'field_total_spent'         => $form_state->getValue('field_total_spent'),
        'field_total_orders'        => $form_state->getValue('field_total_orders'),
        'field_last_order_date'     => $form_state->getValue('field_last_order_date'),
        'field_first_order_date'    => $form_state->getValue('field_first_order_date'),
        'field_average_order_value' => $form_state->getValue('field_average_order_value'),
      ];

      foreach ($rfm_fields as $field_key => $field_alias) {
        if (!$this->mauticFieldExists($field_alias)) {
          $missing_fields[] = $field_alias;
        }
      }

      if (!empty($missing_fields)) {
        $form_state->setErrorByName('enable_customer_metrics', $this->t('Cannot enable Customer Metrics sync: The following fields do not exist in Mautic: @fields. Please create the fields first using the "Create Customer Metrics Fields in Mautic" button below.', [
          '@fields' => implode(', ', $missing_fields),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void {
    $config_factory->getEditable('commerce_mautic_connect.settings')
      ->set('enable_customer_metrics', $form_state->getValue('enable_customer_metrics'))
      ->set('base_currency', $form_state->getValue('base_currency'))
      ->set('field_total_spent', $form_state->getValue('field_total_spent'))
      ->set('field_total_orders', $form_state->getValue('field_total_orders'))
      ->set('field_last_order_date', $form_state->getValue('field_last_order_date'))
      ->set('field_first_order_date', $form_state->getValue('field_first_order_date'))
      ->set('field_average_order_value', $form_state->getValue('field_average_order_value'))
      ->set('customer_metrics_order_states', array_filter($form_state->getValue('customer_metrics_order_states')))
      ->save();
  }

  /**
   * Submit handler for creating Customer Metrics fields in Mautic.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function createCustomerMetricsFieldsSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();

    /** @var \Drupal\commerce_mautic_connect\Plugin\MauticFeature\CustomerMetrics $plugin */
    $plugin = $triggering_element['#plugin_instance'];

    // Save current form values first, then create fields.
    $plugin->saveAndCreateFields($form_state);
  }

  /**
   * Saves config and creates Customer Metrics fields in Mautic.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state containing field aliases.
   */
  public function saveAndCreateFields(FormStateInterface $form_state): void {
    // Save current form values first.
    \Drupal::configFactory()->getEditable('commerce_mautic_connect.settings')
      ->set('field_total_spent', $form_state->getValue('field_total_spent'))
      ->set('field_total_orders', $form_state->getValue('field_total_orders'))
      ->set('field_last_order_date', $form_state->getValue('field_last_order_date'))
      ->set('field_first_order_date', $form_state->getValue('field_first_order_date'))
      ->set('field_average_order_value', $form_state->getValue('field_average_order_value'))
      ->save();

    // Create the fields.
    $this->createCustomerMetricsFields($form_state);
  }

  /**
   * Creates Customer Metrics fields in Mautic.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state containing field aliases.
   */
  public function createCustomerMetricsFields(FormStateInterface $form_state): void {
    $this->createMauticField($form_state->getValue('field_total_spent'), 'Total Spent (CLV)', 'number');
    $this->createMauticField($form_state->getValue('field_total_orders'), 'Total Orders (Frequency)', 'number');
    $this->createMauticField($form_state->getValue('field_last_order_date'), 'Last Order Date (Recency)', 'date');
    $this->createMauticField($form_state->getValue('field_first_order_date'), 'First Order Date', 'date');
    $this->createMauticField($form_state->getValue('field_average_order_value'), 'Average Order Value (AOV)', 'number');
  }

  /**
   * Gets order state options from the Commerce order workflow.
   *
   * @return array
   *   Array of order state options keyed by state ID.
   */
  protected function getOrderStateOptions(): array {
    $options = [];

    try {
      // Load all order types to get their workflows.
      $order_type_storage = $this->entityTypeManager->getStorage('commerce_order_type');
      $order_types        = $order_type_storage->loadMultiple();

      // Collect all unique states from all order types.
      $states = [];
      foreach ($order_types as $order_type) {
        /** @var \Drupal\commerce_order\Entity\OrderTypeInterface $order_type */
        $workflow_id = $order_type->get('workflow');
        if ($workflow_id) {
          // Load the workflow using the workflow manager.
          $workflow = $this->workflowManager->createInstance($workflow_id);
          if ($workflow) {
            foreach ($workflow->getStates() as $state_id => $state) {
              if (!isset($states[$state_id])) {
                $states[$state_id] = $state->getLabel();
              }
            }
          }
        }
      }

      // Convert to options array.
      foreach ($states as $state_id => $label) {
        $options[$state_id] = $label;
      }

      // If no states found, provide fallback.
      if (empty($options)) {
        $options = [
          'draft'       => $this->t('Draft'),
          'completed'   => $this->t('Completed'),
          'fulfillment' => $this->t('Fulfillment'),
        ];
      }
    }
    catch (\Exception $e) {
      // Fallback to default options if there's an error.
      $options = [
        'draft'       => $this->t('Draft'),
        'completed'   => $this->t('Completed'),
        'fulfillment' => $this->t('Fulfillment'),
      ];
      $this->loggerFactory->get('commerce_mautic_connect')->error('Error loading order states: @message', ['@message' => $e->getMessage()]);
    }

    return $options;
  }

  /**
   * Gets available currency options for the select field.
   *
   * @return array
   *   Array of currency options keyed by currency code.
   */
  protected function getCurrencyOptions(): array {
    $options = [];

    try {
      $currency_storage = $this->entityTypeManager->getStorage('commerce_currency');
      $currencies       = $currency_storage->loadMultiple();

      /** @var \Drupal\commerce_price\Entity\CurrencyInterface $currency */
      foreach ($currencies as $currency) {
        $options[$currency->getCurrencyCode()] = $this->t('@code - @name (@symbol)', [
          '@code'   => $currency->getCurrencyCode(),
          '@name'   => $currency->getName(),
          '@symbol' => $currency->getSymbol(),
        ]);
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('commerce_mautic_connect')->error('Error loading currencies: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return $options;
  }

  /**
   * Gets the default currency from the default store.
   *
   * @return string|null
   *   The default currency code, or NULL if not available.
   */
  protected function getDefaultCurrency(): ?string {
    try {
      /** @var \Drupal\commerce_store\StoreStorageInterface $store_storage */
      $store_storage = $this->entityTypeManager->getStorage('commerce_store');
      $default_store = $store_storage->loadDefault();

      if ($default_store) {
        $currency = $default_store->getDefaultCurrency();
        if ($currency) {
          return $currency->getCurrencyCode();
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('commerce_mautic_connect')->error('Error loading default currency: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return NULL;
  }

}

