<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Service for calculating Customer Metrics (RFM analysis).
 */
final class CustomerMetricsCalculationService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The exchanger calculator service (optional).
   *
   * @var object|null
   */
  protected $exchangerCalculator;

  /**
   * Constructs a new CustomerMetricsCalculationService.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container (for optional service loading).
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    LoggerChannelInterface $logger,
    ContainerInterface $container
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->logger = $logger;

    // Try to load commerce_exchanger service if available.
    if ($container->has('commerce_exchanger.calculate')) {
      $this->exchangerCalculator = $container->get('commerce_exchanger.calculate');
    }
  }

  /**
   * Calculates Customer Metrics (RFM) for a customer.
   *
   * @param string $customer_email
   *   The customer's email address.
   *
   * @return array
   *   Array containing:
   *   - total_spent: Total amount spent across all orders (Monetary)
   *   - total_orders: Number of orders (Frequency)
   *   - last_order_date: Timestamp of most recent order (Recency)
   *   - first_order_date: Timestamp of first order
   *   - average_order_value: Average order value
   *   Returns empty array if no orders found or calculation fails.
   */
  public function calculateCustomerMetrics(string $customer_email): array {
    if (empty($customer_email)) {
      $this->logger->warning('Cannot calculate Customer Metrics: empty email provided.');
      return [];
    }

    try {
      // Get configuration.
      $config = $this->configFactory->get('commerce_mautic_connect.settings');
      $included_states = $config->get('customer_metrics_order_states') ?: [];

      // If no states configured, return empty.
      if (empty($included_states)) {
        $this->logger->debug('Customer Metrics calculation skipped: no order states configured.');
        return [];
      }

      // Query orders for this customer email.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');
      $query = $order_storage->getQuery()
        ->condition('mail', $customer_email)
        ->condition('state', array_values($included_states), 'IN')
        ->accessCheck(FALSE)
        ->sort('placed', 'ASC');

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        $this->logger->debug('No orders found for Customer Metrics calculation: @email', ['@email' => $customer_email]);
        return [
          'total_spent' => 0,
          'total_orders' => 0,
          'last_order_date' => NULL,
          'first_order_date' => NULL,
          'average_order_value' => 0,
        ];
      }

      // Get base currency for conversion.
      $base_currency = $config->get('base_currency');

      // Load orders and calculate metrics.
      $orders = $order_storage->loadMultiple($order_ids);
      $total_spent = 0;
      $total_orders = count($orders);
      $first_order_date = NULL;
      $last_order_date = NULL;

      foreach ($orders as $order) {
        // Add to total spent (with currency conversion if available).
        $total_price = $order->getTotalPrice();
        if ($total_price) {
          $converted_price = $this->convertPrice($total_price, $base_currency);
          $total_spent += (float) $converted_price;
        }

        // Track dates.
        $placed_time = $order->getPlacedTime();
        if ($placed_time) {
          if ($first_order_date === NULL || $placed_time < $first_order_date) {
            $first_order_date = $placed_time;
          }
          if ($last_order_date === NULL || $placed_time > $last_order_date) {
            $last_order_date = $placed_time;
          }
        }
      }

      // Calculate average order value.
      $average_order_value = $total_orders > 0 ? $total_spent / $total_orders : 0;

      $metrics = [
        'total_spent' => round($total_spent, 2),
        'total_orders' => $total_orders,
        'last_order_date' => $last_order_date,
        'first_order_date' => $first_order_date,
        'average_order_value' => round($average_order_value, 2),
      ];

      $this->logger->info('Calculated Customer Metrics for @email: @orders orders, @spent total', [
        '@email' => $customer_email,
        '@orders' => $total_orders,
        '@spent' => $total_spent,
      ]);

      return $metrics;
    }
    catch (\Exception $e) {
      $this->logger->error('Error calculating Customer Metrics for @email: @message', [
        '@email' => $customer_email,
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Converts a price to the base currency.
   *
   * @param \Drupal\commerce_price\Price $price
   *   The price to convert.
   * @param string|null $target_currency
   *   The target currency code, or NULL if not configured.
   *
   * @return string
   *   The converted price as a decimal string.
   */
  protected function convertPrice($price, ?string $target_currency): string {
    // If no target currency or same currency, return as-is.
    if (empty($target_currency) || $price->getCurrencyCode() === $target_currency) {
      return $price->getNumber();
    }

    // If commerce_exchanger is available, use it for conversion.
    if ($this->exchangerCalculator) {
      try {
        $converted_price = $this->exchangerCalculator->priceConversion($price, $target_currency);
        return $converted_price->getNumber();
      }
      catch (\Exception $e) {
        $this->logger->warning('Currency conversion failed from @from to @to: @message. Using face value.', [
          '@from' => $price->getCurrencyCode(),
          '@to' => $target_currency,
          '@message' => $e->getMessage(),
        ]);
        // Fall back to face value if conversion fails.
        return $price->getNumber();
      }
    }

    // No exchanger available, log warning and use face value.
    $this->logger->debug('Commerce Exchanger not available. Using face value for @currency price.', [
      '@currency' => $price->getCurrencyCode(),
    ]);

    return $price->getNumber();
  }

}

