# Commerce Mautic Connect

## Overview

**Commerce Mautic Connect** synchronizes Drupal Commerce cart and order data to Mautic for advanced marketing automation. Enable abandoned cart recovery campaigns, customer segmentation based on purchase behavior (RFM analysis), and cross-device cart restoration via secure magic links.

## Features

**Abandoned Cart Recovery:**
- **Real-Time Cart Sync:** Sends HTML representation of shopping carts to Mautic the moment a user adds items - your emails show exactly what customers left behind
- **Time-Based Segmentation:** Tracks cart last updated timestamp, enabling sophisticated drip campaigns based on abandonment age (e.g., email after 1 hour, 24 hours, 72 hours)
- **Anonymous Cart Tracking:** Tracks guest visitors using Mautic's tracking cookie (`mtc_id`) - syncs carts even for users who aren't logged into Drupal
- **Magic Links with Auto-Login:** Generates secure, passwordless authentication links that restore carts across devices - automatically logs users in if the cart belongs to their account, maintaining proper order history
- **Cross-Device Recovery:** Customers can abandon cart on desktop, click email on mobile, and their cart appears instantly with seamless auto-login
- **Multilingual Support:** Automatically renders cart emails and magic links in the same language as the order - product names, URLs, and recovery links all match the customer's language preference
- **Email-Friendly HTML:** Generates table-based layouts compatible with all major email clients, with customizable CSS classes

**Coupon Tags:**
- **Automatic Tagging:** Tags Mautic contacts with coupon codes when they place orders - enables segmentation by promotions used
- **Configurable Prefix:** Customize tag prefix (e.g., `coupon:`, `promo-`, `discount_`)
- **Multiple Coupons:** Supports orders with multiple coupons - each coupon creates a separate tag
- **Historical Sync:** Drush command to batch sync coupon tags from all existing orders
- **Promotion Analysis:** Create Mautic segments like "Customers who used SUMMER-20" for targeted follow-up campaigns

**Customer Metrics & Segmentation:**
- **Automatic Metric Calculation:** Syncs customer metrics to Mautic on order completion, enabling RFM (Recency, Frequency, Monetary) analysis and advanced segmentation:
  - **Recency:** Last Order Date
  - **Frequency:** Total number of orders
  - **Monetary:** Lifetime value (total spent)
  - **Tenure:** First Order Date (customer age)
  - **AOV:** Average Order Value
- **Multi-Currency Support:** Select a base currency for calculating metrics - automatic currency conversion with [Commerce Exchanger](https://www.drupal.org/project/commerce_exchanger) module
- **Queue-Based Processing:** Uses Drupal's Queue API for background processing - RFM calculations never slow down checkout
- **Configurable Order States:** Choose which order states count towards metrics (completed, shipped, etc.)

**Developer-Friendly:**
- **Extensible Plugin Architecture:** Add custom Mautic features via the `MauticFeature` plugin system - other modules can add their own settings tabs
- **Automatic Field Creation:** Creates required custom fields in Mautic automatically - zero manual configuration needed
- **Event-Driven Architecture:** Uses Drupal's Event system for cart sync - no cron jobs required for real-time updates
- **Robust Error Handling:** All API errors are caught and logged to prevent checkout disruption
- **Drush Commands:** Force RFM sync for all customers or specific users via command line

## Use Cases

- **Cart Recovery:** Send "You left something behind" emails 1 hour after cart abandonment with actual product images and one-click restoration
- **VIP Segmentation (RFM - Monetary):** Create VIP segments in Mautic for customers who've spent over $500
- **Anniversary Campaigns (Tenure):** Send "Happy Anniversary" coupons based on first purchase date
- **Win-Back Campaigns (RFM - Recency):** Target customers who haven't purchased in 6+ months with special offers
- **Customer Lifecycle (RFM - Frequency):** Trigger different email sequences based on order frequency (new vs. repeat customers)
- **AOV Optimization:** Segment by average order value to personalize upsell opportunities
- **RFM Champions:** Identify and reward customers who score high across all RFM dimensions
- **Promotion Analysis (Coupon Tags):** Segment customers by which coupons they've used - follow up with customers who used "SUMMER-20" with related fall promotions

## Why This Module?

Unlike simple tracking pixels or analytics integrations, Commerce Mautic Connect provides:

- **Bidirectional Sync:** Data flows from Drupal → Mautic → back to Drupal via magic links
- **Passwordless Authentication:** Magic links automatically log users in, maintaining proper order history while providing frictionless cart recovery
- **Rich Cart Data:** Sends actual cart HTML to Mautic, not just event tracking
- **Multilingual Ready:** Automatically renders emails and recovery links in the customer's language
- **RFM Analysis Ready:** Automatically calculates and syncs metrics that enable RFM (Recency, Frequency, Monetary) segmentation for advanced customer targeting
- **Anonymous Support:** Works with anonymous visitors via Mautic's existing tracking cookie
- **Automatic Setup:** Creates all required Mautic custom fields automatically during configuration
- **Performance Optimized:** Queue-based processing ensures checkout speed isn't impacted
- **Enterprise Ready:** Built following Drupal 10 best practices with proper service injection and event architecture

## Requirements

- Drupal 10+
- PHP 8.1+
- Drupal Commerce (`commerce`, `commerce_cart`, `commerce_order`)
- **Advanced Mautic Integration** module (provides API connectivity to Mautic)
- A Mautic instance with API credentials (Basic Auth)

### Optional Dependencies

- **[Commerce Exchanger](https://www.drupal.org/project/commerce_exchanger)** - Required for automatic currency conversion in multi-currency stores. Without this module, orders in different currencies will be calculated at face value (e.g., $100 USD = €100 EUR).

## Installation

1. Install via Composer: `composer require drupal/commerce_mautic_connect`
2. Enable the module: `drush en commerce_mautic_connect`
3. Configure the module at `/admin/commerce/config/mautic-connect`

## Configuration

### Prerequisites

Ensure the **Advanced Mautic Integration** module is configured with valid Mautic API credentials (Basic Auth) before proceeding.

### Quick Setup

1. Navigate to **Commerce → Configuration → Mautic Connect** (`/admin/commerce/config/mautic-connect`)

2. **Abandoned Cart Tab:**
   - Configure field aliases:
     - Cart Items HTML (default: `commerce_cart_items_html`) - stores the rendered HTML cart table
     - Cart Last Updated (default: `commerce_cart_updated`) - timestamp for time-based segmentation
     - Cart Language (default: `commerce_cart_language`) - language code (e.g., pt, en, es) for language-based campaigns
   - Click **"Create Abandoned Cart Fields in Mautic"** - automatically creates all 3 required fields
   - Enable **"Enable Abandoned Cart Sync"** checkbox

   **Template Preview:**
   - Expand the "Template Preview" section to test how the cart email will look
   - Select a draft cart order from the dropdown (shows last 10 carts with items)
   - Select a theme to preview (Module Default or any installed theme)
   - Click **"Preview Template in New Tab"** to see exactly how the email will render
   - Useful for testing template customizations before sending real emails

3. **Customer Metrics Tab:**
   - Select your **Base Currency** - all metrics (CLV, AOV) will be calculated in this currency
   - Configure field aliases (or use defaults: `commerce_total_spent`, `commerce_total_orders`, etc.)
   - Select which **Order States** count as completed orders (e.g., Completed, Shipped)
   - Click **"Create Customer Metrics Fields in Mautic"** - automatically creates all 5 analytics fields
   - Enable **"Enable Customer Metrics Sync"** checkbox

   **Multi-Currency Stores:** If you have multiple currencies enabled and want accurate conversion, install the [Commerce Exchanger](https://www.drupal.org/project/commerce_exchanger) module. Without it, orders in different currencies will be summed at face value.

4. **Coupon Tags Tab:**
   - Enable **"Enable Coupon Tags Sync"** checkbox
   - Configure **Tag Prefix** (default: `coupon:`) - this is prepended to coupon codes when creating tags
   - Example: Coupon code `SUMMER-20` with prefix `coupon:` creates tag `coupon:SUMMER-20`

   **Use Cases:**
   - Segment customers by promotion usage (e.g., "All customers who used BLACKFRIDAY-2024")
   - Measure coupon campaign effectiveness
   - Create follow-up campaigns targeting users of specific promotions
   - Exclude recent coupon users from discount campaigns

**Important:** Fields must exist in Mautic before enabling features. The form validates this and prevents enabling until fields are created.

### Mautic Configuration

**Note:** Custom fields are automatically created by the module when you click the "Create Fields" buttons in the Drupal configuration form. Manual field creation is not required.

**Create Abandoned Cart Segment:**
1. In Mautic, go to **Segments > New**
2. Add filter: "Cart Items HTML" (`commerce_cart_items_html`) → "is not empty"
3. Save the segment

**Create Time-Based Abandoned Cart Segments** (Recommended):

Use the `Cart Last Updated` field for sophisticated drip campaigns:

- **Fresh Carts (1-2 hours):**
  - Filter: `commerce_cart_updated` → "greater than" → "now -2 hours"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send immediate "Don't forget!" email

- **Warm Carts (24 hours):**
  - Filter: `commerce_cart_updated` → "between" → "now -48 hours" and "now -24 hours"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send reminder with 10% discount code

- **Cold Carts (72 hours):**
  - Filter: `commerce_cart_updated` → "between" → "now -7 days" and "now -3 days"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send final "Last chance!" email with 15% discount

- **Cleanup (7+ days):**
  - Filter: `commerce_cart_updated` → "less than" → "now -7 days"
  - Action: Stop sending emails (stale carts)

**Create Language-Based Abandoned Cart Segments:**

Use the `Cart Language` field for multilingual campaigns:

- **Portuguese Abandoned Carts:**
  - Filter: `commerce_cart_language` → "equals" → "pt"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send Portuguese cart recovery emails

- **English Abandoned Carts:**
  - Filter: `commerce_cart_language` → "equals" → "en"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send English cart recovery emails

- **Spanish Abandoned Carts:**
  - Filter: `commerce_cart_language` → "equals" → "es"
  - AND `commerce_cart_items_html` → "is not empty"
  - Campaign: Send Spanish cart recovery emails

**Combine Time and Language Filters:**

For maximum conversion, combine both filters:
- Filter 1: `commerce_cart_updated` → "greater than" → "now -2 hours"
- Filter 2: `commerce_cart_language` → "equals" → "pt"
- Filter 3: `commerce_cart_items_html` → "is not empty"
- Result: Fresh Portuguese carts for immediate targeting

**Create Abandoned Cart Campaign:**
1. Go to **Campaigns > New** or **Emails > New**
2. Create an email targeting one of the time-based segments above
3. Use `{contactfield=commerce_cart_items_html}` in your email template to display cart contents
   - The cart HTML includes a pre-built "Recover My Cart" button with the magic link already embedded
   - No additional configuration needed - the recovery URL is automatically included in the template
4. Set up multiple emails with different timing for optimal conversion

**Create Customer Segments** (RFM examples):
- **VIP Customers (Monetary):** `commerce_total_spent` > 500
- **Recent Buyers (Recency):** `commerce_last_order_date` within last 30 days
- **Win-Back (Recency):** `commerce_last_order_date` more than 180 days ago
- **Repeat Customers (Frequency):** `commerce_total_orders` > 3
- **High AOV:** `commerce_aov` > 100
- **RFM Champions:** High on all three RFM dimensions (Recent + Frequent + High Spend)

## How It Works

### Cart Updates (Real-Time Event-Driven)

When a user adds, updates, or removes items from their cart, the module responds **immediately** via Drupal's Event system - no cron jobs required:

1. **Identity Check:**
   - **Priority 1:** Email address (if user is logged in or has entered email)
   - **Priority 2:** Mautic tracking cookie (`mtc_id`) for anonymous users

2. **If either identifier exists:**
   - Generates HTML summary of cart contents
   - Generates a secure "Magic Link" to restore the cart session
   - **Syncs to Mautic immediately** via API

3. **In Mautic:**
   - Contact enters "Abandoned Cart" segment
   - Abandoned cart email campaign is triggered based on your timing rules
   - Cart data includes recovery URL for one-click restoration

**Event-Driven = Real-Time:** Cart changes appear in Mautic within seconds, enabling time-sensitive campaigns.

### The "Magic Link" Recovery (Cross-Device Restoration with Auto-Login)

The module automatically generates secure, passwordless magic links for seamless cart recovery:

**How It Works:**
- Link is embedded in the "Recover My Cart" button in email templates
- Contains a cryptographically secure token (HMAC-based)
- **Auto-Login:** If the cart belongs to a registered user, clicking the link automatically logs them in
- **Cross-Device Compatible:** Customer can abandon cart on desktop, click email on mobile, and cart appears instantly
- **Security:** Token validates the order ID and creation time - prevents unauthorized access
- **Maintains Order History:** Cart stays with original owner's account (no orphaned carts)

**Cart Ownership Flow:**

1. **Anonymous user + User's cart** → Auto-login as cart owner
   - Example: Logged-in user abandons cart, clicks email when logged out
   - System automatically logs them back in
   - Cart remains in their account history

2. **Logged-in user + Same user's cart** → Restore cart
   - User already logged in as cart owner
   - Cart simply restored to session

3. **Logged-in user + Anonymous cart** → Assign to logged-in user
   - Guest cart assigned to authenticated user
   - Becomes part of their account

4. **Logged-in user + Different user's cart** → Access denied
   - Security: Cannot steal another user's cart

5. **User account blocked/deleted** → Fallback to anonymous
   - Cart transferred to anonymous session
   - Allows checkout without account

**Example Flow:**
1. Customer (logged in) adds items on laptop, closes browser
2. Customer logs out or uses different device
3. Receives email on phone 2 hours later
4. Clicks "Recover My Cart" magic link
5. **Automatically logged in** with "Welcome back!" message
6. Mobile browser opens with full cart ready for checkout
7. Order history maintained in customer account

This passwordless authentication approach (similar to Slack, Medium, etc.) significantly improves conversion rates while maintaining proper customer order history.

### Multilingual Email Rendering

The module automatically detects and respects the language of each order, ensuring a fully localized experience:

**What's Localized:**
- **Cart Email Template:** All template strings (headings, buttons, labels) render in the order's language
- **Product Names:** Product titles display in the customer's language (if translations exist)
- **Product URLs:** Links point to the translated product pages (e.g., `/pt/products/...` or `/en/products/...`)
- **Magic Link (Recovery URL):** Restoration link includes language parameter - customers land on the correct language version of your site
- **Cart URL:** Generic cart links also include the appropriate language prefix

**How It Works:**
1. When an order is created, Drupal records the customer's language preference (`langcode`)
2. Before rendering the cart email, the module temporarily switches to the order's language context
3. All URLs are generated with the correct language parameter (e.g., `?language=pt` or `/en/cart`)
4. Product entities are loaded in their translated version (if available)
5. After rendering, the original language context is restored

**Example:**
- Portuguese customer (`langcode: pt`) → Email in Portuguese with `/pt/cart/restore/...` magic link
- English customer (`langcode: en`) → Email in English with `/en/cart/restore/...` magic link
- Spanish customer (`langcode: es`) → Email in Spanish with `/es/cart/restore/...` magic link

This ensures customers receive emails in their preferred language and clicking recovery links lands them on the correct language version of your store - significantly improving conversion rates for multilingual stores.

### Email Address Capture

When a customer provides their email during checkout:

1. The order update event fires
2. The module checks for the `mtc_id` cookie
3. **If `mtc_id` exists:** Uses `edit()` to update the existing anonymous contact with the email (PATCH)
4. **If no `mtc_id`:** Uses `create()` to create/update contact by email
5. The cart contents are synced along with the email

This ensures:
- Anonymous visitors tracked by Mautic are properly linked to their email
- No duplicate contacts are created in Mautic
- Abandoned cart tracking starts as soon as the email is known
- Guest checkout flows are fully supported

### Order Completion

When an order is completed:

1. The module clears the cart HTML field in Mautic
2. Contact is removed from "Abandoned Cart" segment
3. No abandoned cart emails are sent

## The "Cookie Bridge" with Smart Merge Logic

This module's key differentiator is the **Cookie Bridge** - bidirectional data sync with anonymous visitor tracking and smart duplicate prevention:

**How It Works:**
- Mautic's tracking script sets an `mtc_id` cookie for all visitors
- This module reads that cookie and uses it to identify contacts in Mautic
- Cart data syncs to Mautic **even for anonymous users** (guests who haven't logged in)

**Smart Merge Logic:**
When an email is provided during checkout:
1. Module searches Mautic by email first
2. **If email exists:** Updates that contact (prevents duplicates and 422 API errors)
3. **If email is new:** Links the anonymous cookie contact to the email OR creates new contact
4. No duplicate contacts are created in Mautic

**Real-World Scenario:**
- Customer clicks email link from previous campaign → Mautic cookie is set → They're "Known" in Mautic
- They browse your store as a guest (not logged into Drupal)
- Add items to cart → Cart syncs to Mautic using their cookie ID
- Abandon cart → Mautic sends recovery email because they're already a known contact
- Works with Drupal guest checkout flow perfectly

This allows abandoned cart emails for users who **never logged into Drupal**, as long as Mautic knows them from any previous interaction (form submission, email click, etc.)

## Troubleshooting

### Carts Not Syncing

1. Check that Advanced Mautic Integration is properly configured
2. Verify the custom field alias matches in both Drupal and Mautic
3. Check Drupal logs: **Reports > Recent log messages** (filter by `commerce_mautic_connect`)

### Anonymous Users Not Being Tracked

1. Verify Mautic tracking script is installed on your site
2. Check that the `mtc_id` cookie is being set (use browser developer tools)
3. Ensure the anonymous user is already "known" in Mautic (has an email from a previous form submission)

### Errors During Checkout

All Mautic API errors are caught and logged to prevent checkout disruption. Check the Drupal logs for details.

## Customizing the Email Template

The module uses a Twig template to generate the cart HTML. You can customize it by copying the template to your theme.

### Template Location

The cart email template is available in the module's templates directory:

`templates/cart-email.html.twig`

### Available Variables

- `items` - Array of cart items with:
  - `title` - Product title
  - `quantity` - Item quantity
  - `unit_price` - Price per unit
  - `total_price` - Total price for this item
  - `product_url` - URL to the product (if available, already absolute)
  - `entity` - Full product variation entity object (allows accessing images, SKU, etc.)
- `cart_total` - Total cart price (formatted)
- `cart_url` - URL to view cart (already absolute, includes language prefix)
- `base_url` - Base URL of the site (e.g., `https://example.com`) - use this to build absolute URLs for images in emails

### CSS Classes for Styling

All classes are prefixed with `mautic-cart-` to avoid conflicts:

- `.mautic-cart` - Main table container
- `.mautic-cart-header` - Header row
- `.mautic-cart-item` - Individual cart item row
- `.mautic-cart-item-title` - Product title cell
- `.mautic-cart-item-qty` - Quantity cell
- `.mautic-cart-item-price` - Price cell
- `.mautic-cart-total` - Total row
- `.mautic-cart-button` - Call-to-action button

See `templates/cart-email-styles.css` for example styles you can use in Mautic.

### Override in Your Theme

1. Copy `cart-email.html.twig` to your theme's `templates/mautic/` directory
2. Modify as needed
3. Clear Drupal cache
4. Use the **Template Preview** feature to test your changes

**Multilingual Notes:**
- The template automatically renders in the order's language
- Use Twig's `t()` filter for translatable strings: `{{ 'Your Cart'|t }}`
- Product entities (`item.entity`) are automatically loaded in the correct translation
- Magic links (`cart_url`) automatically include the language prefix
- No additional configuration needed - language detection is automatic

**Using Images in Custom Templates:**

Email clients require absolute URLs for images. Use the `base_url` variable to convert relative image paths:

```twig
{% for item in items %}
  {% set product = item.entity.getProduct() %}
  {% if product.field_image is not empty %}
    {# Get the relative image URL #}
    {% set image_uri = product.field_image.0.entity.uri.value %}
    {% set styled_uri = image_uri|image_style('medium') %}
    {% set relative_url = file_url(styled_uri) %}

    {# Convert to absolute URL for email compatibility #}
    {% set image_url = relative_url starts with 'http' ? relative_url : (base_url ~ relative_url) %}

    <img src="{{ image_url }}" alt="{{ product.label }}" />
  {% endif %}
{% endfor %}
```

The `base_url` variable contains the full domain (e.g., `https://example.com`), so you can simply concatenate it with any relative path.

### Testing Template Changes

The module includes a built-in **Template Preview** tool to test your template customizations:

1. Navigate to **Commerce → Configuration → Mautic Connect**
2. Open the **Abandoned Cart** tab
3. Expand the **"Template Preview"** section
4. Select a draft cart order (must have items)
5. Select which theme to use:
   - **Module Default** - Uses the module's built-in template
   - **Your Theme** - Uses your theme's override (if it exists)
6. Click **"Preview Template in New Tab"**

The preview opens in a new browser tab showing exactly how the cart will render in emails. This is especially useful when:
- Testing template overrides in your theme
- Comparing different theme implementations
- Verifying product images and data display correctly
- Checking responsive email layouts

**Pro Tip:** Keep multiple preview tabs open with different themes to compare side-by-side while developing.

## Customer Metrics

This module also supports syncing RFM (Recency, Frequency, Monetary) analytics to Mautic for customer segmentation and personalization.

### Customer Metrics Tracked (Enables RFM Analysis)

This module tracks five key metrics that enable RFM (Recency, Frequency, Monetary) analysis and advanced customer segmentation:

- **Total Spent (Monetary/CLV)** - Customer Lifetime Value across all orders - the "M" in RFM
- **Total Orders (Frequency)** - Number of completed orders - the "F" in RFM
- **Last Order Date (Recency)** - Date of most recent order - the "R" in RFM
- **First Order Date (Tenure)** - Customer age - when they became a customer (enables cohort analysis)
- **Average Order Value (AOV)** - Average value per order (enables value-based segmentation)

These metrics enable traditional RFM segmentation plus sophisticated strategies like cohort analysis, customer lifecycle stages, and value-based targeting.

### How Customer Metrics Sync Works

Customer metrics (enabling RFM analysis) are calculated and synced **asynchronously** using Drupal's Queue API for optimal performance:

1. **Order State Transition** - When an order transitions to a configured state (e.g., "completed", "shipped")
2. **Queue Job Created** - A queue item is created with the customer's email
3. **Background Processing** - Cron automatically processes the queue every minute
4. **Calculate Metrics** - The system queries ALL orders for that customer and calculates RFM metrics
5. **Currency Conversion** - If Commerce Exchanger is installed, order totals are automatically converted to the base currency using exchange rates
6. **Sync to Mautic** - Metrics are sent to Mautic via API for segmentation

**Important:** Each queue item recalculates metrics from ALL customer orders, ensuring data accuracy even if multiple orders complete simultaneously. This provides accurate RFM scores for segmentation.

**Multi-Currency Example:**
- Customer places 3 orders: $100 USD, €50 EUR, £75 GBP
- Base currency is set to EUR
- With Commerce Exchanger: Converted to €84 + €50 + €87 = **€221 CLV**
- Without Commerce Exchanger: $100 + €50 + £75 = **€225 CLV** (face value - inaccurate)

### Queue Processing

The queue system ensures RFM updates don't slow down checkout:

- **Automatic Processing:** Cron runs every minute to process queued items
- **Manual Processing:** Use `drush queue:run commerce_mautic_connect_customer_metrics_sync`
- **Monitor Queue:** Use `drush queue:list` to see pending items
- **Idempotent:** Multiple queue items for the same customer are safe - the last sync wins

### Configuration

1. Navigate to **Configuration > Services > Commerce Mautic Connect**
2. Go to the **Customer Metrics** tab
3. **Select Base Currency** - Choose the currency for calculating CLV and AOV (all order totals will be converted to this currency)
4. Configure field aliases (or use defaults)
5. Select which order states should count towards RFM calculations
6. Click **"Create Customer Metrics Fields in Mautic"** - automatically creates all 5 fields for RFM analysis
7. Enable **"Enable Customer Metrics Sync"** checkbox

**Note:** Fields must exist in Mautic before enabling features. The form validates this and prevents enabling until fields are created. Once enabled, you can use these metrics for RFM segmentation in your Mautic campaigns.

**Multi-Currency Stores:** The configuration form will show a warning if multiple currencies are detected. For accurate conversion, install the [Commerce Exchanger](https://www.drupal.org/project/commerce_exchanger) module and configure exchange rates.

## Drush Commands

### Force Customer Metrics Sync for All Customers

Queue customer metrics (RFM) calculation for all customers with orders:

```bash
drush commerce-mautic-connect:customer-metrics-sync-all
# or use aliases:
drush cmcma
drush mautic-customer-metrics-sync-all
```

**Options:**
- `--limit=N` - Process only the first N customers (useful for testing)

**Example:**
```bash
# Test with first 50 customers
drush cmcma --limit=50

# Process all customers
drush cmcma
```

### Force Customer Metrics Sync for Specific Customer

Queue customer metrics (RFM) calculation for a single customer:

```bash
drush commerce-mautic-connect:customer-metrics-sync-customer customer@example.com
# or use aliases:
drush cmcmc customer@example.com
drush mautic-customer-metrics-sync-customer customer@example.com
```

### Sync All Abandoned Carts

Queue abandoned cart sync for all draft orders with email addresses:

```bash
drush commerce-mautic-connect:abandoned-cart-sync-all
# or use aliases:
drush cmcac
drush mautic-abandoned-cart-sync-all
```

**Options:**
- `--limit=N` - Process only the first N carts (useful for testing)

**Example:**
```bash
# Test with first 25 carts
drush cmcac --limit=25

# Process all abandoned carts
drush cmcac
```

> **⚠️ Important:** When running this command, always use the `--uri` parameter to ensure magic links are generated with the correct base URL:
> ```bash
> drush --uri=https://www.example.com cmcac
> ```
> Without `--uri`, Drush defaults to `http://default` which will create broken recovery links in your emails.

**Use Cases:**
- **Bulk Migration:** Sync existing draft orders when first installing the module
- **Data Recovery:** Re-sync carts after API connectivity issues
- **Testing:** Verify abandoned cart functionality with real data

### Sync Coupon Tags from Historical Orders

Batch sync coupon tags from all historical orders to Mautic contacts. This is useful for retroactively tagging customers who placed orders with coupons before the feature was enabled.

```bash
drush commerce-mautic-connect:coupon-tags-sync
# or use aliases:
drush cmccts
drush mautic-coupon-tags-sync
```

**Options:**
- `--limit=N` - Process only the first N orders (useful for testing)
- `--dry-run` - Preview what would be synced without making changes to Mautic
- `--states=state1,state2` - Comma-separated order states to include (default: completed,fulfillment,paid,processing,shipped)

**Examples:**
```bash
# Preview what would be synced (recommended first run)
drush cmccts --dry-run

# Test with first 100 orders
drush cmccts --limit=100

# Sync all historical orders
drush cmccts

# Only sync completed and paid orders
drush cmccts --states=completed,paid
```

**Output:**
The command provides detailed statistics including:
- Orders processed and skipped
- Contacts tagged (existing and new)
- Errors encountered
- Breakdown of tags by coupon code

**Use Cases:**
- **Initial Setup:** Tag all existing customers based on their historical coupon usage
- **Campaign Analysis:** See which coupons were most popular
- **Retroactive Segmentation:** Enable segmentation by promotions that were used before the feature was enabled

### Queue Management

After queuing jobs, manage the queues:

```bash
# Check queue status for all queues
drush queue:list

# Process customer metrics queue manually
drush queue:run commerce_mautic_connect_customer_metrics_sync

# Process abandoned cart queue manually
drush queue:run commerce_mautic_connect_abandoned_cart_sync

# Or wait for cron to process automatically (both queues run during cron)
```

### Use Cases

- **Initial Setup:** Run `drush cmcma` to sync all existing customers to Mautic for RFM analysis
- **Data Refresh:** Periodically run to ensure all RFM metrics are up-to-date
- **Fix Individual Customer:** Use `drush cmcmc` to recalculate specific customer's RFM scores
- **Testing:** Use `--limit` flag to test with small dataset first

## Extending the Module

### MauticFeature Plugin System

Commerce Mautic Connect uses an extensible plugin architecture. Each feature (Abandoned Cart, Customer Metrics) is a `MauticFeature` plugin. You can create your own plugins to add new Mautic integration features.

### Creating a Custom MauticFeature Plugin

1. Create a class in your module's `src/Plugin/MauticFeature/` directory:

```php
<?php

declare(strict_types=1);

namespace Drupal\my_module\Plugin\MauticFeature;

use Drupal\commerce_mautic_connect\Attribute\MauticFeature;
use Drupal\commerce_mautic_connect\MauticFeaturePluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides Wishlist sync to Mautic.
 */
#[MauticFeature(
  id: 'wishlist_sync',
  label: new TranslatableMarkup('Wishlist Sync'),
  weight: 20,
)]
class WishlistSync extends MauticFeaturePluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): array {
    $config = $config_factory->get('my_module.settings');

    $element = [
      '#type'  => 'details',
      '#title' => $this->t('Wishlist Sync'),
      '#group' => 'tabs',
    ];

    $element['enable_wishlist_sync'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Wishlist Sync'),
      '#description'   => $this->t('Sync wishlist items to Mautic.'),
      '#default_value' => $config->get('enable_wishlist_sync') ?? FALSE,
    ];

    // Add more form elements...

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Add validation logic...
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void {
    $config_factory->getEditable('my_module.settings')
      ->set('enable_wishlist_sync', $form_state->getValue('enable_wishlist_sync'))
      ->save();
  }

}
```

2. Clear Drupal cache: `drush cr`

3. Your new tab will appear in the Commerce Mautic Connect settings form.

### MauticFeature Plugin Attribute Properties

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `id` | string | Yes | Unique plugin identifier |
| `label` | TranslatableMarkup | Yes | Tab title displayed in the UI |
| `weight` | int | No | Order of tabs (lower = first). Default: 0 |

### Available Base Class Methods

The `MauticFeaturePluginBase` class provides these helper methods:

| Method | Description |
|--------|-------------|
| `createMauticField($alias, $label, $type)` | Creates a custom field in Mautic |
| `mauticFieldExists($field_alias)` | Checks if a Mautic field exists |
| `flattenErrorArray($errors)` | Converts nested errors to readable string |

### Injected Services

The base class provides access to:

- `$this->mauticApi` - Mautic API wrapper for API calls
- `$this->loggerFactory` - Logger factory for logging
- `$this->entityTypeManager` - Entity type manager for loading entities

To inject additional services, override the `create()` method in your plugin.

### Adding Custom Submit Buttons

For buttons that need plugin access in static callbacks, store the plugin instance in the form element (following the Commerce module pattern):

```php
public function buildForm(...): array {
  $element['my_button'] = [
    '#type'            => 'submit',
    '#value'           => $this->t('Do Something'),
    '#submit'          => [[static::class, 'myButtonSubmit']],
    '#plugin_instance' => $this,  // Store plugin reference
  ];
  return $element;
}

public static function myButtonSubmit(array &$form, FormStateInterface $form_state): void {
  $triggering_element = $form_state->getTriggeringElement();
  $plugin = $triggering_element['#plugin_instance'];  // Retrieve plugin
  $plugin->doSomething($form_state);
}
```

This avoids `\Drupal::service()` calls and maintains proper dependency injection.

### Altering Plugin Definitions

Use `hook_mautic_feature_info_alter()` to modify plugin definitions:

```php
/**
 * Implements hook_mautic_feature_info_alter().
 */
function my_module_mautic_feature_info_alter(array &$definitions): void {
  // Change the weight of abandoned_cart to appear last.
  if (isset($definitions['abandoned_cart'])) {
    $definitions['abandoned_cart']['weight'] = 100;
  }
}
```

## API

### Events Subscribed

**Abandoned Cart:**
- `CartEvents::CART_ENTITY_ADD` - When items are added to cart
- `CartEvents::CART_ORDER_ITEM_UPDATE` - When cart items are updated
- `CartEvents::CART_ORDER_ITEM_REMOVE` - When items are removed from cart
- `commerce_order.commerce_order.update` - When order is updated (captures email address)
- `commerce_order.place.post_transition` - When order is completed

**Customer Metrics:**
- `commerce_order.*.post_transition` - All order state transitions (when configured states are triggered)

### Services

- `plugin.manager.commerce_mautic_connect.mautic_feature` - Plugin manager for MauticFeature plugins
- `commerce_mautic_connect.cart_update_subscriber` - Abandoned cart event subscriber
- `commerce_mautic_connect.order_transition_subscriber` - Customer Metrics order transition subscriber
- `commerce_mautic_connect.customer_metrics_calculation` - Customer Metrics (RFM) calculation service with optional currency conversion
- `commerce_mautic_connect.recovery` - Cart recovery token service
- `logger.channel.commerce_mautic_connect` - Logger channel for debugging

### Queue Workers

- `commerce_mautic_connect_customer_metrics_sync` - Processes Customer Metrics (RFM) sync jobs in background with automatic currency conversion
- `commerce_mautic_connect_abandoned_cart_sync` - Processes abandoned cart sync jobs for bulk operations

### MauticFeature Plugins

- `abandoned_cart` - Abandoned cart sync settings (weight: 0)
- `coupon_tags` - Coupon tags sync settings (weight: 5)
- `customer_metrics` - Customer metrics/RFM sync settings (weight: 10)

### Plugin Classes

| Class | Purpose |
|-------|---------|
| `MauticFeaturePluginManager` | Discovers and manages MauticFeature plugins |
| `MauticFeaturePluginInterface` | Interface for all MauticFeature plugins |
| `MauticFeaturePluginBase` | Base class with shared services and helpers |
| `MauticFeature` (Attribute) | PHP 8 attribute for plugin discovery |

### Theme Hook

- `commerce_mautic_connect_cart_email` - Theme hook for the cart email template

## Development

### Testing

1. Add item to cart
2. Check Mautic contact record for cart HTML
3. Complete checkout
4. Verify cart HTML is cleared in Mautic

### Logging

Enable verbose logging by setting your Drupal log level to "Debug" and filtering for `commerce_mautic_connect`.

## Support

For issues, please check:
- Drupal logs for error messages
- Mautic API logs
- Network tab in browser developer tools for API calls

## License

This module is licensed under GPL-2.0+

