<?php

namespace Drupal\commerce_mautic_connect\Controller;

use Drupal\commerce_cart\CartProviderInterface;
use Drupal\commerce_cart\CartSessionInterface;
use Drupal\commerce_mautic_connect\Service\CartRecoveryService;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Drupal\Core\Url;

/**
 * Controller for restoring abandoned carts via magic links.
 */
class CartRestoreController extends ControllerBase {

  /**
   * The cart recovery service.
   *
   * @var \Drupal\commerce_mautic_connect\Service\CartRecoveryService
   */
  protected $recoveryService;

  /**
   * The cart session service.
   *
   * @var \Drupal\commerce_cart\CartSessionInterface
   */
  protected $cartSession;

  /**
   * The cart provider service.
   *
   * @var \Drupal\commerce_cart\CartProviderInterface
   */
  protected $cartProvider;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new CartRestoreController.
   *
   * @param \Drupal\commerce_mautic_connect\Service\CartRecoveryService $recovery_service
   *   The cart recovery service.
   * @param \Drupal\commerce_cart\CartSessionInterface $cart_session
   *   The cart session service.
   * @param \Drupal\commerce_cart\CartProviderInterface $cart_provider
   *   The cart provider service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(CartRecoveryService $recovery_service, CartSessionInterface $cart_session, CartProviderInterface $cart_provider, AccountInterface $current_user, RequestStack $request_stack) {
    $this->recoveryService = $recovery_service;
    $this->cartSession = $cart_session;
    $this->cartProvider = $cart_provider;
    $this->currentUser = $current_user;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_mautic_connect.recovery'),
      $container->get('commerce_cart.cart_session'),
      $container->get('commerce_cart.cart_provider'),
      $container->get('current_user'),
      $container->get('request_stack')
    );
  }

  /**
   * Restores the cart session.
   *
   * @param int $order
   *   The order ID to restore.
   * @param string $token
   *   The security token.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirects to the cart page.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   If the token is invalid or access is denied.
   */
  public function restore($order, $token) {
    // Load the order fresh from database.
    $order_storage = $this->entityTypeManager()->getStorage('commerce_order');
    $commerce_order = $order_storage->load($order);

    if (!$commerce_order) {
      throw new AccessDeniedHttpException('Order not found.');
    }

    // Validate the token.
    if (!$this->recoveryService->validateToken($commerce_order, $token)) {
      throw new AccessDeniedHttpException('Invalid recovery token.');
    }

    $order_id = $commerce_order->id();

    // Verify the cart is in draft state.
    if ($commerce_order->getState()->getId() !== 'draft') {
      $this->messenger()->addError($this->t('This cart has already been completed.'));
      $url = Url::fromRoute('commerce_cart.page')->toString();
      return new RedirectResponse($url);
    }

    // Verify the cart has items.
    $items = $commerce_order->getItems();
    if (empty($items)) {
      $this->messenger()->addError($this->t('This cart is empty and cannot be restored.'));
      $url = Url::fromRoute('commerce_cart.page')->toString();
      return new RedirectResponse($url);
    }

    // Log the restoration attempt.
    $this->getLogger('commerce_mautic_connect')->info('Restoring cart @order_id with @count items for user @uid', [
      '@order_id' => $order_id,
      '@count' => count($items),
      '@uid' => $this->currentUser->id(),
    ]);

    // Handle cart ownership based on current user status.
    $order_owner_id = (int) $commerce_order->getCustomerId();
    $current_user_id = (int) $this->currentUser->id();

    $needs_save = FALSE;

    if ($current_user_id > 0) {
      // User is logged in.
      if ($order_owner_id === 0) {
        // Cart is anonymous: Assign it to the logged-in user.
        $commerce_order->setCustomerId($current_user_id);
        $commerce_order->setEmail($this->currentUser->getEmail());
        $needs_save = TRUE;
        $this->getLogger('commerce_mautic_connect')->info('Assigning anonymous cart @order_id to logged-in user @uid', [
          '@order_id' => $order_id,
          '@uid' => $current_user_id,
        ]);
      }
      elseif ($order_owner_id !== $current_user_id) {
        // Cart belongs to a different user - deny access only if logged in as different user.
        $this->getLogger('commerce_mautic_connect')->warning('User @current_uid attempting to restore cart @order_id owned by user @owner', [
          '@current_uid' => $current_user_id,
          '@order_id' => $order_id,
          '@owner' => $order_owner_id,
        ]);
        throw new AccessDeniedHttpException('You do not have access to this cart.');
      }
    }
    else {
      // Anonymous user restoring a cart that belongs to a logged-in user.
      // Auto-login the user so the cart remains properly associated with their account.
      if ($order_owner_id !== 0) {
        $user_storage = $this->entityTypeManager()->getStorage('user');
        $original_user = $user_storage->load($order_owner_id);

        if ($original_user && $original_user->isActive()) {
          // Store the cart destination before user_login_finalize changes things.
          $request = $this->requestStack->getCurrentRequest();

          // Use user_login_finalize to properly log in the user.
          // This properly sets up the session and invokes all necessary hooks.
          \user_login_finalize($original_user);

          // After login finalization, explicitly set our destination in the request.
          // This will be picked up by the redirect subscriber.
          $request->query->set('destination', '/cart');

          $this->getLogger('commerce_mautic_connect')->info('Auto-logged in user @uid when restoring cart @order_id via magic link', [
            '@uid' => $order_owner_id,
            '@order_id' => $order_id,
          ]);

          $this->messenger()->addStatus($this->t('Welcome back! You have been automatically logged in.'));

          // Update current user ID for the rest of this request.
          $current_user_id = $order_owner_id;
        }
        else {
          // User account is blocked or deleted - transfer cart to anonymous.
          $commerce_order->setCustomerId(0);
          $needs_save = TRUE;

          $this->getLogger('commerce_mautic_connect')->warning('Cannot auto-login user @uid (blocked or deleted). Transferring cart @order_id to anonymous.', [
            '@uid' => $order_owner_id,
            '@order_id' => $order_id,
          ]);
        }
      }
    }

    // Ensure the order is marked as a cart (cart flag = 1).
    if (!$commerce_order->get('cart')->value) {
      $commerce_order->set('cart', TRUE);
      $needs_save = TRUE;
    }

    // Save changes if needed.
    if ($needs_save) {
      $commerce_order->save();
    }

    // Adopt the session (Magic!).
    // This forces the cart ID into the current session.
    $this->cartSession->addCartId($commerce_order->id());

    // Force save the session to ensure cart ID persists.
    $request = $this->requestStack->getCurrentRequest();
    $session = $request->getSession();
    $session->save();

    // Clear all cart-related caches to ensure the cart is properly recognized.
    $this->cartProvider->clearCaches();

    // Force reload the cart to ensure it's fresh in the provider's cache.
    $order_storage->resetCache([$order_id]);

    // Recalculate the order total to ensure prices are up to date.
    $commerce_order = $order_storage->load($order_id);
    if ($commerce_order) {
      try {
        $commerce_order->recalculateTotalPrice();
        $commerce_order->save();
      }
      catch (\Exception $e) {
        $this->getLogger('commerce_mautic_connect')->warning('Could not recalculate total price for cart @order_id: @message', [
          '@order_id' => $order_id,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    // Verify the cart is now visible to the cart provider.
    // After auto-login, we need to use the actual user account, not $this->currentUser.
    $this->cartProvider->clearCaches();

    // Get the actual user account to verify against.
    // If we just logged someone in, $current_user_id was updated, but $this->currentUser is stale.
    $user_storage = $this->entityTypeManager()->getStorage('user');
    $verification_user = $user_storage->load($current_user_id);

    if (!$verification_user) {
      $verification_user = $this->currentUser;
    }

    $visible_carts = $this->cartProvider->getCarts($verification_user);
    $cart_found = FALSE;

    foreach ($visible_carts as $visible_cart) {
      if ($visible_cart->id() == $order_id) {
        $cart_found = TRUE;
        break;
      }
    }

    if (!$cart_found) {
      $this->getLogger('commerce_mautic_connect')->warning('Cart @order_id was restored but is not visible in cart provider for user @uid.', [
        '@order_id' => $order_id,
        '@uid' => $verification_user->id(),
      ]);
    }

    // Set a success message (additional message for auto-login was already set above if applicable).
    $this->messenger()->addStatus($this->t('Your cart has been restored with @count item(s).', [
      '@count' => count($items),
    ]));

    // Redirect to cart page.
    // If user_login_finalize() was called, it sets up a redirect to /?check_logged_in=1
    // We need to override that with our cart page redirect.

    // Build URL with destination parameter to ensure we end up at cart page.
    $cart_url = Url::fromRoute('commerce_cart.page', [], ['absolute' => TRUE])->toString();

    return new RedirectResponse($cart_url);
  }

}

