<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Controller;

use Drupal\commerce_mautic_connect\Service\CartRecoveryService;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Theme\ThemeInitializationInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for rendering cart email template preview.
 */
final class TemplatePreviewController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The cart recovery service.
   *
   * @var \Drupal\commerce_mautic_connect\Service\CartRecoveryService
   */
  protected CartRecoveryService $recoveryService;

  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected ThemeManagerInterface $themeManager;

  /**
   * The theme initialization service.
   *
   * @var \Drupal\Core\Theme\ThemeInitializationInterface
   */
  protected ThemeInitializationInterface $themeInitialization;

  /**
   * Constructs a new TemplatePreviewController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\commerce_mautic_connect\Service\CartRecoveryService $recovery_service
   *   The cart recovery service.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   * @param \Drupal\Core\Theme\ThemeInitializationInterface $theme_initialization
   *   The theme initialization service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    RendererInterface $renderer,
    CartRecoveryService $recovery_service,
    ThemeManagerInterface $theme_manager,
    ThemeInitializationInterface $theme_initialization,
    LanguageManagerInterface $language_manager
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
    $this->recoveryService = $recovery_service;
    $this->themeManager = $theme_manager;
    $this->themeInitialization = $theme_initialization;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('commerce_mautic_connect.recovery'),
      $container->get('theme.manager'),
      $container->get('theme.initialization'),
      $container->get('language_manager')
    );
  }

  /**
   * Renders cart email template preview.
   *
   * @param int $order_id
   *   The order ID.
   * @param string $theme
   *   The theme name (default: '_module').
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rendered template as HTML response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If order is not found.
   */
  public function preview(int $order_id, string $theme = '_module'): Response {
    /** @var \Drupal\commerce_order\OrderStorageInterface $order_storage */
    $order_storage = $this->entityTypeManager->getStorage('commerce_order');
    /** @var \Drupal\commerce_order\Entity\OrderInterface|null $order */
    $order = $order_storage->load($order_id);

    if (!$order) {
      throw new NotFoundHttpException('Order not found.');
    }

    // Switch theme if requested.
    $original_theme = NULL;
    if ($theme !== '_module') {
      $original_theme = $this->themeManager->getActiveTheme();
      $active_theme = $this->themeInitialization->initTheme($theme);
      $this->themeManager->setActiveTheme($active_theme);
    }

    // Generate cart HTML.
    $html = $this->generateCartHtml($order);

    // Restore original theme.
    if ($original_theme) {
      $this->themeManager->setActiveTheme($original_theme);
    }

    return new Response($html);
  }

  /**
   * Generates HTML representation of cart contents.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order entity.
   *
   * @return string
   *   HTML string representing cart items.
   */
  protected function generateCartHtml($cart): string {
    $items = $cart->getItems();

    if (empty($items)) {
      return '<p>Cart is empty.</p>';
    }

    // Get the cart language using Commerce's approach:
    // Use customer's preferred language if available, otherwise current interface language.
    // This matches how Commerce sends order receipt emails.
    $order_language = $this->languageManager->getCurrentLanguage()->getId();
    $customer = $cart->getCustomer();
    if ($customer && !$customer->isAnonymous()) {
      $preferred_langcode = $customer->get('preferred_langcode')->value;
      if (!empty($preferred_langcode)) {
        $order_language = $preferred_langcode;
      }
    }
    $current_language = $this->languageManager->getCurrentLanguage()->getId();

    // Switch to order's language if different.
    $original_language = NULL;
    if ($order_language !== $current_language) {
      $original_language = $this->languageManager->getCurrentLanguage();
      $this->languageManager->setConfigOverrideLanguage($this->languageManager->getLanguage($order_language));
    }

    // Prepare items array for template.
    $template_items = [];
    foreach ($items as $item) {
      $purchased_entity = $item->getPurchasedEntity();
      $product_url = NULL;

      // Try to get product URL in the correct language.
      if ($purchased_entity && $purchased_entity->hasLinkTemplate('canonical')) {
        try {
          // Get translated entity if available.
          if ($purchased_entity->hasTranslation($order_language)) {
            $purchased_entity = $purchased_entity->getTranslation($order_language);
          }

          $product_url = $purchased_entity->toUrl('canonical', [
            'absolute' => TRUE,
            'language' => $this->languageManager->getLanguage($order_language),
          ])->toString();
        }
        catch (\Exception $e) {
          // URL generation failed, continue without URL.
        }
      }

      $unit_price = $item->getUnitPrice();
      $total_price = $item->getTotalPrice();

      $template_items[] = [
        'title' => $item->getTitle(),
        'quantity' => $item->getQuantity(),
        'unit_price' => $unit_price ? $unit_price->__toString() : '',
        'total_price' => $total_price ? $total_price->__toString() : '',
        'product_url' => $product_url,
        'entity' => $purchased_entity,
      ];
    }

    // Get cart URL in the correct language.
    $cart_url = NULL;
    try {
      $cart_url = Url::fromRoute('commerce_cart.page', [], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      // Cart URL generation failed.
    }

    // Generate Recovery URL in the correct language.
    $recovery_url = NULL;
    try {
      $token = $this->recoveryService->generateToken($cart);
      $recovery_url = Url::fromRoute('commerce_mautic_connect.restore', [
        'order' => $cart->id(),
        'token' => $token,
      ], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      // Failed to generate recovery URL.
    }

    // Get cart total.
    $cart_total = $cart->getTotalPrice();
    $cart_total_formatted = $cart_total ? $cart_total->__toString() : '';

    // Get base URL for email compatibility (absolute URLs for images, links).
    $base_url_value = \Drupal::request()->getSchemeAndHttpHost();

    // Render template in the order's language.
    $build = [
      '#theme' => 'commerce_mautic_connect_cart_email',
      '#items' => $template_items,
      '#cart_total' => $cart_total_formatted,
      '#cart_url' => $recovery_url ?: $cart_url,
      '#base_url' => $base_url_value,
    ];

    $html = (string) $this->renderer->renderPlain($build);

    // Restore original language.
    if ($original_language) {
      $this->languageManager->setConfigOverrideLanguage($original_language);
    }

    return $html;
  }

}

