<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Drush\Commands;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Url;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Commerce Mautic Connect.
 */
final class CommerceMauticConnectCommands extends DrushCommands {

  /**
   * Constructs a new CommerceMauticConnectCommands object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected QueueFactory $queueFactory,
    protected ConfigFactoryInterface $configFactory,
    protected MauticApiWrapperInterface $mauticApi,
  ) {
    parent::__construct();
  }

  /**
   * Force update Customer Metrics (RFM) for all customers in Mautic.
   */
  #[CLI\Command(name: 'commerce-mautic-connect:customer-metrics-sync-all', aliases: ['cmcma', 'mautic-customer-metrics-sync-all'])]
  #[CLI\Option(name: 'limit', description: 'Limit the number of customers to process. Useful for testing.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:customer-metrics-sync-all', description: 'Queue Customer Metrics sync for all customers with orders.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:customer-metrics-sync-all --limit=100', description: 'Queue Customer Metrics sync for the first 100 customers only.')]
  public function customerMetricsSyncAll(array $options = ['limit' => NULL]) {
    // Check if Customer Metrics is enabled.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    if (!$config->get('enable_customer_metrics')) {
      $this->logger()->warning('Customer Metrics sync is disabled in configuration. Enable it first at /admin/commerce/config/mautic-connect');
      return;
    }

    $this->logger()->notice('Starting Customer Metrics sync for all customers...');

    try {
      // Get all unique customer emails from orders.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');
      $query = $order_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('mail', '', '<>')
        ->sort('order_id', 'ASC');

      // Apply limit if specified.
      if (!empty($options['limit']) && is_numeric($options['limit'])) {
        $query->range(0, (int) $options['limit']);
      }

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        $this->logger()->warning('No orders found with email addresses.');
        return;
      }

      $this->logger()->notice('Found ' . count($order_ids) . ' orders. Extracting unique customer emails...');

      // Load orders and collect unique emails.
      $orders = $order_storage->loadMultiple($order_ids);
      $unique_emails = [];
      foreach ($orders as $order) {
        $email = $order->getEmail();
        if (!empty($email) && !isset($unique_emails[$email])) {
          $unique_emails[$email] = $order->id();
        }
      }

      $customer_count = count($unique_emails);
      $this->logger()->notice('Found ' . $customer_count . ' unique customers. Queueing Customer Metrics calculations...');

      // Queue Customer Metrics sync for each unique customer.
      $queue = $this->queueFactory->get('commerce_mautic_connect_customer_metrics_sync');
      $queued = 0;

      foreach ($unique_emails as $email => $order_id) {
        $queue->createItem([
          'email' => $email,
          'order_id' => $order_id,
        ]);
        $queued++;

        // Progress indicator for large datasets.
        if ($queued % 100 === 0) {
          $this->logger()->notice('Queued ' . $queued . ' of ' . $customer_count . ' customers...');
        }
      }

      $this->logger()->success('Successfully queued Customer Metrics sync for ' . $queued . ' customers.');
      $this->logger()->notice('Check queue status with: drush queue:list');
      $this->logger()->notice('Process the queue with: drush queue:run commerce_mautic_connect_customer_metrics_sync');
      $this->logger()->notice('Or wait for cron to process automatically.');
    }
    catch (\Exception $e) {
      $this->logger()->error('Error during Customer Metrics sync: ' . $e->getMessage());
    }
  }

  /**
   * Force update Customer Metrics (RFM) for a specific customer email.
   */
  #[CLI\Command(name: 'commerce-mautic-connect:customer-metrics-sync-customer', aliases: ['cmcmc', 'mautic-customer-metrics-sync-customer'])]
  #[CLI\Argument(name: 'email', description: 'The customer email address.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:customer-metrics-sync-customer customer@example.com', description: 'Queue Customer Metrics sync for a specific customer.')]
  public function customerMetricsSyncCustomer($email) {
    if (empty($email)) {
      $this->logger()->error('Email address is required.');
      return;
    }

    // Check if Customer Metrics is enabled.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    if (!$config->get('enable_customer_metrics')) {
      $this->logger()->warning('Customer Metrics sync is disabled in configuration. Enable it first at /admin/commerce/config/mautic-connect');
      return;
    }

    $this->logger()->notice('Queueing Customer Metrics sync for: ' . $email);

    try {
      // Verify customer has orders.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');
      $query = $order_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('mail', $email)
        ->range(0, 1);

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        $this->logger()->warning('No orders found for email: ' . $email);
        return;
      }

      // Queue the Customer Metrics sync.
      $queue = $this->queueFactory->get('commerce_mautic_connect_customer_metrics_sync');
      $queue->createItem([
        'email' => $email,
        'order_id' => reset($order_ids),
      ]);

      $this->logger()->success('Successfully queued Customer Metrics sync for: ' . $email);
      $this->logger()->notice('Check queue status with: drush queue:list');
      $this->logger()->notice('Process the queue with: drush queue:run commerce_mautic_connect_customer_metrics_sync');
    }
    catch (\Exception $e) {
      $this->logger()->error('Error queueing Customer Metrics sync: ' . $e->getMessage());
    }
  }

  /**
   * Sync all abandoned carts (draft orders with email) to Mautic.
   */
  #[CLI\Command(name: 'commerce-mautic-connect:abandoned-cart-sync-all', aliases: ['cmcac', 'mautic-abandoned-cart-sync-all'])]
  #[CLI\Option(name: 'limit', description: 'Limit the number of carts to process. Useful for testing.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:abandoned-cart-sync-all', description: 'Queue abandoned cart sync for all draft orders with email addresses.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:abandoned-cart-sync-all --limit=50', description: 'Queue abandoned cart sync for the first 50 draft orders only.')]
  public function abandonedCartSyncAll(array $options = ['limit' => NULL]): void {
    // Check if abandoned cart sync is enabled.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    if (!$config->get('enable_abandoned_cart')) {
      $this->logger()->warning('Abandoned cart sync is disabled in configuration. Enable it first at /admin/commerce/config/mautic-connect');
      return;
    }

    $this->logger()->notice('Starting abandoned cart sync for all draft orders...');

    try {
      // Get all draft orders with email addresses.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');
      $query = $order_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('state', 'draft')
        ->condition('mail', '', '<>')
        ->sort('order_id', 'ASC');

      // Apply limit if specified.
      if (!empty($options['limit']) && is_numeric($options['limit'])) {
        $query->range(0, (int) $options['limit']);
      }

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        $this->logger()->warning('No draft orders found with email addresses.');
        return;
      }

      $order_count = count($order_ids);
      $this->logger()->notice('Found ' . $order_count . ' draft orders with email addresses. Queueing abandoned cart sync...');

      // Load orders to verify they have items.
      $orders = $order_storage->loadMultiple($order_ids);
      $queue = $this->queueFactory->get('commerce_mautic_connect_abandoned_cart_sync');
      $queued = 0;

      foreach ($orders as $order) {
        $items = $order->getItems();
        $item_count = count($items);

        // Skip orders without items.
        if (empty($items)) {
          $this->logger()->debug('Skipping order @id: no items', ['@id' => $order->id()]);
          continue;
        }

        $this->logger()->debug('Queueing order @id with @count items (email: @email, state: @state)', [
          '@id' => $order->id(),
          '@count' => $item_count,
          '@email' => $order->getEmail(),
          '@state' => $order->getState()->getId(),
        ]);

        // Get base URL for queue item (works with drush --uri parameter).
        try {
          $base_url = \Drupal::request()->getSchemeAndHttpHost();
        }
        catch (\Exception $e) {
          // Fallback: generate from absolute URL.
          $temp_url = Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString();
          $parsed = parse_url($temp_url);
          $base_url = ($parsed['scheme'] ?? 'https') . '://' . ($parsed['host'] ?? 'localhost');
          if (isset($parsed['port']) && !in_array($parsed['port'], [80, 443])) {
            $base_url .= ':' . $parsed['port'];
          }
        }

        $queue->createItem([
          'order_id' => $order->id(),
          'email' => $order->getEmail(),
          'base_url' => $base_url,
        ]);
        $queued++;

        // Progress indicator for large datasets.
        if ($queued % 50 === 0) {
          $this->logger()->notice('Queued ' . $queued . ' of ' . $order_count . ' carts...');
        }
      }

      $this->logger()->success('Successfully queued abandoned cart sync for ' . $queued . ' carts.');
      $this->logger()->notice('Check queue status with: drush queue:list');
      $this->logger()->notice('Process the queue with: drush queue:run commerce_mautic_connect_abandoned_cart_sync');
      $this->logger()->notice('Or wait for cron to process automatically.');
    }
    catch (\Exception $e) {
      $this->logger()->error('Error during abandoned cart sync: ' . $e->getMessage());
    }
  }

  /**
   * Batch sync coupon tags from historical orders to Mautic contacts.
   *
   * Finds all placed orders with coupons and tags the corresponding Mautic
   * contacts, enabling retroactive segmentation by coupon usage.
   */
  #[CLI\Command(name: 'commerce-mautic-connect:coupon-tags-sync', aliases: ['cmccts', 'mautic-coupon-tags-sync'])]
  #[CLI\Option(name: 'limit', description: 'Limit the number of orders to process. Useful for testing.')]
  #[CLI\Option(name: 'dry-run', description: 'Show what would be done without making changes to Mautic.')]
  #[CLI\Option(name: 'states', description: 'Comma-separated order states to include (default: completed,fulfillment,paid,processing,shipped).')]
  #[CLI\Usage(name: 'commerce-mautic-connect:coupon-tags-sync', description: 'Sync coupon tags for all historical orders with coupons.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:coupon-tags-sync --limit=100', description: 'Sync coupon tags for the first 100 orders only.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:coupon-tags-sync --dry-run', description: 'Preview what tags would be synced without making changes.')]
  #[CLI\Usage(name: 'commerce-mautic-connect:coupon-tags-sync --states=completed,paid', description: 'Only sync orders in completed or paid states.')]
  public function couponTagsSync(array $options = ['limit' => NULL, 'dry-run' => FALSE, 'states' => NULL]): void {
    // Check if coupon tags feature is enabled.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');
    if (!$config->get('enable_coupon_tags')) {
      $this->logger()->warning('Coupon tags sync is disabled in configuration. Enable it first at /admin/commerce/config/mautic-connect');
      return;
    }

    $dry_run = $options['dry-run'];
    $prefix  = $config->get('coupon_tag_prefix') ?: 'coupon:';

    $this->logger()->notice('Starting coupon tags sync for historical orders...');
    $this->logger()->notice('Tag prefix: ' . $prefix);

    if ($dry_run) {
      $this->logger()->notice('DRY RUN MODE - No changes will be made to Mautic');
    }

    // Get Mautic API.
    if (!$dry_run) {
      $api = $this->mauticApi->getApi('contacts');
      if (!$api) {
        $this->logger()->error('Failed to connect to Mautic API.');
        return;
      }
    }

    // Determine which order states to include.
    $states = ['completed', 'fulfillment', 'paid', 'processing', 'shipped'];
    if (!empty($options['states'])) {
      $states = array_map('trim', explode(',', $options['states']));
    }

    $this->logger()->notice('Including order states: ' . implode(', ', $states));

    try {
      // Find all orders with coupons in the specified states.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');
      $query         = $order_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('state', $states, 'IN')
        ->condition('mail', '', '<>')
        ->condition('coupons', NULL, 'IS NOT NULL')
        ->sort('order_id', 'ASC');

      // Apply limit if specified.
      if (!empty($options['limit']) && is_numeric($options['limit'])) {
        $query->range(0, (int) $options['limit']);
      }

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        $this->logger()->warning('No orders found with coupons in the specified states.');
        return;
      }

      $this->logger()->notice('Found ' . count($order_ids) . ' orders. Processing...');

      // Statistics.
      $stats = [
        'processed'     => 0,
        'tagged'        => 0,
        'skipped'       => 0,
        'errors'        => 0,
        'contacts_new'  => 0,
        'tags_by_code'  => [],
      ];

      // Group orders by email to batch process.
      $orders          = $order_storage->loadMultiple($order_ids);
      $email_to_tags   = [];
      $email_to_orders = [];

      foreach ($orders as $order) {
        /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
        $email = $order->getEmail();

        if (empty($email)) {
          $stats['skipped']++;
          continue;
        }

        // Check for coupons.
        if (!$order->hasField('coupons') || $order->get('coupons')->isEmpty()) {
          $stats['skipped']++;
          continue;
        }

        // Collect coupon codes.
        foreach ($order->get('coupons') as $coupon_reference) {
          if ($coupon = $coupon_reference->entity) {
            $code = $coupon->getCode();
            $tag  = $prefix . $code;

            if (!isset($email_to_tags[$email])) {
              $email_to_tags[$email]   = [];
              $email_to_orders[$email] = [];
            }

            $email_to_tags[$email][$tag]     = $tag;
            $email_to_orders[$email][]       = $order->id();
            $stats['tags_by_code'][$code]    = ($stats['tags_by_code'][$code] ?? 0) + 1;
          }
        }

        $stats['processed']++;
      }

      // Now sync tags to Mautic contacts.
      $contact_count = count($email_to_tags);
      $this->logger()->notice('Found ' . $contact_count . ' unique customers with coupons. Syncing tags...');

      $synced = 0;
      foreach ($email_to_tags as $email => $tags) {
        $tags_array = array_values($tags);

        if ($dry_run) {
          $this->logger()->notice(sprintf(
            '[DRY RUN] Would tag %s with: %s (orders: %s)',
            $email,
            implode(', ', $tags_array),
            implode(', ', array_unique($email_to_orders[$email]))
          ));
          $stats['tagged']++;
        }
        else {
          try {
            // Search for existing contact.
            $search = $api->getList('email:' . $email, 0, 1);

            if (!empty($search['contacts'])) {
              // Contact found: Update with tags.
              $contact   = reset($search['contacts']);
              $target_id = $contact['id'];
              $result    = $api->edit($target_id, ['tags' => $tags_array]);

              if (isset($result['contact'])) {
                $stats['tagged']++;
                $this->logger()->debug(sprintf('Tagged contact %d (%s): %s', $target_id, $email, implode(', ', $tags_array)));
              }
              else {
                $stats['errors']++;
                $this->logger()->warning(sprintf('Failed to tag contact %d (%s)', $target_id, $email));
              }
            }
            else {
              // Contact not found: Create with email and tags.
              $result = $api->create([
                'email' => $email,
                'tags'  => $tags_array,
              ]);

              if (isset($result['contact'])) {
                $stats['tagged']++;
                $stats['contacts_new']++;
                $this->logger()->debug(sprintf('Created and tagged new contact for %s: %s', $email, implode(', ', $tags_array)));
              }
              else {
                $stats['errors']++;
                $this->logger()->warning(sprintf('Failed to create contact for %s', $email));
              }
            }
          }
          catch (\Exception $e) {
            $stats['errors']++;
            $this->logger()->error(sprintf('Error syncing %s: %s', $email, $e->getMessage()));
          }
        }

        $synced++;

        // Progress indicator every 50 contacts.
        if ($synced % 50 === 0) {
          $this->logger()->notice(sprintf('Processed %d of %d contacts...', $synced, $contact_count));
        }
      }

      // Summary.
      $this->logger()->success('Coupon tags sync completed!');
      $this->logger()->notice('');
      $this->logger()->notice('=== Summary ===');
      $this->logger()->notice(sprintf('Orders processed: %d', $stats['processed']));
      $this->logger()->notice(sprintf('Orders skipped (no email/coupons): %d', $stats['skipped']));
      $this->logger()->notice(sprintf('Contacts tagged: %d', $stats['tagged']));
      if (!$dry_run) {
        $this->logger()->notice(sprintf('New contacts created: %d', $stats['contacts_new']));
      }
      $this->logger()->notice(sprintf('Errors: %d', $stats['errors']));
      $this->logger()->notice('');
      $this->logger()->notice('=== Tags by Coupon Code ===');

      // Sort by count descending.
      arsort($stats['tags_by_code']);
      foreach ($stats['tags_by_code'] as $code => $count) {
        $this->logger()->notice(sprintf('  %s%s: %d orders', $prefix, $code, $count));
      }

      if ($dry_run) {
        $this->logger()->notice('');
        $this->logger()->notice('This was a DRY RUN. Run without --dry-run to apply changes.');
      }
    }
    catch (\Exception $e) {
      $this->logger()->error('Error during coupon tags sync: ' . $e->getMessage());
    }
  }

}

