<?php

namespace Drupal\commerce_mautic_connect\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for order state transitions to queue RFM calculations.
 */
class OrderTransitionSubscriber implements EventSubscriberInterface {

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new OrderTransitionSubscriber.
   *
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(QueueFactory $queue_factory, ConfigFactoryInterface $config_factory, LoggerChannelInterface $logger) {
    $this->queueFactory = $queue_factory;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Subscribe to all commerce_order state transitions.
    // The event names follow the pattern: commerce_order.{workflow_id}.post_transition
    // We'll use a generic approach and check in the handler.
    $events = [
      // Subscribe to common workflow transitions.
      'commerce_order.place.post_transition' => 'onOrderTransition',
      'commerce_order.validate.post_transition' => 'onOrderTransition',
      'commerce_order.fulfill.post_transition' => 'onOrderTransition',
      'commerce_order.cancel.post_transition' => 'onOrderTransition',
      // Custom Josefinas workflow transitions.
      'commerce_order.paid.post_transition' => 'onOrderTransition',
      'commerce_order.process.post_transition' => 'onOrderTransition',
      'commerce_order.shipping.post_transition' => 'onOrderTransition',
      'commerce_order.complete.post_transition' => 'onOrderTransition',
      'commerce_order.complete_with_refund.post_transition' => 'onOrderTransition',
      'commerce_order.complete_with_exchange.post_transition' => 'onOrderTransition',
      'commerce_order.cancel_with_refund.post_transition' => 'onOrderTransition',
    ];
    return $events;
  }

  /**
   * Handles order state transition events.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The workflow transition event.
   */
  public function onOrderTransition(WorkflowTransitionEvent $event) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getEntity();

    // Get configuration.
    $config = $this->configFactory->get('commerce_mautic_connect.settings');

    // Check if Customer Metrics sync is enabled.
    if (!$config->get('enable_customer_metrics')) {
      return;
    }

    // Get the email address.
    $email = $order->getEmail();
    if (empty($email)) {
      $this->logger->debug('Order @order_id has no email address. Skipping RFM queue.', [
        '@order_id' => $order->id(),
      ]);
      return;
    }

    // Get the new state.
    $new_state = $event->getToState()->getId();

    // Get configured included states.
    $included_states = $config->get('customer_metrics_order_states') ?: [];

    // Check if the new state is in the included states.
    if (!empty($included_states) && in_array($new_state, array_values($included_states))) {
      // Queue the Customer Metrics calculation.
      $queue = $this->queueFactory->get('commerce_mautic_connect_customer_metrics_sync');
      $queue->createItem([
        'email' => $email,
        'order_id' => $order->id(),
      ]);

      $this->logger->info('Queued RFM sync for @email (order @order_id transitioned to @state)', [
        '@email' => $email,
        '@order_id' => $order->id(),
        '@state' => $new_state,
      ]);
    }

    // Also queue if transitioning OUT of an included state (e.g., refund/cancellation).
    // This ensures metrics are recalculated when orders are removed from the calculation.
    $old_state = $event->getFromState()->getId();
    if (!empty($included_states) && in_array($old_state, array_values($included_states)) && !in_array($new_state, array_values($included_states))) {
      // Queue the Customer Metrics recalculation.
      $queue = $this->queueFactory->get('commerce_mautic_connect_customer_metrics_sync');
      $queue->createItem([
        'email' => $email,
        'order_id' => $order->id(),
      ]);

      $this->logger->info('Queued RFM recalculation for @email (order @order_id transitioned out of included state: @old_state -> @new_state)', [
        '@email' => $email,
        '@order_id' => $order->id(),
        '@old_state' => $old_state,
        '@new_state' => $new_state,
      ]);
    }
  }

}

