<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for MauticFeature plugins.
 *
 * Provides common functionality and shared services for all MauticFeature
 * plugins, including Mautic API access, entity management, and helper methods
 * for field creation and validation.
 *
 * @see \Drupal\commerce_mautic_connect\Attribute\MauticFeature
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginInterface
 */
abstract class MauticFeaturePluginBase extends PluginBase implements MauticFeaturePluginInterface {

  /**
   * The Mautic API wrapper service.
   *
   * @var \Drupal\advanced_mautic_integration\MauticApiWrapperInterface
   */
  protected MauticApiWrapperInterface $mauticApi;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a MauticFeaturePluginBase instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\advanced_mautic_integration\MauticApiWrapperInterface $mautic_api
   *   The Mautic API wrapper service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    MauticApiWrapperInterface $mautic_api,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->mauticApi         = $mautic_api;
    $this->loggerFactory     = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('advanced_mautic_integration.api'),
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): TranslatableMarkup {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return $this->pluginDefinition['weight'] ?? 0;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Default implementation does nothing.
    // Override in subclasses to add validation.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void {
    // Default implementation does nothing.
    // Override in subclasses to handle form submission.
  }

  /**
   * Creates or updates a custom field in Mautic.
   *
   * Handles API failures gracefully and provides user-friendly error messages.
   *
   * @param string $alias
   *   The field alias.
   * @param string $label
   *   The field label.
   * @param string $type
   *   The field type (text, html, number, date, datetime).
   */
  protected function createMauticField(string $alias, string $label, string $type): void {
    try {
      $api = $this->mauticApi->getApi('contactFields');
      if (!$api) {
        $this->messenger()->addError($this->t('Failed to connect to Mautic API. Please check your connection.'));
        return;
      }

      // Prepare field data.
      $data = [
        'label'       => $label,
        'alias'       => $alias,
        'type'        => $type,
        'isPublished' => TRUE,
        'group'       => 'core',
        'object'      => 'lead',
      ];

      // Try to create.
      $result = $api->create($data);

      if (isset($result['field'])) {
        $this->messenger()->addStatus($this->t('✓ Created field: @label (@alias)', [
          '@label' => $label,
          '@alias' => $alias,
        ]));
        $this->loggerFactory->get('commerce_mautic_connect')->info('Created field: @alias', ['@alias' => $alias]);
      }
      elseif (isset($result['errors'])) {
        // Check if it exists and update.
        $fields = $api->getList('', 0, 1000);
        if (isset($fields['fields'])) {
          foreach ($fields['fields'] as $field) {
            if ($field['alias'] === $alias) {
              $api->edit($field['id'], $data);
              $this->messenger()->addStatus($this->t('✓ Updated field: @label (@alias)', [
                '@label' => $label,
                '@alias' => $alias,
              ]));
              return;
            }
          }
        }
        $this->messenger()->addWarning($this->t('Could not create/update field: @alias. It may already exist or there may be a permissions issue.', ['@alias' => $alias]));
      }
    }
    catch (\Exception $e) {
      $error_message = $e->getMessage();

      // Provide user-friendly error messages.
      if (stripos($error_message, 'could not resolve host') !== FALSE ||
          stripos($error_message, 'connection refused') !== FALSE ||
          stripos($error_message, 'connection timed out') !== FALSE) {
        $this->messenger()->addError($this->t('✗ Mautic server is unreachable. Cannot create field: @alias (@label)', [
          '@alias' => $alias,
          '@label' => $label,
        ]));
      }
      else {
        $this->messenger()->addError($this->t('✗ Error creating field @alias (@label): @message', [
          '@alias'   => $alias,
          '@label'   => $label,
          '@message' => $error_message,
        ]));
      }

      $this->loggerFactory->get('commerce_mautic_connect')->error('Failed to create/update field @alias: @message', [
        '@alias'   => $alias,
        '@message' => $error_message,
      ]);
    }
  }

  /**
   * Checks if a Mautic contact field exists.
   *
   * @param string $field_alias
   *   The field alias to check.
   *
   * @return bool
   *   TRUE if the field exists, FALSE otherwise.
   */
  protected function mauticFieldExists(string $field_alias): bool {
    if (empty($field_alias)) {
      return FALSE;
    }

    try {
      $api = $this->mauticApi->getApi('contactFields');
      if (!$api) {
        $this->loggerFactory->get('commerce_mautic_connect')->warning('Failed to connect to Mautic API for field validation.');
        // Return TRUE to allow enabling if API is down (fail open).
        return TRUE;
      }

      // Get all fields from Mautic.
      $fields = $api->getList('', 0, 1000);

      if (isset($fields['fields']) && is_array($fields['fields'])) {
        foreach ($fields['fields'] as $field) {
          if (isset($field['alias']) && $field['alias'] === $field_alias) {
            return TRUE;
          }
        }
      }

      return FALSE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('commerce_mautic_connect')->error('Error checking Mautic field existence: @message', ['@message' => $e->getMessage()]);
      // Return TRUE to allow enabling if there's an error (fail open).
      return TRUE;
    }
  }

  /**
   * Flattens a nested error array into a readable string.
   *
   * @param mixed $errors
   *   The error data (can be string, array, or nested array).
   *
   * @return string
   *   A readable error message.
   */
  protected function flattenErrorArray(mixed $errors): string {
    if (is_string($errors)) {
      return $errors;
    }

    if (!is_array($errors)) {
      return (string) $errors;
    }

    $messages = [];
    foreach ($errors as $key => $value) {
      if (is_array($value)) {
        $messages[] = $this->flattenErrorArray($value);
      }
      else {
        $messages[] = (string) $value;
      }
    }

    return implode('; ', array_filter($messages));
  }

}

