<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Interface for MauticFeature plugins.
 *
 * MauticFeature plugins provide settings tabs for the Commerce Mautic Connect
 * configuration form. Each plugin is responsible for building its own form
 * elements, validation, and submission handling.
 *
 * @see \Drupal\commerce_mautic_connect\Attribute\MauticFeature
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginBase
 * @see \Drupal\commerce_mautic_connect\MauticFeaturePluginManager
 */
interface MauticFeaturePluginInterface extends PluginInspectionInterface, ContainerFactoryPluginInterface {

  /**
   * Builds the form elements for this feature's settings tab.
   *
   * The returned array should be a complete form element suitable for use
   * within vertical tabs, including the '#type' => 'details' wrapper.
   *
   * @param array $form
   *   The parent form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service for accessing module configuration.
   *
   * @return array
   *   A form element array representing this feature's settings tab.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): array;

  /**
   * Validates the form elements for this feature.
   *
   * @param array &$form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void;

  /**
   * Handles form submission for this feature.
   *
   * @param array &$form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service for saving module configuration.
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void;

  /**
   * Gets the human-readable label for this feature.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The feature label.
   */
  public function getLabel(): TranslatableMarkup;

  /**
   * Gets the weight for ordering this feature's tab.
   *
   * @return int
   *   The weight value. Lower weights appear first.
   */
  public function getWeight(): int;

}

