<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Plugin\MauticFeature;

use Drupal\commerce_mautic_connect\Attribute\MauticFeature;
use Drupal\commerce_mautic_connect\MauticFeaturePluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides the Abandoned Cart feature settings.
 *
 * Enables synchronization of shopping cart data to Mautic for abandoned cart
 * recovery campaigns. Includes cart HTML rendering, timestamp tracking for
 * time-based segmentation, and language detection for multilingual campaigns.
 */
#[MauticFeature(
  id: 'abandoned_cart',
  label: new TranslatableMarkup('Abandoned Cart'),
  weight: 0,
)]
class AbandonedCart extends MauticFeaturePluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): array {
    $config = $config_factory->get('commerce_mautic_connect.settings');

    $element = [
      '#type'  => 'details',
      '#title' => $this->t('Abandoned Cart'),
      '#group' => 'tabs',
    ];

    $element['enable_abandoned_cart'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Abandoned Cart Sync'),
      '#description'   => $this->t('If enabled, cart contents will be synced to Mautic when updated.'),
      '#default_value' => $config->get('enable_abandoned_cart') ?? FALSE,
    ];

    $element['field_cart_html'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Cart Items HTML Field Alias'),
      '#description'   => $this->t('The Mautic custom field alias to store the rendered HTML cart table.'),
      '#default_value' => $config->get('field_cart_html') ?: 'commerce_cart_items_html',
      '#required'      => TRUE,
    ];

    $element['field_cart_updated'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Cart Last Updated Field Alias'),
      '#description'   => $this->t('The Mautic custom field alias to store the cart last updated timestamp. Enables time-based segmentation (e.g., carts updated in last 2 hours).'),
      '#default_value' => $config->get('field_cart_updated') ?: 'commerce_cart_updated',
      '#required'      => TRUE,
    ];

    $element['field_cart_language'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Cart Language Field Alias'),
      '#description'   => $this->t('The Mautic custom field alias to store the cart language (e.g., pt, en, es). Enables language-based segmentation and campaigns.'),
      '#default_value' => $config->get('field_cart_language') ?: 'commerce_cart_language',
      '#required'      => TRUE,
    ];

    $element['create_fields'] = [
      '#type'  => 'details',
      '#title' => $this->t('Operations'),
      '#open'  => TRUE,
    ];

    $element['create_fields']['create_cart_fields'] = [
      '#type'           => 'submit',
      '#value'          => $this->t('Create Abandoned Cart Fields in Mautic'),
      '#submit'         => [[static::class, 'createAbandonedCartFieldsSubmit']],
      '#button_type'    => 'secondary',
      '#plugin_instance' => $this,
    ];

    // Template Preview section.
    $element['template_preview'] = [
      '#type'  => 'details',
      '#title' => $this->t('Template Preview'),
      '#open'  => FALSE,
    ];

    $element['template_preview']['description'] = [
      '#markup' => '<p>' . $this->t('Preview how the cart email template will look with real order data. Select an order and theme, then click "Preview" to open in a new tab.') . '</p>',
    ];

    // Get available orders and themes.
    $order_options = $this->getRecentDraftOrders();
    $theme_options = $this->getAvailableThemes();

    $element['template_preview']['preview_order_id'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Select Order'),
      '#description'   => $this->t('Choose a draft cart order to preview.'),
      '#options'       => $order_options,
      '#default_value' => key($order_options),
    ];

    $element['template_preview']['preview_theme'] = [
      '#type'          => 'select',
      '#title'         => $this->t('Select Theme'),
      '#description'   => $this->t('Choose which theme to use for rendering the template.'),
      '#options'       => $theme_options,
      '#default_value' => '_module',
    ];

    $element['template_preview']['preview_button'] = [
      '#type'       => 'html_tag',
      '#tag'        => 'button',
      '#value'      => $this->t('Preview Template in New Tab'),
      '#attributes' => [
        'type'    => 'button',
        'class'   => ['button', 'button--primary'],
        'onclick' => 'window.open(Drupal.url("admin/commerce/mautic-connect/preview/" + document.getElementById("edit-preview-order-id").value + "/" + document.getElementById("edit-preview-theme").value), "_blank");',
      ],
    ];

    $element['template_preview']['note'] = [
      '#markup' => '<div class="messages messages--status" style="margin-top: 15px;">' .
        $this->t('<strong>Note:</strong> The preview will open in a new browser tab showing exactly how the email will look in Mautic.') .
        '</div>',
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Skip validation if user is clicking "Create Fields" button.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#submit']) &&
        in_array([static::class, 'createAbandonedCartFieldsSubmit'], $triggering_element['#submit'])) {
      return;
    }

    // Validate Abandoned Cart fields if being enabled.
    if ($form_state->getValue('enable_abandoned_cart')) {
      $missing_fields = [];
      $cart_fields    = [
        'field_cart_html'    => $form_state->getValue('field_cart_html'),
        'field_cart_updated' => $form_state->getValue('field_cart_updated'),
      ];

      foreach ($cart_fields as $field_key => $field_alias) {
        if (!$this->mauticFieldExists($field_alias)) {
          $missing_fields[] = $field_alias;
        }
      }

      if (!empty($missing_fields)) {
        $form_state->setErrorByName('enable_abandoned_cart', $this->t('Cannot enable Abandoned Cart sync: The following fields do not exist in Mautic: @fields. Please create the fields first using the "Create Abandoned Cart Fields in Mautic" button below.', [
          '@fields' => implode(', ', $missing_fields),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void {
    $config_factory->getEditable('commerce_mautic_connect.settings')
      ->set('enable_abandoned_cart', $form_state->getValue('enable_abandoned_cart'))
      ->set('field_cart_html', $form_state->getValue('field_cart_html'))
      ->set('field_cart_updated', $form_state->getValue('field_cart_updated'))
      ->set('field_cart_language', $form_state->getValue('field_cart_language'))
      ->save();
  }

  /**
   * Submit handler for creating Abandoned Cart fields in Mautic.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function createAbandonedCartFieldsSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();

    /** @var \Drupal\commerce_mautic_connect\Plugin\MauticFeature\AbandonedCart $plugin */
    $plugin = $triggering_element['#plugin_instance'];

    // Save current form values first, then create fields.
    $plugin->saveAndCreateFields($form_state);
  }

  /**
   * Saves config and creates Abandoned Cart fields in Mautic.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state containing field aliases.
   */
  public function saveAndCreateFields(FormStateInterface $form_state): void {
    // Save current form values first.
    \Drupal::configFactory()->getEditable('commerce_mautic_connect.settings')
      ->set('field_cart_html', $form_state->getValue('field_cart_html'))
      ->set('field_cart_updated', $form_state->getValue('field_cart_updated'))
      ->set('field_cart_language', $form_state->getValue('field_cart_language'))
      ->save();

    // Create the fields.
    $this->createAbandonedCartFields($form_state);
  }

  /**
   * Creates Abandoned Cart fields in Mautic.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state containing field aliases.
   */
  public function createAbandonedCartFields(FormStateInterface $form_state): void {
    $cart_html_alias     = $form_state->getValue('field_cart_html');
    $cart_updated_alias  = $form_state->getValue('field_cart_updated');
    $cart_language_alias = $form_state->getValue('field_cart_language');

    // Create the cart HTML field.
    $this->createMauticField($cart_html_alias, 'Cart Items HTML', 'html');

    // Create the cart last updated timestamp field for time-based segmentation.
    $this->createMauticField($cart_updated_alias, 'Cart Last Updated', 'datetime');

    // Create the cart language field for language-based segmentation.
    $this->createMauticField($cart_language_alias, 'Cart Language', 'text');
  }

  /**
   * Gets recent draft orders with items for preview.
   *
   * @return array
   *   Array of order labels keyed by order ID.
   */
  protected function getRecentDraftOrders(): array {
    $options = [];

    try {
      /** @var \Drupal\commerce_order\OrderStorageInterface $order_storage */
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');

      $query = $order_storage->getQuery()
        ->accessCheck(TRUE)
        ->condition('state', 'draft')
        ->condition('cart', TRUE)
        ->sort('changed', 'DESC')
        ->range(0, 10);

      $order_ids = $query->execute();

      foreach ($order_ids as $order_id) {
        /** @var \Drupal\commerce_order\Entity\OrderInterface|null $order */
        $order = $order_storage->load($order_id);
        if ($order && !empty($order->getItems())) {
          $email       = $order->getEmail() ?: 'No email';
          $items_count = count($order->getItems());
          $total       = $order->getTotalPrice() ? $order->getTotalPrice()->__toString() : 'N/A';
          $date        = date('Y-m-d H:i', $order->getChangedTime());

          $options[$order_id] = sprintf(
            'Order #%d - %s - %d item%s - %s - %s',
            $order_id,
            $email,
            $items_count,
            $items_count > 1 ? 's' : '',
            $total,
            $date
          );
        }
      }

      if (empty($options)) {
        $options[0] = $this->t('No draft orders with items found');
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('commerce_mautic_connect')->error('Error loading draft orders: @message', [
        '@message' => $e->getMessage(),
      ]);
      $options[0] = $this->t('Error loading orders');
    }

    return $options;
  }

  /**
   * Gets available themes for preview.
   *
   * @return array
   *   Array of theme names keyed by machine name.
   */
  protected function getAvailableThemes(): array {
    $themes = [];

    // Add module default.
    $themes['_module'] = $this->t('Module Default (commerce_mautic_connect)');

    // Add all installed themes.
    /** @var \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler */
    $theme_handler = \Drupal::service('theme_handler');
    $theme_list    = $theme_handler->listInfo();
    foreach ($theme_list as $name => $theme) {
      if ($theme->status) {
        $themes[$name] = $theme->info['name'] ?? $name;
      }
    }

    return $themes;
  }

}

