<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Plugin\MauticFeature;

use Drupal\commerce_mautic_connect\Attribute\MauticFeature;
use Drupal\commerce_mautic_connect\MauticFeaturePluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides the Coupon Tags feature settings.
 *
 * Enables automatic tagging of Mautic contacts with coupon codes when they
 * place an order using a coupon. This allows for advanced segmentation in
 * Mautic based on which promotions customers have used (e.g., create segments
 * for customers who used "SUMMER-20" coupon).
 */
#[MauticFeature(
  id: 'coupon_tags',
  label: new TranslatableMarkup('Coupon Tags'),
  weight: 5,
)]
class CouponTags extends MauticFeaturePluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): array {
    $config = $config_factory->get('commerce_mautic_connect.settings');

    $element = [
      '#type'  => 'details',
      '#title' => $this->t('Coupon Tags'),
      '#group' => 'tabs',
    ];

    $element['enable_coupon_tags'] = [
      '#type'          => 'checkbox',
      '#title'         => $this->t('Enable Coupon Tags Sync'),
      '#description'   => $this->t('If enabled, Mautic contacts will be tagged with the coupon codes they use when placing orders. This enables segmentation based on promotions used.'),
      '#default_value' => $config->get('enable_coupon_tags') ?? FALSE,
    ];

    $element['coupon_tag_prefix'] = [
      '#type'          => 'textfield',
      '#title'         => $this->t('Coupon Tag Prefix'),
      '#description'   => $this->t('Prefix added to coupon codes when creating tags. For example, with prefix "coupon:" and code "SUMMER-20", the tag will be "coupon:SUMMER-20".'),
      '#default_value' => $config->get('coupon_tag_prefix') ?: 'coupon:',
      '#required'      => TRUE,
      '#maxlength'     => 50,
    ];

    $element['example'] = [
      '#type'   => 'markup',
      '#markup' => '<div class="messages messages--status">' .
        $this->t('<strong>Example:</strong> If a customer places an order with coupon code "SUMMER-20" and the prefix is "coupon:", the Mautic contact will receive the tag "coupon:SUMMER-20".') .
        '</div>',
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $prefix = $form_state->getValue('coupon_tag_prefix');

    // Validate tag prefix format (alphanumeric, hyphens, underscores, and colon).
    if (!empty($prefix) && !preg_match('/^[a-zA-Z0-9_\-:]+$/', $prefix)) {
      $form_state->setErrorByName('coupon_tag_prefix', $this->t('Tag prefix can only contain letters, numbers, hyphens, underscores, and colons.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state, ConfigFactoryInterface $config_factory): void {
    $config_factory->getEditable('commerce_mautic_connect.settings')
      ->set('enable_coupon_tags', $form_state->getValue('enable_coupon_tags'))
      ->set('coupon_tag_prefix', $form_state->getValue('coupon_tag_prefix'))
      ->save();
  }

}
