<?php

declare(strict_types=1);

namespace Drupal\commerce_mautic_connect\Plugin\QueueWorker;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\commerce_mautic_connect\Service\CartRecoveryService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Abandoned Cart sync queue items.
 *
 * @QueueWorker(
 *   id = "commerce_mautic_connect_abandoned_cart_sync",
 *   title = @Translation("Abandoned Cart Sync to Mautic"),
 *   cron = {"time" = 60}
 * )
 */
final class AbandonedCartSyncQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  protected EntityTypeManagerInterface $entityTypeManager;
  protected MauticApiWrapperInterface $mauticApi;
  protected ConfigFactoryInterface $configFactory;
  protected LoggerChannelInterface $logger;
  protected RendererInterface $renderer;
  protected CartRecoveryService $recoveryService;
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a new AbandonedCartSyncQueueWorker object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    MauticApiWrapperInterface $mautic_api,
    ConfigFactoryInterface $config_factory,
    LoggerChannelInterface $logger,
    RendererInterface $renderer,
    CartRecoveryService $recovery_service,
    LanguageManagerInterface $language_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->mauticApi = $mautic_api;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->renderer = $renderer;
    $this->recoveryService = $recovery_service;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('advanced_mautic_integration.api'),
      $container->get('config.factory'),
      $container->get('logger.channel.commerce_mautic_connect'),
      $container->get('renderer'),
      $container->get('commerce_mautic_connect.recovery'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (empty($data['order_id']) || empty($data['email'])) {
      $this->logger->warning('Abandoned cart queue item missing order_id or email. Skipping.');
      return;
    }

    $order_id = $data['order_id'];
    $email = $data['email'];
    $base_url = $data['base_url'] ?? '';

    try {
      // Check if abandoned cart sync is enabled.
      $config = $this->configFactory->get('commerce_mautic_connect.settings');
      if (!$config->get('enable_abandoned_cart')) {
        return;
      }

      // Load the order with a fresh load to avoid cache issues.
      $order_storage = $this->entityTypeManager->getStorage('commerce_order');

      // Reset the static cache to ensure we get fresh data.
      $order_storage->resetCache([$order_id]);

      $order = $order_storage->load($order_id);

      if (!$order) {
        $this->logger->warning('Order @order_id not found. Skipping abandoned cart sync.', ['@order_id' => $order_id]);
        return;
      }

      // Verify order is still in draft state.
      if ($order->getState()->getId() !== 'draft') {
        $this->logger->warning('Order @order_id is no longer in draft state (@state). Skipping abandoned cart sync.', [
          '@order_id' => $order_id,
          '@state' => $order->getState()->getId(),
        ]);
        return;
      }

      // Verify order still has items.
      $items = $order->getItems();
      if (empty($items)) {
        $this->logger->warning('Order @order_id has no items when reloaded. Skipping abandoned cart sync.', ['@order_id' => $order_id]);
        return;
      }

      // Generate cart HTML.
      $html = $this->generateCartHtml($order, $base_url);

      // Generate recovery URL.
      $token = $this->recoveryService->generateToken($order);
      $recovery_url = Url::fromRoute('commerce_mautic_connect.restore', [
        'order' => $order->id(),
        'token' => $token,
      ], ['absolute' => TRUE])->toString();

      // Get the cart language using Commerce's approach:
      // Use customer's preferred language if available, otherwise site default language.
      // This matches how Commerce sends order receipt emails.
      $order_language = $this->languageManager->getDefaultLanguage()->getId();
      $customer = $order->getCustomer();
      if ($customer && !$customer->isAnonymous()) {
        $preferred_langcode = $customer->get('preferred_langcode')->value;
        if (!empty($preferred_langcode)) {
          $order_language = $preferred_langcode;
        }
      }

      // Get the Mautic field aliases from configuration.
      $field_alias         = $config->get('field_cart_html') ?: 'commerce_cart_items_html';
      $field_cart_updated  = $config->get('field_cart_updated') ?: 'commerce_cart_updated';
      $field_cart_language = $config->get('field_cart_language') ?: 'commerce_cart_language';

      // Prepare cart data with cart's last changed timestamp and language.
      $mautic_data = [
        $field_alias => $html,
        'cart_recovery_url' => $recovery_url,
        $field_cart_updated => date('c', $order->getChangedTime()), // ISO 8601 format
        $field_cart_language => $order_language, // Order language code (e.g., pt, en, es)
      ];

      // Get the Mautic contacts API.
      $api = $this->mauticApi->getApi('contacts');
      if (!$api) {
        throw new \Exception('Failed to retrieve Mautic contacts API.');
      }

      // Search for existing contact by email.
      $search = $api->getList('email:' . $email, 0, 1);

      if (!empty($search['contacts'])) {
        // Contact exists: Update with cart data.
        $contact = reset($search['contacts']);
        $contact_id = $contact['id'];
        $result = $api->edit($contact_id, $mautic_data);

        if (isset($result['contact'])) {
          $this->logger->info('Synced abandoned cart to Mautic contact @id for email @email (order: @order_id)', [
            '@id' => $contact_id,
            '@email' => $email,
            '@order_id' => $order_id,
          ]);
        }
        else {
          $this->logger->warning('Unexpected response when updating abandoned cart for contact @id: @response', [
            '@id' => $contact_id,
            '@response' => json_encode($result),
          ]);
        }
      }
      else {
        // Contact doesn't exist: Create new contact with cart data.
        $mautic_data['email'] = $email;
        $result = $api->create($mautic_data);

        if (isset($result['contact'])) {
          $this->logger->info('Created Mautic contact with abandoned cart for email @email (order: @order_id)', [
            '@email' => $email,
            '@order_id' => $order_id,
          ]);
        }
        else {
          $this->logger->warning('Unexpected response when creating contact for @email: @response', [
            '@email' => $email,
            '@response' => json_encode($result),
          ]);
        }
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error syncing abandoned cart to Mautic for order @order_id: @message', [
        '@order_id' => $order_id,
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Generates HTML representation of cart contents.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order entity.
   * @param string $base_url
   *   The base URL for building absolute image URLs.
   *
   * @return string
   *   HTML string representing cart items.
   */
  protected function generateCartHtml($cart, string $base_url = ''): string {
    $items = $cart->getItems();

    if (empty($items)) {
      return '';
    }

    // Get the cart language using Commerce's approach:
    // Use customer's preferred language if available, otherwise site default language.
    // This matches how Commerce sends order receipt emails.
    $order_language = $this->languageManager->getDefaultLanguage()->getId();
    $customer = $cart->getCustomer();
    if ($customer && !$customer->isAnonymous()) {
      $preferred_langcode = $customer->get('preferred_langcode')->value;
      if (!empty($preferred_langcode)) {
        $order_language = $preferred_langcode;
      }
    }
    $current_language = $this->languageManager->getCurrentLanguage()->getId();

    // Switch to order's language if different.
    $original_language = NULL;
    if ($order_language !== $current_language) {
      $original_language = $this->languageManager->getCurrentLanguage();
      $this->languageManager->setConfigOverrideLanguage($this->languageManager->getLanguage($order_language));
    }

    // Prepare items array for template.
    $template_items = [];
    foreach ($items as $item) {
      $purchased_entity = $item->getPurchasedEntity();
      $product_url = NULL;

      // Try to get product URL in the correct language.
      if ($purchased_entity && $purchased_entity->hasLinkTemplate('canonical')) {
        try {
          // Get translated entity if available.
          if ($purchased_entity->hasTranslation($order_language)) {
            $purchased_entity = $purchased_entity->getTranslation($order_language);
          }

          $product_url = $purchased_entity->toUrl('canonical', [
            'absolute' => TRUE,
            'language' => $this->languageManager->getLanguage($order_language),
          ])->toString();
        }
        catch (\Exception $e) {
          // URL generation failed, continue without URL.
        }
      }

      $unit_price = $item->getUnitPrice();
      $total_price = $item->getTotalPrice();

      $template_items[] = [
        'title' => $item->getTitle(),
        'quantity' => $item->getQuantity(),
        'unit_price' => $unit_price ? $unit_price->__toString() : '',
        'total_price' => $total_price ? $total_price->__toString() : '',
        'product_url' => $product_url,
        'entity' => $purchased_entity,
      ];
    }

    // Get cart URL in the correct language.
    $cart_url = NULL;
    try {
      $cart_url = Url::fromRoute('commerce_cart.page', [], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      // Cart URL generation failed.
    }

    // Generate Recovery URL in the correct language.
    $recovery_url = NULL;
    try {
      $token = $this->recoveryService->generateToken($cart);
      $recovery_url = Url::fromRoute('commerce_mautic_connect.restore', [
        'order' => $cart->id(),
        'token' => $token,
      ], [
        'absolute' => TRUE,
        'language' => $this->languageManager->getLanguage($order_language),
      ])->toString();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to generate recovery URL: @message', ['@message' => $e->getMessage()]);
    }

    // Get cart total.
    $cart_total = $cart->getTotalPrice();
    $cart_total_formatted = $cart_total ? $cart_total->__toString() : '';

    // Render template in the order's language.
    $build = [
      '#theme' => 'commerce_mautic_connect_cart_email',
      '#items' => $template_items,
      '#cart_total' => $cart_total_formatted,
      '#cart_url' => $recovery_url ?: $cart_url,
      '#base_url' => $base_url,
    ];

    $html = (string) $this->renderer->renderPlain($build);

    // Restore original language.
    if ($original_language) {
      $this->languageManager->setConfigOverrideLanguage($original_language);
    }

    return $html;
  }

}

