<?php

namespace Drupal\commerce_mautic_connect\Plugin\QueueWorker;

use Drupal\advanced_mautic_integration\MauticApiWrapperInterface;
use Drupal\commerce_mautic_connect\Service\CustomerMetricsCalculationService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Customer Metrics sync queue items.
 *
 * @QueueWorker(
 *   id = "commerce_mautic_connect_customer_metrics_sync",
 *   title = @Translation("Customer Metrics Sync to Mautic"),
 *   cron = {"time" = 60}
 * )
 */
class CustomerMetricsSyncQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The Customer Metrics calculation service.
   *
   * @var \Drupal\commerce_mautic_connect\Service\CustomerMetricsCalculationService
   */
  protected $customerMetricsCalculation;

  /**
   * The Mautic API wrapper.
   *
   * @var \Drupal\advanced_mautic_integration\MauticApiWrapperInterface
   */
  protected $mauticApi;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a new CustomerMetricsSyncQueueWorker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\commerce_mautic_connect\Service\CustomerMetricsCalculationService $customer_metrics_calculation
   *   The Customer Metrics (RFM) calculation service.
   * @param \Drupal\advanced_mautic_integration\MauticApiWrapperInterface $mautic_api
   *   The Mautic API wrapper.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, CustomerMetricsCalculationService $customer_metrics_calculation, MauticApiWrapperInterface $mautic_api, ConfigFactoryInterface $config_factory, LoggerChannelInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->customerMetricsCalculation = $customer_metrics_calculation;
    $this->mauticApi = $mautic_api;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('commerce_mautic_connect.customer_metrics_calculation'),
      $container->get('advanced_mautic_integration.api'),
      $container->get('config.factory'),
      $container->get('logger.channel.commerce_mautic_connect')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    // Validate data structure.
    if (empty($data['email'])) {
      $this->logger->warning('Customer Metrics queue item missing email address. Skipping.');
      return;
    }

    $email = $data['email'];
    $order_id = $data['order_id'] ?? NULL;

    $this->logger->debug('Processing Customer Metrics sync for @email (order: @order)', [
      '@email' => $email,
      '@order' => $order_id ?: 'N/A',
    ]);

    try {
      // Get configuration.
      $config = $this->configFactory->get('commerce_mautic_connect.settings');

      // Check if Customer Metrics sync is enabled.
      if (!$config->get('enable_customer_metrics')) {
        $this->logger->debug('Customer Metrics sync is disabled. Skipping.');
        return;
      }

      // Calculate Customer Metrics.
      $metrics = $this->customerMetricsCalculation->calculateCustomerMetrics($email);

      if (empty($metrics)) {
        $this->logger->warning('Customer Metrics calculation returned empty metrics for @email', ['@email' => $email]);
        return;
      }

      // Get field aliases from config.
      $field_total_spent = $config->get('field_total_spent') ?: 'commerce_total_spent';
      $field_total_orders = $config->get('field_total_orders') ?: 'commerce_total_orders';
      $field_last_order_date = $config->get('field_last_order_date') ?: 'commerce_last_order_date';
      $field_first_order_date = $config->get('field_first_order_date') ?: 'commerce_first_order_date';
      $field_average_order_value = $config->get('field_average_order_value') ?: 'commerce_aov';

      // Prepare data for Mautic.
      $mautic_data = [
        $field_total_spent => $metrics['total_spent'],
        $field_total_orders => $metrics['total_orders'],
        $field_average_order_value => $metrics['average_order_value'],
      ];

      // Add dates if available (Mautic expects Y-m-d format for date fields).
      if (!empty($metrics['last_order_date'])) {
        $mautic_data[$field_last_order_date] = date('Y-m-d', $metrics['last_order_date']);
      }
      if (!empty($metrics['first_order_date'])) {
        $mautic_data[$field_first_order_date] = date('Y-m-d', $metrics['first_order_date']);
      }

      // Get Mautic contacts API.
      $api = $this->mauticApi->getApi('contacts');
      if (!$api) {
        throw new \Exception('Failed to retrieve Mautic contacts API.');
      }

      // Search for contact by email.
      $search = $api->getList('email:' . $email, 0, 1);

      if (!empty($search['contacts'])) {
        // Contact found: Update existing contact.
        $contact = reset($search['contacts']);
        $contact_id = $contact['id'];
        $result = $api->edit($contact_id, $mautic_data);

        if (isset($result['contact'])) {
          $this->logger->info('Synced Customer Metrics to Mautic contact @id for @email', [
            '@id' => $contact_id,
            '@email' => $email,
          ]);
        }
        else {
          $this->logger->warning('Unexpected response when updating Customer Metrics for contact @id: @response', [
            '@id' => $contact_id,
            '@response' => json_encode($result),
          ]);
        }
      }
      else {
        // Contact not found: Create new contact with Customer Metrics data.
        $mautic_data['email'] = $email;
        $result = $api->create($mautic_data);

        if (isset($result['contact'])) {
          $this->logger->info('Created Mautic contact with Customer Metrics for @email', ['@email' => $email]);
        }
        else {
          $this->logger->warning('Unexpected response when creating contact for @email: @response', [
            '@email' => $email,
            '@response' => json_encode($result),
          ]);
        }
      }
    }
    catch (\Exception $e) {
      // Log error but don't re-throw to prevent queue item from being retried indefinitely.
      $this->logger->error('Error syncing Customer Metrics to Mautic for @email: @message', [
        '@email' => $email,
        '@message' => $e->getMessage(),
      ]);
    }
  }

}


