<?php

namespace Drupal\commerce_micb\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\commerce_micb\MicbGatewayServiceInterface;
use Drupal\commerce_micb\Plugin\Commerce\PaymentGateway\MicbOffsiteRedirect;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for IPN requests.
 *
 * @package Drupal\commerce_micb\Controller
 */
class PaymentController implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The commerce micb logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  private LoggerChannelInterface $logger;

  /**
   * The entity type manager.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Micb gateway service.
   *
   * @var \Drupal\commerce_micb\MicbGatewayServiceInterface
   */
  protected MicbGatewayServiceInterface $micb;

  /**
   * Constructs a new PaymentCheckoutController object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory object.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   Request stack object.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\commerce_micb\MicbGatewayServiceInterface $micb_gateway
   *   Micb gateway service.
   */
  final public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    RequestStack $requestStack,
    EntityTypeManagerInterface $entity_type_manager,
    MicbGatewayServiceInterface $micb_gateway,
  ) {
    $this->logger = $logger_factory->get('commerce_micb');
    $this->requestStack = $requestStack;
    $this->entityTypeManager = $entity_type_manager;
    $this->micb = $micb_gateway;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('logger.factory'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('commerce_micb.gateway')
    );
  }

  /**
   * Provides the page for instant payment notification.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response represents an HTTP response.
   */
  public function ipnPage(Request $request) {
    $received_data = $request->request->all();

    $this->logger->info(
      'IPN request TRTYPE @trtype: @data',
      [
        '@trtype' => (string) $received_data['TRTYPE'] ?? '-',
        '@data' => Json::encode($received_data),
      ]
    );

    $missing_fields = [];
    $mac = '';
    foreach (MicbGatewayServiceInterface::MANDATORY_IPN_FIELDS as $field) {
      if (!isset($received_data[$field])) {
        $missing_fields[] = $field;
      }
      if ($field !== "P_SIGN") {
        $mac .= ($received_data[$field] == NULL) ?
          '-' : strlen($received_data[$field]) . $received_data[$field];
      }
    }

    if (count($missing_fields)) {
      $this->logger->critical(
        'IPN error: following fields are missing in the request @fields',
        ['@fields' => implode(',', $missing_fields)]
      );
      return new Response('', 200);
    }

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $this->entityTypeManager->getStorage('commerce_order')
      ->load((int) $received_data['ORDER']);
    if (!$order) {
      $this->logger->critical(
        'IPN error: order @order_id cannot be loaded',
        ['@order_id' => (int) $received_data['ORDER']]
      );
      return new Response('', 200);
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $order->payment_gateway->entity ?? NULL;
    /** @var \Drupal\commerce_micb\Plugin\Commerce\PaymentGateway\MicbOffsiteRedirect $payment_gateway_plugin */
    $payment_gateway_plugin = $payment_gateway ?
      $payment_gateway->getPlugin() : NULL;
    if (!$payment_gateway_plugin instanceof MicbOffsiteRedirect) {
      $this->logger->critical(
        'IPN error: order\'s payment gateway is not Micb Offsite'
      );
      return new Response('', 200);
    }
    $calculated_md5_hash = strtoupper(md5($mac));

    $micb_public_key = $this->micb->getMicbPublicKey(
      $payment_gateway_plugin->getConfiguration()['bank_public_key'] ?? ''
    );

    try {
      $decryted_md5_hash = $this->micb->decryptPsign(
        $micb_public_key,
        $received_data['P_SIGN']
      );
    }
    catch (\Exception $e) {
      $this->logger->critical(
        'IPN decryption error: @message', ['@message' => $e->getMessage()]
      );
      return new Response('', 200);
    }

    if ($calculated_md5_hash !== $decryted_md5_hash) {
      $this->logger->critical(
        "IPN error: failed to verify encrypted hash @chash vs @dhash",
        ['@chash' => $calculated_md5_hash, '@dhash' => $decryted_md5_hash]
      );
      return new Response('', 200);
    }

    try {
      $response = $payment_gateway_plugin->onNotify($request);
    }
    catch (\Exception $e) {
      $this->logger->critical(
        'IPN onNotify error: @message', ['@message' => $e->getMessage()]
      );
    }

    if (empty($response)) {
      $response = new Response('', 200);
    }

    return $response;
  }

  /**
   * Provides a page for waiting payment notification from the bank.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   *
   * @return array
   *   Wait page markup.
   */
  public function waitPage(Request $request, RouteMatchInterface $route_match) {
    $order = $route_match->getParameter('commerce_order');

    $return_url = Url::fromRoute(
      'commerce_payment.checkout.return',
      [
        'commerce_order' => $order->id(),
        'step' => 'payment',
      ],
      [
        'absolute' => TRUE,
        'query' => ['nonce' => $request->get('nonce')],
      ]
    );

    $build['message'] = [
      '#type' => 'item',
      '#markup' => $this->t('Please wait for transaction to complete...'),
      '#attached' => ['library' => ['commerce_micb/wait']],
    ];

    $build['link'] = [
      '#title' => $this->t('Or click here to check manually'),
      '#type' => 'link',
      '#attributes' => ['id' => 'order-return-page-link'],
      '#url' => $return_url,
    ];

    return $build;
  }

  /**
   * Checks access for the wait page.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  public function checkAccess(RouteMatchInterface $route_match, AccountInterface $account): AccessResult {
    $order = $route_match->getParameter('commerce_order');
    $nonce = $this->requestStack->getCurrentRequest()->get('nonce');
    if (empty($nonce)) {
      $this->logger->notice('Wait page access without providing nonce.');
      return AccessResult::forbidden();
    }

    $access = AccessResult::allowed()
      ->addCacheableDependency($order);

    return $access;
  }

}
