<?php

namespace Drupal\commerce_micb;

use Drupal\commerce_micb\Exception\MicbEncryptionException;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Library of MICB related constants and methods.
 */
class MicbGatewayService implements MicbGatewayServiceInterface {

  /**
   * Cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheDefault;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new MicbGatewayService object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_default
   *   The cache service for default bin.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager object.
   */
  public function __construct(
    CacheBackendInterface $cache_default,
    ModuleHandlerInterface $module_handler,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->cacheDefault = $cache_default;
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function getNonce(): string {
    return strtoupper(md5(uniqid('nonce', TRUE)));
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyLength($rsa_key): float {
    $rsa_key_details = openssl_pkey_get_details($rsa_key);
    return $rsa_key_details['bits'] / 8;
  }

  /**
   * {@inheritdoc}
   */
  public function generateMacSource($data): string {
    $mac_source = "";
    foreach ($data as $value) {
      if ($value != NULL) {
        $mac_source .= strlen($value) . $value;
      }
      else {
        $mac_source .= "-";
      }
    }
    return $mac_source;
  }

  /**
   * {@inheritdoc}
   */
  public function generateHashedData($rsa_key_length, $data): string {
    $first = self::FIRST;
    $prefix = self::PREFIX;
    $md5_hash = strtoupper(md5(self::generateMacSource($data)));
    $hash = $first;
    $paddingLength = $rsa_key_length - strlen($md5_hash) / 2 - strlen($prefix) / 2 - strlen($first) / 2;
    for ($i = 0; $i < $paddingLength; $i++) {
      $hash .= "FF";
    }
    $hash .= $prefix . $md5_hash;
    return $hash;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataHashableString($data): string {
    $hash_str = '';
    if ($data && count($data) > 0) {
      ksort($data);
      foreach ($data as $value) {
        if (is_array($value)) {
          $array_hash_str = $this->getDataHashableString($value);
          if ($array_hash_str !== '-') {
            $hash_str .= $array_hash_str;
          }
        }
        elseif (is_bool($value)) {
          $hash_str .= $value ? '1' : '0';
        }
        elseif (!empty($value)) {
          $hash_str .= $value;
        }
      }
    }
    if (strlen($hash_str) == 0) {
      $hash_str = '-';
    }
    return $hash_str;
  }

  /**
   * {@inheritdoc}
   */
  public function generatePsign($rsa_key, $hashed_data): string {
    $bin = pack("H*", $hashed_data);
    if (!openssl_private_encrypt($bin, $encrypted_bin, $rsa_key, \OPENSSL_NO_PADDING)) {
      $errors = ['Failed to encrypt data for P_SIGN'];
      while ($msg = openssl_error_string()) {
        $errors[] = $msg;
      }
      throw new MicbEncryptionException(implode('; ', $errors));
    }
    $hex = bin2hex($encrypted_bin);
    return strtoupper($hex);
  }

  /**
   * {@inheritdoc}
   */
  public function decryptPsign($rsa_key, $p_sign): string {
    $bin = pack("H*", $p_sign);
    if (!openssl_public_decrypt($bin, $decrypted_bin, $rsa_key)) {
      $errors = ['Failed to decrypt P_SIGN'];
      while ($msg = openssl_error_string()) {
        $errors[] = $msg;
      }
      throw new MicbEncryptionException(implode('; ', $errors));
    }
    $hex = strtoupper(bin2hex($decrypted_bin));
    return str_replace(self::DECRYPT_PREFIX, '', $hex);
  }

  /**
   * {@inheritdoc}
   */
  public function getPrivateKey($private_key_file_path, $private_key_pass) {
    $content = file_get_contents($private_key_file_path, $private_key_pass);
    return openssl_get_privatekey($content);
  }

  /**
   * {@inheritdoc}
   */
  public function getMicbPublicKey($certificate_file_path) {
    $content = file_get_contents($certificate_file_path);
    $cert = @openssl_x509_read($content);
    return $cert ?
      openssl_get_publickey($cert) : openssl_get_publickey($content);
  }

  /**
   * {@inheritdoc}
   */
  public function getGmt() {
    return ((date('Z') > 0) ? '+' : '') . (date('Z') / 3600);
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeStamp() {
    return date('YmdHis');
  }

  /**
   * {@inheritdoc}
   */
  public function getClientSource() {
    if (!empty($_SERVER['HTTP_USER_AGENT'])) {
      return preg_match('/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows (ce|phone)|xda|xiino/i', $_SERVER['HTTP_USER_AGENT']) || preg_match('/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i', substr($_SERVER['HTTP_USER_AGENT'], 0, 4))
        ? self::CLIENT_SOURCE_MOBILE : self::CLIENT_SOURCE_DESKTOP_WEB;
    }
    return self::CLIENT_SOURCE_DESKTOP_WEB;
  }

  /**
   * {@inheritdoc}
   */
  public function getStoreCategories():array {
    $cache_key = 'micb_store_categories';
    if ($categories = $this->cacheDefault->get($cache_key)) {
      return $categories->data;
    }
    $options = [];
    $csv_file = $this->moduleHandler
      ->getModule('commerce_micb')->getPath() . '/assets/store-categories.csv';
    $file = new \SplFileObject($csv_file);
    $file->setFlags(\SplFileObject::READ_CSV);
    foreach ($file as $row) {
      if (empty($row[0])) {
        continue;
      }
      [$id, $name, , $group] = $row;
      if (intval($id)) {
        $options[$group][$id] = $name;
      }
    }
    if ($options) {
      $this->cacheDefault->set($cache_key, $options, Cache::PERMANENT);
    }
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function storeTransactionData(OrderInterface $order, array $transaction_data) {
    $order_micb_data = $order->getData(self::ORDER_DATA_TRANSACTIONS_KEY, []);
    $rrn = $transaction_data['RRN'] ?: '-';
    $transaction_type = $transaction_data['TRTYPE'] ?? '-';
    $order_micb_data[$rrn][$transaction_type] = $transaction_data;
    $order->setData('micb_transactions', $order_micb_data);
    /** @var \Drupal\commerce_order\OrderStorageInterface $order_storage */
    $order_storage = $this->entityTypeManager->getStorage('commerce_order');
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $order_storage->loadForUpdate($order->id());
    $order->setData('micb_transactions', $order_micb_data);
    $order->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getTransactionData(OrderInterface $order, string $rrn, string $transaction_type) {
    $order_micb_data = $order->getData(self::ORDER_DATA_TRANSACTIONS_KEY, []);
    return $order_micb_data[$rrn][$transaction_type] ?? [];
  }

}
