<?php

namespace Drupal\commerce_micb;

use Drupal\commerce_order\Entity\OrderInterface;

/**
 * Library of MICB related constants and methods.
 */
interface MicbGatewayServiceInterface {
  const TRTYPE_AUTHORIZE = '0';

  const TRTYPE_CAPTURE = '21';

  const TRTYPE_REVERSAL = '24';

  const RESPONSE_CODE_OK = 0;

  const FIRST = '0001';

  const PREFIX = '003020300C06082A864886F70D020505000410';

  const DECRYPT_PREFIX = '3020300C06082A864886F70D020505000410';

  const CLIENT_SOURCE_DESKTOP_WEB = 'desktop_web';

  const CLIENT_SOURCE_MOBILE = 'mobile';

  const MANDATORY_IPN_FIELDS = [
    'TRTYPE',
    'TERMINAL',
    'ORDER',
    'CURRENCY',
    'AMOUNT',
    'TIMESTAMP',
    'NONCE',
    'RRN',
    'INT_REF',
    'ACTION',
    'RC',
    'APPROVAL',
  ];

  const ORDER_DATA_TRANSACTIONS_KEY = 'micb_transactions';

  /**
   * Gets a random value to be used as nonce.
   *
   * @return string
   *   The nonce value.
   */
  public function getNonce(): string;

  /**
   * Get RSA key length.
   *
   * @param \OpenSSLAsymmetricKey $rsa_key
   *   RSA key.
   *
   * @return float
   *   Key length.
   */
  public function getKeyLength($rsa_key): float;

  /**
   * Generate MAC source.
   *
   * @param array $data
   *   Source data array for source MAC generation.
   *
   * @return string
   *   The source MAC.
   */
  public function generateMacSource($data): string;

  /**
   * Create hashed data.
   *
   * @param float $rsa_key_length
   *   Key length.
   * @param array $data
   *   Data array to hash.
   *
   * @return string
   *   Hashed data string.
   */
  public function generateHashedData($rsa_key_length, $data): string;

  /**
   * Get a a string for hashable data array.
   *
   * @param array $data
   *   Data array to hash.
   *
   * @return string
   *   The string to hash.
   */
  public function getDataHashableString($data): string;

  /**
   * Generate P_SIGN.
   *
   * @param \OpenSSLAsymmetricKey $rsa_key
   *   Private RSA key.
   * @param string $hashed_data
   *   Hash of the sent data.
   *
   * @throws \Drupal\commerce_micb\Exception\MicbEncryptionException
   *   P_SIGN encryption failed.
   *
   * @return string
   *   The P_SIGN string.
   */
  public function generatePsign($rsa_key, $hashed_data): string;

  /**
   * Decrypt P_SIGN.
   *
   * @param \OpenSSLAsymmetricKey $rsa_key
   *   Private RSA key.
   * @param string $p_sign
   *   Received P_SIGN.
   *
   * @throws \Drupal\commerce_micb\Exception\MicbEncryptionException
   *   P_SIGN decryption failed.
   *
   * @return string
   *   The P_SIGN string.
   */
  public function decryptPsign($rsa_key, $p_sign): string;

  /**
   * Get local private key.
   *
   * @param string $private_key_file_path
   *   Path to the local private key file.
   * @param string $private_key_pass
   *   Private key password.
   *
   * @return \OpenSSLAsymmetricKey|false
   *   Local private key.
   */
  public function getPrivateKey($private_key_file_path, $private_key_pass);

  /**
   * Get Micb public key.
   *
   * @param string $certificate_file_path
   *   Path to the bank public certificate file.
   *
   * @return \OpenSSLAsymmetricKey|false
   *   Public key from banks certificate.
   */
  public function getMicbPublicKey($certificate_file_path);

  /**
   * Get timezone.
   *
   * @return string
   *   Timezone.
   */
  public function getGmt();

  /**
   * Get time stamp.
   *
   * @return string
   *   Time stamp.
   */
  public function getTimeStamp();

  /**
   * Get client source type.
   *
   * @return string
   *   Client source type.
   */
  public function getClientSource();

  /**
   * Return a list of MICB options for store categories.
   *
   * @return array
   *   Categories.
   */
  public function getStoreCategories(): array;

  /**
   * Store transaction details in order data.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param array $transaction_data
   *   Transaction data.
   */
  public function storeTransactionData(OrderInterface $order, array $transaction_data);

  /**
   * Get transaction details from order data.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param string $rrn
   *   The RRN for payment.
   * @param string $transaction_type
   *   Transaction type.
   *
   * @return array
   *   Transaction data.
   */
  public function getTransactionData(OrderInterface $order, string $rrn, string $transaction_type);

}
