<?php

namespace Drupal\commerce_micb\Plugin\Commerce\PaymentGateway;

use Drupal\commerce\Response\NeedsRedirectException;
use Drupal\commerce_micb\Exception\MicbException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Url;
use Drupal\commerce_micb\MicbGatewayServiceInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsAuthorizationsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_price\Price;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Render\Markup;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\MessageFormatter;
use GuzzleHttp\Middleware;
use GuzzleHttp\RequestOptions;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides the Off-site Redirect payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "micb_redirect",
 *   label = "MICB (Off-site redirect)",
 *   display_label = "MICB",
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_micb\PluginForm\OffsiteRedirect\MicbPaymentOffsiteForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "mastercard", "visa",
 *   },
 *   requires_billing_information = FALSE,
 * )
 */
class MicbOffsiteRedirect extends OffsitePaymentGatewayBase implements SupportsAuthorizationsInterface, SupportsRefundsInterface {

  /**
   * The commerce micb logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The Micb gateway service.
   *
   * @var \Drupal\commerce_micb\MicbGatewayServiceInterface
   */
  protected MicbGatewayServiceInterface $micb;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var \Drupal\Core\Config\ConfigFactoryInterface $config_factory */
    $config_factory = $container->get('config.factory');
    if (empty($configuration['merchant_name'])) {
      $configuration['merchant_name'] = $config_factory->get('system.site')
        ->get('name');
    }
    if (empty($configuration['merchant_email'])) {
      $configuration['merchant_email'] = $config_factory->get('system.site')
        ->get('mail');
    }
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.factory')->get('commerce_micb');
    $instance->micb = $container->get('commerce_micb.gateway');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'merchant_id' => '',
      'terminal' => '',
      'merchant_name' => '',
      'merchant_url' => Url::fromRoute('<front>', [], ['absolute' => TRUE])->toString(),
      'merchant_email' => '',
      'store_category' => '162', /* Other IT services. */
      'country' => '',
      'send_shipping_info' => TRUE,
      'private_key_path' => 'modules/contrib/commerce_micb/assets/test-private-key.pem',
      'private_key_password' => '',
      'public_key_path' => 'modules/contrib/commerce_micb/assets/test-pub-key.pem',
      'bank_public_key' => 'modules/contrib/commerce_micb/assets/MICB-pub-key.pem',
      'intent' => 'capture',
      'data_tokens' => NULL,
      'live_gateway_url' => 'https://ecomgw3.micb.md/ecom/cgi-bin/checkout',
      'live_gateway_url_complete' => 'https://ecomgw3.micb.md/ecom/cgi-bin/sale_completion',
      'live_gateway_url_reversal' => 'https://ecomgw3.micb.md/ecom/cgi-bin/reversal',
      'test_gateway_url' => 'https://ecomgw3.micb.md/ecom/cgi-bin/checkout',
      'test_gateway_url_complete' => 'https://ecomgw3.micb.md/ecom/cgi-bin/sale_completion',
      'test_gateway_url_reversal' => 'https://ecomgw3.micb.md/ecom/cgi-bin/reversal',
      'debug' => TRUE,
      'debug_file' => 'private://micb-requests.log',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['ipn_url'] = [
      '#type' => 'item',
      '#title' => $this->t('IPN callback for sending notifications to be provided to the bank'),
      '#markup' => Url::fromRoute('commerce_micb.ipn', [], ['absolute' => TRUE])->toString(),
    ];

    $form['merchant_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Name'),
      '#description' => $this->t('Merchant name (recognizable by cardholder)'),
      '#default_value' => $this->configuration['merchant_name'],
      '#required' => TRUE,
    ];

    $form['merchant_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant URL'),
      '#description' => $this->t('Merchant web page'),
      '#default_value' => $this->configuration['merchant_url'],
      '#required' => TRUE,
    ];

    $form['merchant_email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Email'),
      '#description' => $this->t('Merchant Email address'),
      '#default_value' => $this->configuration['merchant_email'],
      '#required' => TRUE,
    ];

    $form['merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant ID'),
      '#description' => $this->t('Merchant ID assigned by bank'),
      '#default_value' => $this->configuration['merchant_id'],
      '#required' => TRUE,
    ];

    $form['terminal'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Terminal'),
      '#description' => $this->t('Merchant Terminal ID assigned by bank'),
      '#default_value' => $this->configuration['terminal'],
      '#required' => TRUE,
    ];

    $form['country'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Country'),
      '#description' => $this->t('Merchant company country ISO2 code'),
      '#default_value' => $this->configuration['country'],
      '#required' => TRUE,
    ];

    $form['send_shipping_info'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send shipping info to the bank'),
      '#description' => $this->t('Add shipping address to payment request'),
      '#default_value' => $this->configuration['send_shipping_info'],
    ];

    $form['store_category'] = [
      '#type' => 'select',
      '#title' => $this->t('Store category'),
      '#description' => $this->t('Specify what do you sell on your store'),
      '#options' => $this->micb->getStoreCategories(),
      '#default_value' => $this->configuration['store_category'],
      '#required' => TRUE,
    ];

    $form['keys'] = [
      '#type' => 'details',
      '#title' => $this->t('Instructions to create certificate/private key'),
      '#open' => FALSE,
    ];

    $form['keys']['keys_info'] = [
      '#type' => 'item',
      '#title' => $this->t('A certificate and private key must be created.'),
      '#markup' => '<i>The certificate must be transmited to the bank.
        <br>The Private key should be stored securely and should not be
        transmitted to unauthorized persons. <br>You can create both by using
        openssl as in following example:</i><code>
        openssl req -new -newkey rsa:2048 -text -nodes -out my-organisation.csr
        -keyout my-organisation_private_key.pem -days 3600</code>',
    ];

    $form['private_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path to the private key PEM file'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['private_key_path'],
      '#description' => $this->t('Use modules/contrib/commerce_micb/assets/test-private-key.pem for testing.'),
    ];

    $form['private_key_password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password for private key if required'),
      '#default_value' => '',
    ];

    $form['public_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path to the certificate PEM file containing public key'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['public_key_path'],
      '#description' => $this->t('Not used, can be ignored. Use modules/contrib/commerce_micb/assets/test-pub-key.pem for testing.'),
    ];

    $form['test_keys'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('On submit test private public keys match'),
    ];

    $form['bank_public_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Bank public key provided by MICB'),
      '#default_value' => $this->configuration['bank_public_key'],
      '#required' => TRUE,
      '#description' => $this->t('Use modules/contrib/commerce_micb/assets/MICB-pub-key.pem for testing.'),
    ];

    $form['live_gateway_urls'] = [
      '#type' => 'details',
      '#title' => $this->t('Live gateway URLs'),
      '#open' => FALSE,
    ];

    $form['live_gateway_urls']['live_gateway_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live authorize URL'),
      '#default_value' => $this->configuration['live_gateway_url'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['live_gateway_url']),
    ];

    $form['live_gateway_urls']['live_gateway_url_complete'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live finalize URL'),
      '#default_value' => $this->configuration['live_gateway_url_complete'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['live_gateway_url_complete']),
    ];

    $form['live_gateway_urls']['live_gateway_url_reversal'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live reverse URL'),
      '#default_value' => $this->configuration['live_gateway_url_reversal'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['live_gateway_url_reversal']),
    ];

    $form['test_gateway_urls'] = [
      '#type' => 'details',
      '#title' => $this->t('Test gateway URLs'),
      '#open' => FALSE,
    ];

    $form['test_gateway_urls']['test_gateway_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test authorize URL'),
      '#default_value' => $this->configuration['test_gateway_url'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['test_gateway_url']),
    ];

    $form['test_gateway_urls']['test_gateway_url_complete'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test finalize URL'),
      '#default_value' => $this->configuration['test_gateway_url_complete'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['test_gateway_url_complete']),
    ];

    $form['test_gateway_urls']['test_gateway_url_reversal'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test reverse URL'),
      '#default_value' => $this->configuration['test_gateway_url_reversal'],
      '#required' => TRUE,
      '#parents' => array_merge($form['#parents'], ['test_gateway_url_reversal']),
    ];

    $form['data_tokens'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Replace authorization values with tokens (YAML)'),
      '#required' => FALSE,
      '#default_value' => Yaml::encode($this->configuration['data_tokens']),
      '#attributes' => ['data-yaml-editor' => 'true'],
      '#element_validate' => [[$this, 'validateIsYaml']],
      '#description' => $this->t('Examples: @examples', [
        '@examples' => Markup::create(
          '<br>COUNTRY: \'[commerce_order:field_order_country:value]\'' .
          '<br>CUSTOMER:<br>&nbsp;&nbsp;PHONE: \'[commerce_order:field_order_phone_number:value]\'' .
          '<br>LINE_ITEMS:<br>&nbsp;&nbsp;0:<br>&nbsp;&nbsp;&nbsp;&nbsp;BRAND: ' .
          '\'[commerce_order:order_items:0:entity:field_first_brand:value]\'' .
          '<br>LINE_ITEMS:<br>&nbsp;&nbsp;_NR_:<br>&nbsp;&nbsp;&nbsp;&nbsp;BRAND: ' .
          '\'[commerce_order:order_items:_NR_:entity:purchased_entity:entity:field_product_brand:value]\''
        ),
      ]),
    ];
    if (function_exists('token_element_validate')) {
      $form['data_tokens']['#element_validate'][] = 'token_element_validate';
      $form['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['commerce_order', 'commerce_payment'],
        '#show_restricted' => TRUE,
        '#show_nested' => TRUE,
        '#recursion_limit' => 4,
      ];
    }
    else {
      $form['token_tree'] = [
        '#item' => $this->t('Install token module in order to use this option'),
      ];
    }

    $form['intent'] = [
      '#type' => 'radios',
      '#title' => $this->t('Transaction type'),
      '#options' => [
        'capture' => $this->t("Capture (capture payment immediately after customer's approval)"),
        'authorize' => $this->t('Authorize (requires manual or automated capture after checkout)'),
      ],
      '#description' => $this->t('For more information on capturing a prior authorization, please refer to <a href=":url" target="_blank">Capture an authorization</a>.',
        [':url' => 'https://docs.drupalcommerce.org/commerce2/user-guide/payments/capture']),
      '#default_value' => $this->configuration['intent'],
    ];

    $form['debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log debug info'),
      '#default_value' => $this->configuration['debug'],
    ];

    $form['debug_file'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Log file'),
      '#default_value' => $this->configuration['debug_file'],
      '#description' => $this->t('Ex: /tmp/micb-requests.log'),
      '#states' => [
        'visible' => [
          ':input[name="configuration[micb_redirect][debug]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);
    $values = $form_state->getValue($form['#parents']);
    $public_key_ok = FALSE;
    $private_key_ok = FALSE;
    $old_configuration = $this->getConfiguration();
    // Validate public key.
    if (!file_exists($values['public_key_path'])) {
      $form_state->setErrorByName(
        'public_key_path',
        $this->t("Incorrect path to public key")
      );
    }
    else {
      $rsa_public_key = file_get_contents($values['public_key_path']);
      if (openssl_get_publickey($rsa_public_key)) {
        $public_key_ok = TRUE;
      }
      else {
        $form_state->setErrorByName(
          'public_key_path',
          $this->t("Can't get public key from file")
        );
      }
    }
    // Validate private key.
    if (!file_exists($values['private_key_path'])) {
      $form_state->setErrorByName(
        'private_key_path',
        $this->t("Incorrect path to private key")
      );
    }
    else {
      $rsa_private_key = file_get_contents($values['private_key_path']);
      $old_pass = $old_configuration['private_key_password'] ?? '';
      $key_pass = empty($values['private_key_password']) ?
        $old_pass : $values['private_key_password'];
      if (openssl_get_privatekey($rsa_private_key, $key_pass)) {
        $private_key_ok = TRUE;
      }
      else {
        $form_state->setErrorByName(
          'private_key_path',
          $this->t("Can't get private key from file")
        );
      }
    }

    if (!empty($values['test_keys']) && $public_key_ok && $private_key_ok &&
      isset($rsa_private_key) && isset($key_pass) && isset($rsa_public_key)) {
      $data = NULL;
      $decripted = NULL;
      openssl_private_encrypt('test', $data, [$rsa_private_key, $key_pass]);
      openssl_public_decrypt($data, $decripted, $rsa_public_key);
      if ($decripted !== 'test') {
        $form_state->setErrorByName(
          'private_key_path',
          $this->t("Private and public keys do not match")
        );
      }
    }

    if ($values['debug'] && empty(trim($values['debug_file']))) {
      $form_state->setErrorByName('debug_file', $this->t("Path to log file missing"));
    }
  }

  /**
   * Validate inserted Yaml is usable.
   *
   * @param array $element
   *   Submitted element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateIsYaml(array $element, FormStateInterface $form_state) {
    if (!empty($element['#value'])) {
      $error = '';
      try {
        $decoded = Yaml::decode($element['#value']);
        $form_state->setValueForElement($element, $decoded);
      }
      catch (\Exception $e) {
        $decoded = NULL;
        $error = $e->getMessage();
      }
      if (!$decoded) {
        $form_state
          ->setError($element, $this->t('Invalid YAML. @error', ['@error' => $error]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $keys = [
        'merchant_id',
        'terminal',
        'merchant_name',
        'merchant_url',
        'merchant_email',
        'store_category',
        'country',
        'send_shipping_info',
        'private_key_path',
        'public_key_path',
        'bank_public_key',
        'data_tokens',
        'intent',
        'live_gateway_url',
        'live_gateway_url_complete',
        'live_gateway_url_reversal',
        'test_gateway_url',
        'test_gateway_url_complete',
        'test_gateway_url_reversal',
        'debug',
        'debug_file',
      ];
      $values = $form_state->getValue($form['#parents']);
      foreach ($keys as $key) {
        $this->configuration[$key] = $values[$key];
      }
      if (!empty($values['private_key_password'])) {
        $this->configuration['private_key_password'] = $values['private_key_password'];
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $nonce = $request->get('nonce');
    $transactions_data = $order->getData(
      MicbGatewayServiceInterface::ORDER_DATA_TRANSACTIONS_KEY, []
    );
    $rrn = 'rrn';
    foreach ($transactions_data as $rrn_key => $transaction_data) {
      // Search RRN in stored transactions based on received NONCE.
      $transaction_nonce = $transaction_data[MicbGatewayServiceInterface::TRTYPE_AUTHORIZE]['NONCE'] ?? NULL;
      if ($transaction_nonce === $nonce && $rrn_key) {
        $rrn = $rrn_key;
        break;
      }
    }
    $payment = $this->entityTypeManager
      ->getStorage('commerce_payment')
      ->loadByProperties([
        'order_id' => $order->id(),
        'remote_id' => $rrn,
      ]);

    if ($payment) {
      $payment = reset($payment);
    }
    else {
      // Send client to a page to wait for bank to notify about payment status.
      $wait_url = Url::fromRoute('commerce_micb.wait',
        [
          'commerce_order' => $order->id(),
        ],
        [
          'absolute' => TRUE,
          'query' => ['nonce' => $request->get('nonce')],
        ]
      );
      throw new NeedsRedirectException($wait_url->toString());
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment_status = $payment->getState()->getId();
    if ($payment_status === 'authorization_voided') {
      // Cancelled or invalid payment, go back to checkout page.
      throw new PaymentGatewayException('Payment cancelled or declined');
    }
    elseif ($payment_status !== 'authorization' && $payment_status !== 'completed') {
      // Unexpected payment status, go back to checkout page.
      throw new PaymentGatewayException('Unexpected payment status ' . $payment_status);
    }

    // Successfull payment - do nothing to reach order completed page.
  }

  /**
   * Gets the authorize URL.
   *
   * @return string
   *   The authorize URL.
   */
  public function getGatewayAuthorizeUrl() {
    if ($this->getMode() === 'live') {
      return $this->configuration['live_gateway_url'];
    }
    else {
      return $this->configuration['test_gateway_url'];
    }
  }

  /**
   * Gets the capture URL.
   *
   * @return string
   *   The capture URL.
   */
  public function getCaptureUrl() {
    if ($this->getMode() === 'live') {
      return $this->configuration['live_gateway_url_complete'];
    }
    else {
      return $this->configuration['test_gateway_url_complete'];
    }
  }

  /**
   * Gets the reversal URL.
   *
   * @return string
   *   The reversal URL.
   */
  public function getReversalUrl() {
    if ($this->getMode() === 'live') {
      return $this->configuration['live_gateway_url_reversal'];
    }
    else {
      return $this->configuration['test_gateway_url_reversal'];
    }
  }

  /**
   * Gets Guzzle HTTP Client.
   *
   * @return \GuzzleHttp\ClientInterface
   *   Guzzle HTTP Client.
   */
  public function getHttpClient(): ClientInterface {
    $configuration = $this->getConfiguration();
    $options = [
      RequestOptions::DEBUG => FALSE,
      RequestOptions::HEADERS => [
        'Referer' => $configuration['merchant_url'],
      ],
    ];

    if (!empty($configuration['debug']) && !empty($configuration['debug_file'])) {
      $log = new Logger('micb_guzzle_request');
      $log->pushHandler(new StreamHandler($configuration['debug_file'], 100));
      $stack = HandlerStack::create();
      $stack->push(
        Middleware::log($log, new MessageFormatter(MessageFormatter::DEBUG))
      );
      $options['handler'] = $stack;
    }

    return new Client($options);
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    $this->assertPaymentState($payment, ['authorization']);
    // If specified, capture the exact amount.
    if ($amount) {
      $payment->setAmount($amount);
    }
    $configuration = $this->getConfiguration();

    $received_authorization_data = $this->micb->getTransactionData(
      $payment->getOrder(),
      $payment->getRemoteId(),
      MicbGatewayServiceInterface::TRTYPE_AUTHORIZE
    );

    $to_be_signed = [
      'TRTYPE' => MicbGatewayServiceInterface::TRTYPE_CAPTURE,
      'TERMINAL' => $received_authorization_data['TERMINAL'],
      'ORDER' => $received_authorization_data['ORDER'],
      'CURRENCY' => $received_authorization_data['CURRENCY'],
      'AMOUNT' => (float) $payment->getAmount()->getNumber(),
      'TIMESTAMP' => $this->micb->getTimeStamp(),
      'NONCE' => $this->micb->getNonce(),
      'RRN' => $received_authorization_data['RRN'],
      'INT_REF' => $received_authorization_data['INT_REF'],
    ];

    $rsa_priv_key = $this->micb->getPrivateKey(
      $configuration['private_key_path'],
      $configuration['private_key_password']
    );
    $hashed_data = $this->micb->generateHashedData(
      $this->micb->getKeyLength($rsa_priv_key),
      $to_be_signed
    );
    $capture_data = array_merge($to_be_signed, [
      'P_SIGN' => $this->micb->generatePsign($rsa_priv_key, $hashed_data),
      'MERCHANT' => $configuration['merchant_id'],
      'MERCH_GMT' => $this->micb->getGmt(),
    ]);

    try {
      /** @var \Psr\Http\Message\ResponseInterface $response_obj */
      $response_obj = $this->getHttpClient()
        ->request(
          'POST',
          $this->getCaptureUrl(),
          [
            RequestOptions::JSON => ['SALE_COMPLETION' => $capture_data],
          ]
        );

      $response = $response_obj->getBody()->__toString();

      if (!preg_match('/Referenced Transaction was successfully completed\./', $response)) {
        throw new \Exception('Invalid capture response: ' . $response);
      }
    }
    catch (\Exception $e) {
      throw new MicbException($e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment): void {
    $this->assertPaymentState($payment, ['authorization']);
    $this->refundPayment($payment);
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL): void {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded', 'authorization']);
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    $configuration = $this->getConfiguration();

    // @todo use capture data?
    $received_authorization_data = $this->micb->getTransactionData(
      $payment->getOrder(),
      $payment->getRemoteId(),
      MicbGatewayServiceInterface::TRTYPE_AUTHORIZE
    );

    $to_be_signed = [
      'TRTYPE' => MicbGatewayServiceInterface::TRTYPE_REVERSAL,
      'TERMINAL' => $received_authorization_data['TERMINAL'],
      'ORDER' => $received_authorization_data['ORDER'],
      'CURRENCY' => $amount->getCurrencyCode(),
      'AMOUNT' => (float) $amount->getNumber(),
      'TIMESTAMP' => $this->micb->getTimeStamp(),
      'NONCE' => $this->micb->getNonce(),
      'RRN' => $received_authorization_data['RRN'],
      'INT_REF' => $received_authorization_data['INT_REF'],
    ];

    $rsa_priv_key = $this->micb->getPrivateKey(
      $configuration['private_key_path'],
      $configuration['private_key_password']
    );
    $hashed_data = $this->micb->generateHashedData(
      $this->micb->getKeyLength($rsa_priv_key),
      $to_be_signed
    );
    $reversal_data = array_merge($to_be_signed, [
      'P_SIGN' => $this->micb->generatePsign($rsa_priv_key, $hashed_data),
      'MERCHANT' => $configuration['merchant_id'],
      'MERCH_GMT' => $this->micb->getGmt(),
    ]);

    try {
      /** @var \Psr\Http\Message\ResponseInterface $response_obj */
      $response_obj = $this->getHttpClient()
        ->request(
          'POST',
          $this->getReversalUrl(),
          [
            RequestOptions::JSON => ['REVERSAL' => $reversal_data],
          ]
        );

      $response = $response_obj->getBody()->__toString();

      if (!preg_match('/Referenced Transaction was successfully reversed\./', $response)) {
        throw new \Exception('Invalid reversal response: ' . $response);
      }
    }
    catch (\Exception $e) {
      throw new MicbException($e->getMessage());
    }

    // Change states in order to prevent further changes before IPN is received.
    if ($payment->getState()->getId() == 'authorization') {
      $payment->setState('authorization_voided');
    }
    else {
      $payment->setState('refunded');
    }
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    $received_data = $request->request->all();
    $successfull = $received_data['ACTION'] === '0';
    $remote_state_segment = 'TRTYPE-' . $received_data['TRTYPE'] . ':' .
      $received_data['ACTION'] . '-' . $received_data['TEXT'] . ';';

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $this->entityTypeManager->getStorage('commerce_order')
      ->load((int) $received_data['ORDER']);
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $existing_payment = $payment_storage->loadByProperties([
      'order_id' => $order->id(),
      'remote_id' => $received_data['RRN'],
    ]);

    $payment = NULL;
    if ($received_data['TRTYPE'] === MicbGatewayServiceInterface::TRTYPE_AUTHORIZE && empty($existing_payment)) {
      // Create payment for authorization IPN.
      /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
      $payment = $payment_storage->create([
        'state' => $successfull ? 'authorization' : 'authorization_voided',
        'payment_gateway' => $this->parentEntity->id(),
        'order_id' => $order->id(),
        'remote_id' => $received_data['RRN'] ?: '-',
        'authorized' => $successfull ? time() : NULL,
        'amount' => new Price($received_data['AMOUNT'], $received_data['CURRENCY']),
      ]);
    }
    elseif ($existing_payment) {
      /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
      $payment = reset($existing_payment);
    }

    if (!$payment instanceof PaymentInterface) {
      throw new MicbException('No payment detected/created for on Notify');
    }

    $message_params = [
      '@rrn' => $received_data['RRN'],
      '@payment' => '#' . $payment->id(),
      '@nonce' => $received_data['NONCE'],
      '@order' => '#' . $payment->getOrderId(),
    ];

    switch ($received_data['TRTYPE']) {
      case MicbGatewayServiceInterface::TRTYPE_AUTHORIZE:
        $logger_message = $successfull ? $this->t(
          'Authorized payment @payment rrn @rrn nonce @nonce order @order.',
          $message_params
        ) : NULL;
        break;

      case MicbGatewayServiceInterface::TRTYPE_CAPTURE:
        $payment->setState('completed');
        $capture_amount = new Price($received_data['AMOUNT'], $received_data['CURRENCY']);
        $payment->setAmount($capture_amount);
        $logger_message = $this->t(
          'Captured @amount payment @payment rrn @rrn nonce @nonce order @order.',
          array_merge(['@amount' => (string) $capture_amount], $message_params)
        );
        break;

      case MicbGatewayServiceInterface::TRTYPE_REVERSAL:
        $refunded_amount = new Price($received_data['AMOUNT'], $received_data['CURRENCY']);
        // Reversal is possible for both authorized and captured payments.
        if ($payment->getState()->getId() == 'authorization') {
          $payment->setState('authorization_voided');
        }
        elseif ($payment->getState()->getId() == 'completed') {
          // Multiple refunds are not accepted so partially_refunded is omitted.
          $payment->setState('refunded');
        }
        $payment->setRefundedAmount($refunded_amount);
        $logger_message = $this->t(
          'Refunded @amount payment @payment rrn @rrn nonce @nonce order @order.',
          array_merge(['@amount' => (string) $refunded_amount], $message_params)
        );
        break;

    }

    $payment->setRemoteState(
      trim($payment->getRemoteState() . ' ' . $remote_state_segment)
    );
    $payment->save();

    if (!empty($logger_message)) {
      $this->logger->notice($logger_message);
    }

    $this->micb->storeTransactionData($order, $received_data);

    // Initiate capture if requested by payment gateway configuration.
    if ($received_data['TRTYPE'] === MicbGatewayServiceInterface::TRTYPE_AUTHORIZE && empty($existing_payment)) {
      $intent = $this->getConfiguration()['intent'] ?? '';
      if ($successfull && $intent === 'capture') {
        $this->capturePayment($payment, $payment->getAmount());
      }
    }

    return new Response();
  }

}
