<?php

namespace Drupal\commerce_micb\PluginForm\OffsiteRedirect;

use Drupal\address\Repository\CountryRepository;
use Drupal\commerce_micb\Event\MicbEvents;
use Drupal\commerce_micb\Event\MicbSendAuthorizationDataEvent;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\commerce_micb\Exception\MicbException;
use Drupal\commerce_micb\MicbGatewayServiceInterface;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use GuzzleHttp\RequestOptions;
use Psr\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Payment offsite redirect form for Micb.
 */
class MicbPaymentOffsiteForm extends BasePaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The commerce micb logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The Micb gateway service.
   *
   * @var \Drupal\commerce_micb\MicbGatewayServiceInterface
   */
  protected MicbGatewayServiceInterface $micb;

  /**
   * The countries repository service.
   *
   * @var \Drupal\address\Repository\CountryRepository
   */
  protected CountryRepository $countryRepository;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * Constructs a new PaymentOffsiteForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager object.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory object.
   * @param \Drupal\commerce_micb\MicbGatewayServiceInterface $micb_gateway
   *   Micb gateway service.
   * @param \Drupal\address\Repository\CountryRepository $country_repository
   *   The country repository.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  final public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    LoggerChannelFactoryInterface $logger_factory,
    MicbGatewayServiceInterface $micb_gateway,
    CountryRepository $country_repository,
    EventDispatcherInterface $event_dispatcher,
    Token $token,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->logger = $logger_factory->get('commerce_micb');
    $this->micb = $micb_gateway;
    $this->countryRepository = $country_repository;
    $this->eventDispatcher = $event_dispatcher;
    $this->token = $token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('logger.factory'),
      $container->get('commerce_micb.gateway'),
      $container->get('address.country_repository'),
      $container->get('event_dispatcher'),
      $container->get('token'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();
    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $payment->getPaymentGateway();
    /** @var \Drupal\commerce_micb\Plugin\Commerce\PaymentGateway\MicbOffsiteRedirect $payment_gateway_plugin */
    $payment_gateway_plugin = $payment_gateway->getPlugin();
    $configuration = $payment_gateway_plugin->getConfiguration();

    // Build data array to be sent to micb gateway.
    $to_be_signed = [
      'TRTYPE' => '0',
      'TERMINAL' => $configuration['terminal'],
      'ORDER' => sprintf('%06d', $order->id()),
      'CURRENCY' => $payment->getAmount()->getCurrencyCode(),
      'AMOUNT' => (float) $payment->getAmount()->getNumber(),
      'TIMESTAMP' => $this->micb->getTimeStamp(),
      'NONCE' => $this->micb->getNonce(),
      'DESC' => (string) $this->t('Order #@id', ['@id' => $payment->getOrderId()]),
      'BACKREF' => '',
      'CHECKOUT_HASH' => '',
    ];
    $to_be_signed['BACKREF'] = Url::fromRoute(
      'commerce_payment.checkout.return',
      [
        'commerce_order' => $order->id(),
        'step' => 'payment',
      ],
      [
        'absolute' => TRUE,
        'query' => ['nonce' => $to_be_signed['NONCE']],
      ]
    )->toString();

    $authorization_data = array_merge(
      $to_be_signed,
      [
        'P_SIGN' => '',
        'MERCH_URL' => $configuration['merchant_url'],
        'MERCHANT' => $configuration['merchant_id'],
        'MERCH_NAME' => $configuration['merchant_name'],
        'MERCH_GMT' => $this->micb->getGmt(),
        'COUNTRY' => $configuration['country'],
        'EMAIL' => $configuration['merchant_email'],
        'LANG' => $this->languageManager->getCurrentLanguage()->getId(),
      ]
    );

    $authorization_data['LINE_ITEMS'] = [];
    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    foreach ($order->getItems() as $order_item) {
      $authorization_data['LINE_ITEMS'][] = [
        'CATEGORY_ID' => (int) $configuration['store_category'],
        'PRODUCT_NAME' => $order_item->getTitle(),
        'PRICE' => (float) $order_item->getTotalPrice()->getNumber(),
        'QUANTITY' => (float) $order_item->getQuantity(),
        'BRAND' => $configuration['merchant_name'],
        'MERCHANT_PRODUCT_ID' => $order_item->id(),
      ];
    }
    // Shipping and other adjustments.
    foreach ($order->getAdjustments() as $adjustment) {
      if ($adjustment->isIncluded() || $adjustment->isNegative()) {
        // @todo negative & zero adjustments?
        continue;
      }
      $authorization_data['LINE_ITEMS'][] = [
        'CATEGORY_ID' => (int) $configuration['store_category'],
        'PRODUCT_NAME' => $adjustment->getLabel(),
        'PRICE' => (float) $adjustment->getAmount()->getNumber(),
        'QUANTITY' => 1.0,
        'BRAND' => $configuration['merchant_name'],
        'MERCHANT_PRODUCT_ID' => $adjustment->getSourceId(),
      ];
    }

    $customer = $order->getCustomer();
    $profiles = $order->collectProfiles();
    $billing_profile = $profiles['billing'] ?? NULL;
    /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $billing_address */
    $billing_address = $billing_profile->address[0] ?? NULL;
    $shipping_profile = $profiles['shipping'] ?? NULL;
    /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $shipping_address */
    $shipping_address = $shipping_profile->address[0] ?? NULL;

    $authorization_data['CUSTOMER'] = [
      'EMAIL' => $order->getEmail(),
      'EMAIL_VERIFIED' => $customer->id() > 0,
      'FIRST_NAME' => $billing_address ? $billing_address->getGivenName() : $customer->getAccountName(),
      'LAST_NAME' => $billing_address ? $billing_address->getFamilyName() : $customer->getAccountName(),
      'ID' => (string) $customer->id(),
      'CREATED_AT' => date('YmdHi', $customer->getCreatedTime()),
      'PHONE' => 'No-Phone',
      'PHONE_VERIFIED' => FALSE,
      // 'BDATE' => '19900101',
      // 'SEX' => 'u',
    ];

    $authorization_data['BILLING_ADDRESS'] = [
      'FIRST_NAME' => $billing_address ? $billing_address->getGivenName() : $customer->getAccountName(),
      'LAST_NAME' => $billing_address ? $billing_address->getFamilyName() : $customer->getAccountName(),
      'ZIP_CODE' => $billing_address ? $billing_address->getPostalCode() : '',
      'COUNTRY' => '',
      'COUNTRY_CODE' => '',
      'CITY_NAME' => $billing_address ? $billing_address->getLocality() : '',
      'PROVINCE' => $billing_address ? $billing_address->getAdministrativeArea() : '',
      'STREET' => $billing_address ? $billing_address->getAddressLine1() : '',
      'STREET_NR' => $billing_address ? $billing_address->getAddressLine2() : '',
      'ADDRESS_VERIFIED' => FALSE,
      'PHONE' => 'No-Phone',
    ];
    if ($billing_address && $billing_address->getCountryCode()) {
      try {
        $country = $this->countryRepository
          ->get($billing_address->getCountryCode());
        $authorization_data['BILLING_ADDRESS']['COUNTRY'] = $country->getName();
        $authorization_data['BILLING_ADDRESS']['COUNTRY_CODE'] = $country->getNumericCode();
      }
      catch (\Exception $e) {
        // Ignore.
      }
    }
    if (empty($authorization_data['BILLING_ADDRESS']['PROVINCE'])) {
      $authorization_data['BILLING_ADDRESS']['PROVINCE'] = 'Missing';
    }

    $authorization_data['SHIPPING_ADDRESS'] = NULL;
    if (!empty($configuration['send_shipping_info'])) {
      if (!$order->hasField('shipments') || $order->get('shipments')->isEmpty() || !$shipping_profile) {
        throw new MicbException('Shipping adres is required by plugin configuration but is missing in the order.');
      }
      $authorization_data['SHIPPING_ADDRESS'] = [
        'FIRST_NAME' => $shipping_address ? $shipping_address->getGivenName() : $customer->getAccountName(),
        'LAST_NAME' => $shipping_address ? $shipping_address->getFamilyName() : $customer->getAccountName(),
        'ZIP_CODE' => $shipping_address ? $shipping_address->getPostalCode() : '',
        'COUNTRY' => '',
        'COUNTRY_CODE' => '',
        'CITY_NAME' => $shipping_address ? $shipping_address->getLocality() : '',
        'PROVINCE' => $shipping_address ? $shipping_address->getAdministrativeArea() : '',
        'STREET' => $shipping_address ? $shipping_address->getAddressLine1() : '',
        'STREET_NR' => $shipping_address ? $shipping_address->getAddressLine2() : '',
        'ADDRESS_VERIFIED' => FALSE,
        'PHONE' => 'No-Phone',
      ];
      if ($shipping_address && $shipping_address->getCountryCode()) {
        try {
          $country = $this->countryRepository
            ->get($shipping_address->getCountryCode());
          $authorization_data['SHIPPING_ADDRESS']['COUNTRY'] = $country->getName();
          $authorization_data['SHIPPING_ADDRESS']['COUNTRY_CODE'] = $country->getNumericCode();
        }
        catch (\Exception $e) {
          // Ignore.
        }
      }
    }
    if (empty($authorization_data['SHIPPING_ADDRESS']) && !empty($configuration['send_shipping_info'])) {
      // @todo remove this or remove exception from before.
      $authorization_data['SHIPPING_ADDRESS'] = $authorization_data['BILLING_ADDRESS'];
    }

    $authorization_data['CLIENT_DETAILS'] = [
      'USER_IP' => $order->getIpAddress(),
      'SOURCE' => $this->micb->getClientSource(),
    ];

    // Replace some values by configured tokens.
    if (!empty($configuration['data_tokens'])) {
      $this->replaceAuthorizationDataTokens(
        $authorization_data,
        $configuration['data_tokens']
      );
      // Special case for all authorization line items.
      if (!empty($configuration['data_tokens']['LINE_ITEMS']['_ALL_'])) {
        foreach (array_keys($authorization_data['LINE_ITEMS']) as $index) {
          $this->replaceAuthorizationDataTokens(
            $authorization_data['LINE_ITEMS'][$index],
            array_map(function ($v) use ($index) {
              return str_replace(['_ALL_', '_NR_'], $index, $v);
            }, $configuration['data_tokens']['LINE_ITEMS']['_ALL_'])
          );
        }
      }
      // Special case for all order line items.
      if (!empty($configuration['data_tokens']['LINE_ITEMS']['_NR_'])) {
        foreach ($order->getItems() as $index => $order_item) {
          $this->replaceAuthorizationDataTokens(
            $authorization_data['LINE_ITEMS'][$index],
            array_map(function ($v) use ($index) {
              return str_replace('_NR_', $index, $v);
            }, $configuration['data_tokens']['LINE_ITEMS']['_NR_'])
          );
        }
      }
    }

    // Allow other modules to update authorization data before is sent to micb.
    $event = new MicbSendAuthorizationDataEvent($authorization_data, $payment);
    $this->eventDispatcher
      ->dispatch($event, MicbEvents::SEND_AUTHORIZATION_DATA);
    $authorization_data = $event->getAuthorizationData();

    $checkout_hash_str = '';
    $hash_keys = [
      'BILLING_ADDRESS',
      'CLIENT_DETAILS',
      'CUSTOMER',
      'LINE_ITEMS',
      'SHIPPING_ADDRESS',
    ];
    foreach ($hash_keys as $key) {
      $checkout_hash_str .= md5(
        $this->micb->getDataHashableString($authorization_data[$key])
      );
    }
    $authorization_data['CHECKOUT_HASH'] = md5($checkout_hash_str);
    $to_be_signed['CHECKOUT_HASH'] = $authorization_data['CHECKOUT_HASH'];

    $rsa_priv_key = $this->micb
      ->getPrivateKey(
        $configuration['private_key_path'],
        $configuration['private_key_password']
      );
    $hashed_data = $this->micb
      ->generateHashedData(
          $this->micb->getKeyLength($rsa_priv_key),
          $to_be_signed
      );
    $authorization_data['P_SIGN'] = $this->micb
      ->generatePsign($rsa_priv_key, $hashed_data);

    $redirect_url = NULL;
    try {
      /** @var \Psr\Http\Message\ResponseInterface $response_obj */
      $response_obj = $payment_gateway_plugin->getHttpClient()
        ->request(
          'POST',
          $payment_gateway_plugin->getGatewayAuthorizeUrl(),
          [
            RequestOptions::JSON => ['CHECKOUT' => $authorization_data],
          ]
        );

      $response = Json::decode($response_obj->getBody()->__toString());

      if (!isset($response['ResponseCode'])) {
        throw new MicbException('Invalid authorization response.');
      }
      elseif ($response['ResponseCode'] !== MicbGatewayServiceInterface::RESPONSE_CODE_OK) {
        throw new MicbException(sprintf(
          'MICB code: %s error: %s',
          $response['ResponseCode'],
          $response['ErrorMessage'] ?? 'Missing remote error'
        ));
      }
      elseif (!isset($response['ResponseData']['PaymentLink'])) {
        throw new MicbException('Empty PaymentLink for successfull authorization.');
      }
      else {
        $redirect_url = $response['ResponseData']['PaymentLink'];
      }
    }
    catch (\Exception $e) {
      throw new MicbException($e->getMessage());
    }

    $this->logger->notice($this->t(
      'Obtained payment link @link for order @order nonce @nonce.',
      [
        '@link' => $redirect_url,
        '@nonce' => $to_be_signed['NONCE'],
        '@order' => $payment->getOrderId(),
      ]
    ));

    return $this->buildRedirectForm($form, $form_state, $redirect_url, []);
  }

  /**
   * Replace elements in authorization data by using keys and tokens.
   *
   * @param array $data
   *   Authorization data array.
   * @param array $tokens
   *   Array with tokens to replace in data array.
   */
  private function replaceAuthorizationDataTokens(&$data, $tokens) {
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;

    foreach ($tokens as $key => $token) {
      if (!isset($data[$key])) {
        continue;
      }
      if (is_array($token)) {
        $this->replaceAuthorizationDataTokens($data[$key], $token);
      }
      else {
        $data[$key] = $this->token
          ->replace($token, [
            'commerce_payment' => $payment,
            'commerce_order' => $payment->getOrder(),
          ], [
            'clear' => TRUE,
          ]
        );
      }
    }
  }

}
