<?php

namespace Drupal\commerce_moyasar\PluginForm;

use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm;
use Drupal\commerce_price\MinorUnitsConverter;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Moyasar plugin form.
 */
class MoyasarForm extends BasePaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The minor units converter.
   *
   * @var \Drupal\commerce_price\MinorUnitsConverter
   */
  protected $minorUnitsConverter;

  /**
   * Constructs a new MoyasarForm object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\commerce_price\MinorUnitsConverter $minor_units_converter
   *   The minor units converter.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   */
  public function __construct(LanguageManagerInterface $language_manager, MinorUnitsConverter $minor_units_converter) {
    $this->languageManager = $language_manager;
    $this->minorUnitsConverter = $minor_units_converter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('commerce_price.minor_units_converter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();

    /** @var \Drupal\commerce_moyasar\Plugin\Commerce\PaymentGateway\Moyasar $moyasar_gateway */
    $moyasar_gateway = $payment->getPaymentGateway()->getPlugin();

    $payment_amount = $order->getTotalPrice();
    $payment->setAmount($payment_amount);

    $methods = [];
    foreach ($moyasar_gateway->getConfiguration()['methods'] as $key => $value) {
      if ($value) {
        $methods[] = $key;
      }
    }
    $supported_netwroks = [];
    foreach ($moyasar_gateway->getConfiguration()['supported_networks'] as $key => $value) {
      if ($value) {
        $supported_netwroks[] = $key;
      }
    }
    $js_settings = [
      'amount' => $this->minorUnitsConverter->toMinorUnits($payment_amount),
      'currency' => $payment_amount->getCurrencyCode(),
      'description' => $this->t('Order ID @order_id', ['@order_id' => $order->id()]),
      'publishable_api_key' => $moyasar_gateway->getConfiguration()['publishable_api_key'],
      'callback_url' => $form['#return_url'],
      'methods' => ['creditcard'],
      'language' => $this->languageManager->getCurrentLanguage()->getId(),
      'metadata' => ['order_id' => $order->id()],
      'methods' => $methods,
      'supported_netwroks' => $supported_netwroks,
      'reuse_payment_method' => $moyasar_gateway->getConfiguration()['reuse_payment_method'],
    ];

    $form['#attached']['drupalSettings']['commerce_moyasar'] = $js_settings;
    $form['#attached']['library'][] = 'commerce_moyasar/init';

    $form['cards'] = [
      '#type' => 'hidden',
      // Plugin forms are embedded using #process, so it's too late to attach
      // another #process to $form itself, it must be on a sub-element.
      '#process' => [
        [get_class($this), 'processMoyasarForm'],
      ],
      '#action' => $form['#return_url'],
      '#cancel_url' => $form['#cancel_url'],
    ];

    return $form;
  }

  /**
   * Prepares the complete form in order to work with Moyasar.
   *
   * Sets the form #action, adds a class for the JS to target.
   * Workaround for buildConfigurationForm() not receiving $complete_form.
   *
   * @param array $element
   *   The form element whose value is being processed.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed form element.
   */
  public static function processMoyasarForm(array $element, FormStateInterface $form_state, array &$complete_form) {
    $complete_form['#action'] = $element['#action'];
    $complete_form['#attributes']['class'][] = 'paymentWidgets';
    $complete_form['#attributes']['data-brands'] = $element['#value'];

    if (!empty($element['#amount'])) {
      $complete_form['#prefix'] = $element['#amount'];
    }

    // As the Moyasar fully replaces the HTML form, we need to place the
    // cancel link outside the form as suffix.
    $complete_form['#suffix'] = Link::fromTextAndUrl(t('Cancel'), Url::fromUri($element['#cancel_url']))->toString();

    return $element;
  }

}
