<?php

namespace Drupal\commerce_moyasar\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Exception\InvalidResponseException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsStoredPaymentMethodsInterface;
use Drupal\commerce_price\Price;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Moyasar payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "moyasar_payment",
 *   label = @Translation("Moyasar Payment"),
 *   display_label = @Translation("Moyasar Payment"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_moyasar\PluginForm\MoyasarForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "mastercard", "visa", "mada",
 *   },
 *   requires_billing_information = FALSE,
 * )
 */
class Moyasar extends OffsitePaymentGatewayBase implements MoyasarInterface, SupportsStoredPaymentMethodsInterface {

  /**
   * The http client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The payment storage.
   *
   * @var \Drupal\commerce_payment\PaymentStorageInterface
   */
  protected $paymentStorage;

  /**
   * The price rounder.
   *
   * @var \Drupal\commerce_price\RounderInterface
   */
  protected $rounder;

  /**
   * The minor units converter.
   *
   * @var \Drupal\commerce_price\MinorUnitsConverter
   */
  protected $minorUnitsConverter;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuidService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->httpClient = $container->get('http_client');
    $instance->paymentStorage = $instance->entityTypeManager->getStorage('commerce_payment');
    $instance->rounder = $container->get('commerce_price.rounder');
    $instance->minorUnitsConverter = $container->get('commerce_price.minor_units_converter');
    $instance->uuidService = $container->get('uuid');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'publishable_api_key' => '',
      'secret_api_key' => '',
      'methods' => [],
      'supported_networks' => [],
      'reuse_payment_method' => TRUE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['publishable_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Publishable API key'),
      '#default_value' => $this->configuration['publishable_api_key'],
      '#required' => TRUE,
    ];
    $form['secret_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret API key'),
      '#default_value' => $this->configuration['secret_api_key'],
      '#required' => TRUE,
    ];
    $form['methods'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed methods'),
      '#options' => [
        'creditcard' => 'Credit Card',
        'applepay' => 'Apple Pay',
        'stcpay' => 'STC Pay',
      ],
      '#default_value' => $this->configuration['methods'],
      '#required' => TRUE,
    ];
    $form['supported_networks'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Supported networks'),
      '#options' => [
        'visa' => 'Visa',
        'mastercard' => 'Mastercard',
        'amex' => 'American Express',
        'mada' => 'Mada',
      ],
      '#default_value' => $this->configuration['supported_networks'],
      '#states' => [
        'visible' => [
          ':input[name="configuration[moyasar_payment][methods][creditcard]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['reuse_payment_method'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow reusing payment method'),
      '#default_value' => $this->configuration['reuse_payment_method'],
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValue($form['#parents']);
    if (in_array('creditcard', $values['methods'])) {
      $one_selected = FALSE;
      foreach ($values['supported_networks'] as $key => $value) {
        if ($value !== 0) {
          $one_selected = TRUE;
          break;
        }
      }
      if (!$one_selected) {
        $form_state->setErrorByName('supported_networks', $this->t('At lease one network should be selected.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['publishable_api_key'] = $values['publishable_api_key'];
      $this->configuration['secret_api_key'] = $values['secret_api_key'];
      $this->configuration['methods'] = $values['methods'];
      if (!in_array('creditcard', $values['methods'])) {
        $this->configuration['supported_networks'] = [
          'visa' => 0,
          'mastercard' => 0,
          'amex' => 0,
          'mada' => 0,
        ];
      }
      else {
        $this->configuration['supported_networks'] = $values['supported_networks'];
      }
      $this->configuration['reuse_payment_method'] = $values['reuse_payment_method'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, Request $request) {
    $payment_method->save();
    return $payment_method;
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    $token = $payment_method->getRemoteId();
    $this->sendRequest("tokens/$token", "DELETE");
    $payment_method->delete();
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    $this->assertPaymentMethod($payment_method);

    // Perform the create payment request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // Remember to take into account $capture when performing the request.
    $payment_method_token = $payment_method->getRemoteId();
    $order = $payment->getOrder();
    $payment_amount = $order->getTotalPrice();
    $payment_uuid = $this->uuidService->generate();
    $params = [
      'given_id' => $payment_uuid,
      'amount' => $this->minorUnitsConverter->toMinorUnits($payment_amount),
      'currency' => $payment_amount->getCurrencyCode(),
      'description' => $this->t('Order ID @order_id', ['@order_id' => $order->id()]),
      'source' => [
        'type' => 'token',
        'token' => $payment_method_token,
        'manual' => !$capture,
      ],
      'metadata' => ['order_id' => $order->id()],
    ];

    $payment_response = $this->sendRequest('payments', 'POST', $params);
    if ($payment_response['status']) {
      $payment->setRemoteId($payment_response['data']->id);
      $payment->setRemoteState($payment_response['data']->status);
      if ($payment_response['data']->status == 'paid' || $payment_response['data']->status == 'captured') {
        $capture_transition = $payment->getState()->getWorkflow()->getTransition('authorize_capture');
        $payment->getState()->applyTransition($capture_transition);
      }
      elseif ($payment_response['data']->status == 'authorized') {
        $capture_transition = $payment->getState()->getWorkflow()->getTransition('authorize');
        $payment->getState()->applyTransition($capture_transition);
      }
      $payment->save();
    }
    else {
      throw new InvalidResponseException($json_response['data']->message);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['authorization']);
    // If not specified, capture the entire amount.
    $amount = $amount ?: $payment->getAmount();

    $params = [
      'amount' => $this->minorUnitsConverter->toMinorUnits($amount),
    ];

    $remote_id = $payment->getRemoteId();
    $payment_response = $this->sendRequest("payments/$remote_id/capture", 'POST', $params);

    if ($payment_response['status']) {
      $payment->setState('completed');
      $payment->setAmount($amount);
      $payment->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    $this->assertPaymentState($payment, ['authorization']);

    $remote_id = $payment->getRemoteId();
    $payment_response = $this->sendRequest("payments/$remote_id/void", 'POST', $params);

    if ($payment_response['status']) {
      $payment->setState('authorization_voided');
      $payment->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    parent::onReturn($order, $request);

    $payment_id = $request->query->get('id');
    $payment_response = $this->sendRequest("payments/$payment_id", 'GET');

    if ($payment_response['status']) {
      if ($payment_response['data']->status == 'paid' || $payment_response['data']->status == 'authorized' || $payment_response['data']->status == 'captured') {
        $payment_status = $payment_response['data']->status == 'authorized' ? 'authorization' : 'completed';
        $payment = $this->paymentStorage->create([
          'state' => $payment_status,
          'amount' => $this->minorUnitsConverter->fromMinorUnits($payment_response['data']->amount, $payment_response['data']->currency),
          'payment_gateway' => $this->parentEntity->id(),
          'order_id' => $order->id(),
          'remote_id' => $payment_id,
          'remote_state' => $payment_response['data']->status,
        ]);

        $payment->save();

        if ($this->configuration['reuse_payment_method'] && $payment_response['data']->source->type == 'creditcard') {
          $payment_method_storage = $this->entityTypeManager->getStorage('commerce_payment_method');
          // assert($payment_method_storage instanceof PaymentMethodStorageInterface);.
          $payment_method = $payment_method_storage->createForCustomer(
            'credit_card',
            $this->parentEntity->id(),
            $order->getCustomerId(),
            $order->getBillingProfile()
          );
          $payment_method->card_type = $payment_response['data']->source->company == 'master' ? 'mastercard' : $payment_response['data']->source->company;
          $payment_method->card_number = $payment_response['data']->source->number;
          $payment_method->setRemoteId($payment_response['data']->source->token);
          $payment_method->save();
          $order->set('payment_method', $payment_method->id());
        }
      }
    }

  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    // Perform the refund request here, throw an exception if it fails.
    $remote_id = $payment->getRemoteId();
    $params = [
      'amount' => $this->minorUnitsConverter->toMinorUnits($amount),
    ];
    $json_response = $this->sendRequest("payments/$remote_id/refund", 'POST', $params);

    if ($json_response['status']) {
      // Determine whether payment has been fully or partially refunded.
      $old_refunded_amount = $payment->getRefundedAmount();
      $new_refunded_amount = $old_refunded_amount->add($amount);
      if ($new_refunded_amount->lessThan($payment->getAmount())) {
        $payment->setState('partially_refunded');
      }
      else {
        $payment->setState('refunded');
      }

      $payment->setRefundedAmount($new_refunded_amount);
      $payment->save();
    }
    else {
      throw new InvalidRequestException($this->t('Cannot refund Moyasar payment: ID does not exist.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function sendRequest($resource_path, $method, $params = []) {
    $base_url = 'https://api.moyasar.com/v1/';
    $url = $base_url . $resource_path;

    $options = [
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'auth' => [
        $this->configuration['secret_api_key'],
        '',
      ],
    ];

    if ($params && $method == 'POST') {
      $options['body'] = json_encode($params);
    }
    elseif ($params && $method == 'GET') {
      $query_str = UrlHelper::buildQuery($params);
      $url .= '?' . $query_str;
    }

    try {
      $request = $this->httpClient->request($method, $url, $options);
      $response_status = $request->getStatusCode();
      $data = json_decode($request->getBody()->getContents());

      if ($response_status == 200 || $response_status == 201 || $response_status == 204) {
        return [
          'status' => TRUE,
          'data' => $data,
        ];
      }
      else {
        return [
          'status' => FALSE,
          'data' => $data,
        ];
      }
    }
    catch (\Exception $ex) {
      throw new InvalidResponseException($this->t('Error occurred on calling the specified Moyasar resource path @resource_path with message: @msg', ['@resource_path' => $resource_path, '@msg' => $ex->getMessage()]));
    }

  }

}
