<?php

declare(strict_types=1);

namespace Drupal\commerce_nbg_redirect\Controller;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Returns responses for Commerce National Bank of Greece (Redirect) routes.
 */
final class CommerceNbgRedirectController implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $currentRequest;

  /**
   * The order storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $orderStorage;

  /**
   * The payment gateway storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $paymentGatewayStorage;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * Constructs a new CommerceNbgRedirectController object.
   *
   * @param \Symfony\Component\HttpFoundation\Request $current_request
   *   The current request.
   * @param \Drupal\Core\Entity\EntityStorageInterface $order_storage
   *   The order storage.
   * @param \Drupal\Core\Entity\EntityStorageInterface $payment_gateway_storage
   *   The payment gateway storage.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(Request $current_request, EntityStorageInterface $order_storage, EntityStorageInterface $payment_gateway_storage, RendererInterface $renderer) {
    $this->currentRequest = $current_request;
    $this->orderStorage = $order_storage;
    $this->paymentGatewayStorage = $payment_gateway_storage;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('entity_type.manager')->getStorage('commerce_order'),
      $container->get('entity_type.manager')->getStorage('commerce_payment_gateway'),
      $container->get('renderer')
    );
  }

  /**
   * Handles the return URL callback from GlobalPayments.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response containing the redirect page.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   When signature validation fails.
   */
  public function returnUrl(): Response {
    $gp_signature = $this->currentRequest->headers->get('X-GP-Signature');

    $raw_input = trim($this->currentRequest->getContent());
    $input = json_decode($raw_input, TRUE);

    $reference = explode('_', $input['reference']);
    $order_id = $reference[1];

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $this->orderStorage->load($order_id);

    // Validate the request signature.
    if ($this->validateRequest($raw_input, $gp_signature, $reference[2]) === FALSE) {
      throw new AccessDeniedHttpException();
    }

    // Store data on order, will be used by onReturn() to create the payment entity.
    $order->setData('commerce_nbg_payment', [
      'result_code' => $input['action']['result_code'],
      'transaction_id' => $input['id'],
      'status' => $input['status'],
    ]);
    $order->save();

    $redirect_url = Url::fromRoute('commerce_payment.checkout.return', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();

    $build = [
      '#theme' => 'commerce_nbg_redirect_return',
      '#redirect_url' => $redirect_url,
    ];
    $output = (string) $this->renderer->renderInIsolation($build);

    return new Response($output);
  }

  /**
   * Validates the request signature.
   *
   * @param string $raw_input
   *   The raw JSON input.
   * @param string|null $gp_signature
   *   The signature from X-GP-Signature header.
   * @param string $plugin_id
   *   The payment gateway plugin ID.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  private function validateRequest(string $raw_input, ?string $gp_signature, string $plugin_id): bool {
    if (!$raw_input || !$gp_signature) {
      return FALSE;
    }

    $parsed_input = json_decode($raw_input, TRUE);
    if (!$parsed_input) {
      return FALSE;
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $this->paymentGatewayStorage->load($plugin_id);
    $app_key = $payment_gateway->getPluginConfiguration()['app_key'];
    $minified_input = json_encode($parsed_input, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);

    return hash("sha512", $minified_input . $app_key) === $gp_signature;
  }

}
