<?php

namespace Drupal\commerce_nbg_redirect\PluginForm\OffsiteRedirect;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm;
use Drupal\Component\Utility\Random;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use GlobalPayments\Api\Builders\HPPBuilder;
use GlobalPayments\Api\Entities\Enums\ChallengeRequestIndicator;
use GlobalPayments\Api\Entities\Enums\Channel;
use GlobalPayments\Api\Entities\Enums\Environment;
use GlobalPayments\Api\Entities\Enums\ExemptStatus;
use GlobalPayments\Api\Entities\PayerDetails;
use GlobalPayments\Api\ServiceConfigs\Gateways\GpApiConfig;
use GlobalPayments\Api\ServicesContainer;

/**
 * Provides the redirect form for NBG payment gateway.
 */
class PaymentOffsiteForm extends BasePaymentOffsiteForm {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;

    $redirect_url = $this->createPayByLinkUrl($payment, $form);

    $form = $this->buildRedirectForm($form, $form_state, $redirect_url, []);

    return $form;
  }

  /**
   * Creates a GlobalPayments Hosted Payment Page URL.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment entity.
   * @param array $form
   *   The form array.
   *
   * @return string
   *   The hosted payment page URL.
   */
  protected function createPayByLinkUrl(PaymentInterface $payment, array $form): string {
    /** @var OrderInterface $order */
    $order = $payment->getOrder();

    /** @var \Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface $payment_gateway_plugin */
    $payment_gateway_plugin = $payment->getPaymentGateway()->getPlugin();
    $gateway_plugin_config = $payment_gateway_plugin->getConfiguration();

    // Configure GP API and ServicesContainer.
    $config = new GpApiConfig();
    $config->appId = $gateway_plugin_config['app_id'];
    $config->appKey = $gateway_plugin_config['app_key'];
    $config->country = $gateway_plugin_config['country_code'];
    $config->channel = Channel::CardNotPresent;
    $config->environment = $gateway_plugin_config['mode'] === 'live' ? Environment::PRODUCTION : Environment::TEST;
    ServicesContainer::configureService($config);

    // Required payer information
    $payer = new PayerDetails();
    $payer->firstName = $order->getBillingProfile()->get('address')->given_name;
    $payer->lastName = $order->getBillingProfile()->get('address')->family_name;
    $payer->email = $order->getEmail();
    $payer->status = 'NEW';

    // Transaction details
    ['number' => $amount, 'currency_code' => $currency_code] = $order->getTotalPrice()->toArray();
    $name = 'Order #' . $order->id();
    $description = 'Payment for order #' . $order->id();
    $random = new Random();
    $reference = 'Hosted Payment Page transaction ' . $random->name(16);
    $order_reference = 'order_' . $order->id() . '_' . $payment_gateway_plugin->getPluginId() . '_' . $random->name(16);
    $return_url = Url::fromRoute('commerce_nbg_redirect.return_url', [], ['absolute' => TRUE])->toString();

    // Create hosted payment page link
    $response = HPPBuilder::create()
      ->withName($name)
      ->withDescription($description)
      ->withReference($reference)
      ->withOrderReference($order_reference)
      ->withAmount($amount)
      ->withCurrency($currency_code)
      ->withPayer($payer)
      ->withNotifications(
        $return_url,
        $return_url,
        $return_url,
      )
      ->withTransactionConfig(country: $gateway_plugin_config['country_code'])
      ->withAuthentication(
        ChallengeRequestIndicator::CHALLENGE_PREFERRED,
        ExemptStatus::LOW_VALUE,
        TRUE
      )
      ->withDigitalWallets(["googlepay", "applepay"])
      ->execute();

    return $response->payByLinkResponse->url;
  }

}
