<?php

namespace Drupal\commerce_nbg_redirect\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the National Bank of Greece (NBG) Redirect payment gateway.
 */
#[CommercePaymentGateway(
  id: "nbg_redirect",
  label: new TranslatableMarkup("National Bank of Greece (Redirect)"),
  display_label: new TranslatableMarkup("NBG Payment"),
  modes: [
    "test" => new TranslatableMarkup("Test"),
    "live" => new TranslatableMarkup("Live"),
  ],
  forms: [
    "offsite-payment" => "Drupal\commerce_nbg_redirect\PluginForm\OffsiteRedirect\PaymentOffsiteForm",
  ],
  payment_method_types: ["credit_card"],
  credit_card_types: [
    "amex",
    "dinersclub",
    "discover",
    "jcb",
    "maestro",
    "mastercard",
    "visa",
    "unionpay",
  ],
  requires_billing_information: FALSE,
)]
class OffsiteRedirect extends OffsitePaymentGatewayBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'merchant_id' => '',
      'app_id' => '',
      'app_key' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['country_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant Country Code'),
      '#description' => $this->t('Your merchant country code'),
      '#default_value' => $this->configuration['country_code'] ?? 'GR',
      '#required' => TRUE,
    ];

    $form['merchant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Merchant ID'),
      '#description' => $this->t('Your merchant ID'),
      '#default_value' => $this->configuration['merchant_id'],
      '#required' => TRUE,
    ];

    $form['app_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App ID'),
      '#description' => $this->t('Your application ID for authentication'),
      '#default_value' => $this->configuration['app_id'],
      '#required' => TRUE,
    ];

    $form['app_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('App Key'),
      '#description' => $this->t('Your application key for authentication'),
      '#default_value' => $this->configuration['app_key'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['country_code'] = $values['country_code'];
      $this->configuration['merchant_id'] = $values['merchant_id'];
      $this->configuration['app_id'] = $values['app_id'];
      $this->configuration['app_key'] = $values['app_key'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $data = $order->getData('commerce_nbg_payment');
    if (!isset($data['result_code']) || $data['result_code'] !== 'SUCCESS') {
      throw new PaymentGatewayException();
    }

    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->create([
      'state' => 'completed',
      'amount' => $order->getBalance(),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => $data['transaction_id'],
      'remote_state' => $data['status'],
    ]);
    $payment->save();
  }

}
