<?php

namespace Drupal\commerce_nexi\Controller;

use Drupal\commerce\Response\NeedsRedirectException;
use Drupal\commerce\Utility\Error;
use Drupal\commerce_checkout\CheckoutOrderManagerInterface;
use Drupal\commerce_nexi\Plugin\Commerce\PaymentGateway\NexiGateway;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Controller\PaymentCheckoutController;
use Drupal\commerce_payment\Event\FailedPaymentEvent;
use Drupal\commerce_payment\Event\PaymentEvents;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\Core\Access\AccessException;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides checkout endpoints for off-site payments.
 */
class NexiCheckoutController extends PaymentCheckoutController implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new NexiCheckoutController object.
   */
  public function __construct(CheckoutOrderManagerInterface $checkout_order_manager, MessengerInterface $messenger, LoggerInterface $logger, EntityTypeManagerInterface $entity_type_manager, EventDispatcherInterface $event_dispatcher, RouteMatchInterface $routeMatch) {
    parent::__construct($checkout_order_manager, $messenger, $logger, $entity_type_manager, $event_dispatcher);
    $this->routeMatch = $routeMatch;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_checkout.checkout_order_manager'),
      $container->get('messenger'),
      $container->get('logger.channel.commerce_payment'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function successPage(Request $request, RouteMatchInterface $route_match) {
    return $this->returnPage($request, $route_match);
  }

  /**
   * {@inheritdoc}
   */
  public function failurePage(Request $request, RouteMatchInterface $route_match) {
    return $this->returnPage($request, $route_match);
  }

  /**
   * {@inheritdoc}
   */
  public function returnPage(Request $request, RouteMatchInterface $route_match) {
    $response = NULL;
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $route_match->getParameter('commerce_order');
    $step_id = $route_match->getParameter('step');
    $this->validateStepId($step_id, $order);

    /** @var \Drupal\commerce_order\OrderStorageInterface $order_storage */
    $order_storage = $this->entityTypeManager->getStorage('commerce_order');
    try {
      // Reload the order and mark it for updating, redirecting to step below
      // will save it and free the lock. This must be done before the checkout
      // flow plugin is initiated to make sure that it has the reloaded order
      // object. Additionally, the checkout flow plugin gets the order from
      // the route match object, so update the order there as well with. The
      // passed in route match object is created on-demand in
      // \Drupal\Core\Controller\ArgumentResolver\RouteMatchValueResolver and is
      // not the same object as the current route match service.
      /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
      $order = $order_storage->loadForUpdate($order->id());
      $this->routeMatch->getParameters()->set('commerce_order', $order);
      if ($order->getState()->getId() !== 'draft') {
        // While we waited for the lock, the order state changed.
        // Release our lock, and revalidate the step.
        $order_storage->releaseLock($order->id());
        $this->validateStepId($step_id, $order);
      }
      /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
      $payment_gateway = $order->get('payment_gateway')->entity;
      $payment_gateway_plugin = $payment_gateway->getPlugin();
      if (!$payment_gateway_plugin instanceof NexiGateway) {
        throw new AccessException('The payment gateway for the order does not implement ' . NexiGateway::class);
      }
      /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
      $checkout_flow = $order->get('checkout_flow')->entity;
      $checkout_flow_plugin = $checkout_flow->getPlugin();
      $payment_result_query_param = NULL;
      try {
        $payment_gateway_plugin->onReturn($order, $request);
        $redirect_step_id = $checkout_flow_plugin->getNextStepId($step_id);
      }
      catch (PaymentGatewayException $e) {
        $payment_result_query_param = 'gateway_error';
        $event = new FailedPaymentEvent($order, $payment_gateway, $e);
        $this->eventDispatcher->dispatch($event, PaymentEvents::PAYMENT_FAILURE);
        Error::logException($this->logger, $e);
        $redirect_step_id = $checkout_flow_plugin->getPreviousStepId($step_id);
      }
      catch (\Exception $e) {
        $payment_result_query_param = 'local_error';
        Error::logException($this->logger, $e);
        $redirect_step_id = $checkout_flow_plugin->getPreviousStepId($step_id);
      }
      try {
        // This will always throw a NeedsRedirectException.
        // It will also save the order.
        $checkout_flow_plugin->redirectToStep($redirect_step_id);
      }
      catch (NeedsRedirectException $e) {
        $target_url = $e->getResponse()->getTargetUrl();
        if (!empty($payment_result_query_param)) {
          $target_url .= '?payment_result=' . $payment_result_query_param;
        }
        $response_content = $this->createResponse($target_url);
      }
      $response = new Response($response_content);
    }
    finally {
      $order_storage->releaseLock($order->id());
    }
    return $response;
  }

  /**
   * Create a redirect response to redirect the parent window.
   *
   * @param string $redirect_url
   *   The redirect url.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  protected function createResponse(string $redirect_url) {
    return new Response("
      <!DOCTYPE html>
      <html>
      <head>
          <title>Processing...</title>
      </head>
      <body>
          <p>Processing, please wait...</p>
          <p>If you don't get redirected within 30 seconds, please reload the page</p>
          <script>
              if (window.parent && window.parent !== window) {
                window.parent.postMessage({
                  type: 'payment-redirect',
                  redirect: '{$redirect_url}'
                }, '*');
              }
          </script>
      </body>
      </html>");
  }

  /**
   * {@inheritdoc}
   */
  public function cancelPage(Request $request, RouteMatchInterface $route_match) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $route_match->getParameter('commerce_order');
    $step_id = $route_match->getParameter('step');
    $this->validateStepId($step_id, $order);
    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface $payment_gateway */
    $payment_gateway = $order->get('payment_gateway')->entity;
    $payment_gateway_plugin = $payment_gateway->getPlugin();
    if (!$payment_gateway_plugin instanceof NexiGateway) {
      throw new AccessException('The payment gateway for the order does not implement ' . NexiGateway::class);
    }
    /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
    $checkout_flow = $order->get('checkout_flow')->entity;
    $checkout_flow_plugin = $checkout_flow->getPlugin();

    $payment_gateway_plugin->onCancel($order, $request);
    $previous_step_id = $checkout_flow_plugin->getPreviousStepId($step_id);
    try {
      $checkout_flow_plugin->redirectToStep($previous_step_id);
    }
    catch (NeedsRedirectException $e) {
      $response_content = $this->createResponse($e->getResponse()->getTargetUrl());
    }
    $response = new Response($response_content);
    $response->sendContent();
    exit;
  }

  /**
   * Validates the requested step ID.
   *
   * Redirects to the actual step ID if the requested one is no longer
   * available. This can happen if payment was already cancelled, or if the
   * payment "notify" endpoint created the payment and placed the order
   * before the customer returned to the site.
   *
   * @param string $requested_step_id
   *   The requested step ID, usually "payment".
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   *
   * @throws \Drupal\commerce\Response\NeedsRedirectException
   */
  protected function validateStepId($requested_step_id, OrderInterface $order) {
    $step_id = $this->checkoutOrderManager->getCheckoutStepId($order);
    if ($requested_step_id != $step_id) {
      throw new NeedsRedirectException(Url::fromRoute('commerce_checkout.form', [
        'commerce_order' => $order->id(),
        'step' => $step_id,
      ])->toString());
    }
  }

}
