<?php

namespace Drupal\commerce_nexi;

use phpseclib3\Crypt\Blowfish;

/**
 * Cryptographic service.
 *
 * Used for encrypting and decrypting communication with Nexi.
 */
class CryptographicService {

  /**
   * The encryption key.
   *
   * @var string
   */
  private string $encryptionKey = '';

  /**
   * The HMAC key.
   *
   * @var string
   */
  private string $hmacKey = '';

  /**
   * Sets the encryption keys.
   *
   * @param string $encryptionKey
   *   The encryption key.
   * @param string $hmacKey
   *   The HMAC key.
   */
  public function init(string $encryptionKey, string $hmacKey) {
    $this->encryptionKey = $encryptionKey;
    $this->hmacKey = $hmacKey;
  }

  /**
   * Encrypts string with Blowfish.
   *
   * @param string $input
   *   The plain string.
   *
   * @return string
   *   The encrypted string.
   *
   * @throws \Drupal\commerce_nexi\CryptographicException
   */
  public function encrypt(string $input): string {
    $block_size = 8;
    $len = strlen($input);
    $padding_length = intval(($len + $block_size - 1) / $block_size) * $block_size - $len;
    $padding = str_repeat("\0", $padding_length);
    $data = $input . $padding;

    $blowfish = new Blowfish('ecb');
    $blowfish->disablePadding();
    $blowfish->setKey($this->encryptionKey);

    try {
      $encrypted = $blowfish->encrypt($data);
    }
    catch (\Exception $e) {
      $encrypted = FALSE;
    }

    if ($encrypted === FALSE) {
      throw new CryptographicException(openssl_error_string());
    }
    return bin2hex($encrypted);
  }

  /**
   * Decrypts a blowfish encrypted string.
   *
   * @param string $input
   *   The encrypted hexadecimal string.
   *
   * @return string
   *   The decrypted string.
   *
   * @throws \Drupal\commerce_nexi\CryptographicException
   */
  public function decrypt(string $input): string {
    $blowfish = new Blowfish('ecb');
    $blowfish->disablePadding();
    $blowfish->setKey($this->encryptionKey);
    try {
      $decrypted = $blowfish->decrypt(hex2bin($input));
    }
    catch (\Exception $e) {
      $decrypted = FALSE;
    }

    if ($decrypted === FALSE) {
      throw new CryptographicException(openssl_error_string());
    }

    return rtrim($decrypted, "\0");
  }

  /**
   * Produces a HMAC-hashed string.
   *
   * @param string $payId
   *   The payment id.
   * @param string $transactionId
   *   The transaction id.
   * @param string $merchantId
   *   The merchant id.
   * @param string $amount
   *   The amount.
   * @param string $currency
   *   The currency.
   *
   * @return string
   *   The resulting hash.
   */
  public function calculateHash(string $payId, string $transactionId, string $merchantId, string $amount, string $currency) {
    return hash_hmac(
      'sha256',
      $payId . '*' . $transactionId . '*' . $merchantId . '*' . $amount . '*' . $currency,
      $this->hmacKey,
    );
  }

}
