<?php

namespace Drupal\commerce_nexi\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\Payment;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethod;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\DeclineException;
use Drupal\commerce_payment\Exception\HardDeclineException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsRefundsInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsStoredPaymentMethodsInterface;
use Drupal\commerce_price\Entity\CurrencyInterface;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * The nexi gateway plugin.
 *
 * @CommercePaymentGateway(
 *   id = "nexi",
 *   label = "Nexi (hosted form)",
 *   display_label= "Nexi",
 *   payment_method_types = {"nexi_credit_card"},
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_nexi\PluginForm\InlineWidget\PaymentMethodAddForm"
 *   },
 *   modes = {
 *     "live" = @Translation("Live"),
 *     "test_allowed" = @Translation("Test (allowed)"),
 *     "test_denied" = @Translation("Test (denied)")
 *   },
 *   requires_billing_information = FALSE,
 * )
 */
class NexiGateway extends OffsitePaymentGatewayBase implements OffsitePaymentGatewayInterface, SupportsStoredPaymentMethodsInterface, SupportsRefundsInterface {

  /**
   * Computop Paygate base url.
   */
  public const COMPUTOP_BASE_URL = 'https://www.computop-paygate.com';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The rounder.
   *
   * @var \Drupal\commerce_price\RounderInterface
   */
  protected $rounder;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The Nexi cryptographic service.
   *
   * @var \Drupal\commerce_nexi\CryptographicService
   */
  protected $cryptographicService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->httpClient = $container->get('http_client');
    $instance->rounder = $container->get('commerce_price.rounder');
    $instance->eventDispatcher = $container->get('event_dispatcher');
    $instance->currentUser = $container->get('current_user');
    $instance->cryptographicService = $container->get('commerce_nexi.cryptographic_service');
    $instance->cryptographicService->init($instance->getConfiguration()['encryption_key'], $instance->getConfiguration()['hmac_key']);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'merchant_id' => '',
      'rest_api_key' => '',
      'encryption_key' => '',
      'hmac_key' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['merchant_id'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('Merchant ID'),
      '#description' => $this->t('The Nexi merchant ID. Only needed if you intend to replace the saved value.'),
      '#default_value' => !empty($this->configuration['merchant_id']) ? '****' : '',
      '#required' => FALSE,
    ];
    $form['rest_api_key'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('REST API key'),
      '#description' => $this->t('Needed for REST API authentication. Only needed if you intend to replace the saved value.'),
      '#default_value' => !empty($this->configuration['rest_api_key']) ? '****' : '',
      '#required' => FALSE,
    ];
    $form['encryption_key'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('Encryption key'),
      '#description' => $this->t('Needed for Blowfish encryption. Only needed if you intend to replace the saved value.'),
      '#default_value' => !empty($this->configuration['encryption_key']) ? '****' : '',
      '#required' => FALSE,
    ];
    $form['hmac_key'] = [
      '#type' => 'textfield',
      '#maxlength' => 255,
      '#title' => $this->t('HMAC Key'),
      '#description' => $this->t('Needed for HMAC hashing. Only needed if you intend to replace the saved value.'),
      '#default_value' => !empty($this->configuration['hmac_key']) ? '****' : '',
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $existing_configuration = $this->configuration;
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      foreach (['merchant_id', 'rest_api_key', 'encryption_key', 'hmac_key'] as $key) {
        if (!empty($values[$key])) {
          // Overwrite values only if different or empty.
          if ($values[$key] === '****') {
            $this->configuration[$key] = $existing_configuration[$key];
          }
          else {
            $this->configuration[$key] = $values[$key];
          }
        }
      }
    }
  }

  /**
   * Get the merchant id.
   */
  public function getMerchantId() {
    return $this->configuration['merchant_id'];
  }

  /**
   * Get the rest api key.
   */
  private function getRestApiKey() {
    return $this->configuration['rest_api_key'];
  }

  /**
   * Provides the iframe url to use as src in the frontend.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order object.
   *
   * @return string
   *   The iframe url.
   *
   * @throws \Drupal\commerce_nexi\CryptographicException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getPaymentIframe(OrderInterface $order): string {
    $transaction_id = $this->getUniqueTransactionId($order);
    $purchase_amount = $this->minorUnitsConverter->toMinorUnits($order->getBalance());
    $currency = $this->getCurrency($order);
    $request_values = [
      'MerchantId' => $this->getMerchantId(),
      'MsgVer' => '2.0',
      'TransID' => $transaction_id,
      'Amount' => $purchase_amount,
      'Currency' => $currency,
      'URLSuccess' => $this->getSuccessUrl($order),
      'URLFailure' => $this->getFailureUrl($order),
      'URLBack' => $this->getCancelUrl($order),
      'URLNotify' => $this->getNotifyUrl($order)->toString(),
      'Response' => 'encrypt',
      'MAC' => $this->cryptographicService->calculateHash(
        '',
        $transaction_id,
        $this->getMerchantId(),
        $purchase_amount,
        $currency,
      ),
    ];
    if ($this->getMode() === 'test_allowed') {
      // Simulate a successful payment.
      $request_values['OrderDesc'] = 'Test:0000';
    }
    elseif ($this->getMode() === 'test_denied') {
      // Simulate a failed payment.
      $request_values['OrderDesc'] = 'Test:0305';
    }

    $request_data = $this->massageRequestValues($request_values);
    $len = strlen($request_data);
    $encrypted = $this->cryptographicService->encrypt($request_data);
    return self::COMPUTOP_BASE_URL . '/payssl.aspx?' . $this->massageRequestValues([
      'MerchantID' => $this->getMerchantId(),
      'Len' => $len,
      'Data' => $encrypted,
      'Template' => 'ct_responsive',
      'Language' => $this->currentUser->getPreferredLangcode(),
    ]);
  }

  /**
   * Creates a statistically unique transaction id.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order object.
   *
   * @return string
   *   The unique transaction id.
   *
   * @throws \Random\RandomException
   */
  private function getUniqueTransactionId(OrderInterface $order): string {
    $order_id = $order->id() . '-';
    return $order_id . substr(bin2hex(random_bytes(16)), 0, 16 - strlen($order_id));
  }

  /**
   * Gets the currency code of the order.
   *
   * @return string
   *   The currency code.
   */
  protected function getCurrency(OrderInterface $order): string {
    $currency = $order->getTotalPrice()->getCurrencyCode();
    $currency = $this->entityTypeManager->getStorage('commerce_currency')->loadByProperties(['currencyCode' => $currency]);
    $currency = reset($currency);
    if ($currency instanceof CurrencyInterface) {
      return $currency->getCurrencyCode();
    }
    throw new \Exception('Undefined currency');
  }

  /**
   * Turns a keyed array into a &key=value url-like string.
   *
   * @param array $values
   *   The values.
   *
   * @return string
   *   The resulting string.
   */
  private function massageRequestValues(array $values) {
    return implode('&', array_map(function ($key, $value) {
      return $key . '=' . $value;
    }, array_keys($values), $values));
  }

  /**
   * {@inheritdoc}
   */
  public function getSuccessUrl(OrderInterface $order) {
    return Url::fromRoute('commerce_nexi.checkout.success', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function getFailureUrl(OrderInterface $order) {
    return Url::fromRoute('commerce_nexi.checkout.failure', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(OrderInterface $order) {
    return Url::fromRoute('commerce_nexi.checkout.cancel', [
      'commerce_order' => $order->id(),
      'step' => 'payment',
    ], ['absolute' => TRUE])->toString();
  }

  /**
   * {@inheritdoc}
   */
  public function onSuccess(OrderInterface $order, array $response) {
    // Create payment entity.
    $payment = Payment::create([
      'state' => 'completed',
      'amount' => new Price($order->getTotalPrice()->getNumber(), $order->getTotalPrice()->getCurrencyCode()),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => json_encode([
        'PayID' => $response['PayID'] ?? '',
        'TransID' => $response['TransID'] ?? '',
      ]),
      'remote_state' => $response['Status'] ?? 'fail',
      'authorized' => REQUEST_TIME,
      'completed' => REQUEST_TIME,
    ]);
    $payment->save();

    // Create payment method entity.
    if (!empty($response['card'])) {
      $order = $payment->getOrder();
      $payment_method = $this->entityTypeManager->getStorage('commerce_payment_method')->loadByProperties([
        'card_number' => \substr($response['maskedpan'] ?? '****', -4),
        'uid' => $order->getCustomer()->id(),
      ]);
      $payment_method = \reset($payment_method);
      if (!$payment_method instanceof PaymentMethodInterface) {
        $data = [
          'type' => 'nexi_credit_card',
          'billing_profile' => $order->getBillingProfile(),
          'payment_gateway' => $this->parentEntity->id(),
          'uid' => $order->getCustomer()->id(),
          'remote_id' => json_encode([
            'PayID' => $response['PayID'] ?? '',
            'TransID' => $response['TransID'] ?? '',
          ]),
          'card_type' => $response['CCBrand'] ?? '',
          'card_number' => \substr($response['maskedpan'] ?? '****', -4),
          'card_exp_month' => substr($response['CCExpiry'] ?? '00000', 4, 2),
          'card_exp_year' => substr($response['CCExpiry'] ?? '00000', 0, 4),
        ];
        $payment_method = PaymentMethod::create($data);

        // @todo Handle recurring value when implementing COF.
        $payment_method->get('recurring')->setValue(FALSE);

        $payment_method->save();
      }
      $order->set('payment_method', $payment_method);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onFailure(OrderInterface $order, array $response) {
    $payment = Payment::create([
      'state' => 'failed',
      'amount' => new Price($order->getTotalPrice()->getNumber(), $order->getTotalPrice()->getCurrencyCode()),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => json_encode([
        'PayID' => $response['PayID'] ?? '',
        'TransID' => $response['TransID'] ?? '',
      ]),
      'remote_state' => $response['Status'] ?? 'fail',
      'authorized' => REQUEST_TIME,
    ]);
    $payment->save();
    throw new DeclineException();
  }

  /**
   * Handles the return from the payment provider.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   */
  public function onReturn(OrderInterface $order, Request $request) {
    switch ($request->getMethod()) {
      case 'POST':
        // 3DS flows will POST data back here.
        $body = $request->getContent();
        $body = $this->queryParamsToArray($body);
        $data = $body['Data'] ?? NULL;
        break;

      case 'GET':
        // The other flows will use a GET redirect.
        $query_params = $request->query->all();
        $data = $query_params['Data'] ?? NULL;
        break;

      default:
        throw new \Exception('Unsupported method');
    }
    $decrypted = $this->cryptographicService->decrypt($data);
    $variables = $this->queryParamsToArray($decrypted);
    if (!isset($variables['Status'])) {
      throw new PaymentGatewayException('Invalid response from payment gateway');
    }

    switch ($variables['Code']) {
      case '00000000':
        $this->onSuccess($order, $variables);
        break;

      default:
        $this->onFailure($order, $variables);
        break;
    }

  }

  /**
   * Turns a string of query parameters into a key-value array.
   *
   * @param string $input
   *   The input.
   *
   * @return array
   *   The result.
   */
  protected function queryParamsToArray(string $input): array {
    $result = [];
    foreach (explode('&', $input) as $param) {
      $parts = explode('=', $param, 2);
      if (count($parts) === 2) {
        if (in_array(strtolower($parts[0]), ['card', 'threedsdata', 'resultsresponse'])) {
          $result[$parts[0]] = json_decode(base64_decode($parts[1]), TRUE);
        }
        else {
          $result[$parts[0]] = $parts[1];
        }
      }
    }
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request) {
    // Display a message indicating the customer canceled payment.
    $this->messenger()->addMessage($this->t('You have canceled payment but may resume the checkout process here when you are ready.'));

    // Remove the payment information from the order data array.
    $order->unsetData('commerce_nexi');
  }

  /**
   * Attempt to validate payment information according to a payment state.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment to validate.
   * @param string|null $payment_state
   *   The payment state to validate the payment for.
   */
  protected function validatePayment(PaymentInterface $payment, $payment_state = 'new') {
    $this->assertPaymentState($payment, [$payment_state]);

    $payment_method = $payment->getPaymentMethod();
    if (empty($payment_method)) {
      throw new \InvalidArgumentException('The provided payment has no payment method referenced.');
    }

    switch ($payment_state) {
      case 'new':
        if ($payment_method->isExpired()) {
          throw new HardDeclineException('The provided payment method has expired.');
        }

        break;

      case 'authorization':
        if ($payment->isExpired()) {
          throw new \InvalidArgumentException('Authorizations are guaranteed for up to 29 days.');
        }
        if (empty($payment->getRemoteId())) {
          throw new \InvalidArgumentException('Could not retrieve the transaction ID.');
        }
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL) {}

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details) {}

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    // Notify support not needed for this offsite payment method.
    // We don't use webhooks.
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    // Get the Nexi transaction ID from the payment.
    $remote_id = $payment->getRemoteId();
    $decoded_state = $remote_id ? json_decode($remote_id, TRUE) : [];
    $transaction_id = $decoded_state['TransID'] ?? '';
    $remote_id = $decoded_state['PayID'] ?? '';
    if (empty($transaction_id)) {
      throw new PaymentGatewayException('Cannot refund payment: missing TransID');
    }
    if (empty($remote_id)) {
      throw new PaymentGatewayException('Cannot refund payment: missing PayID');
    }

    try {
      $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
      // Refund the entire amount.
      $amount = $amount ?: $payment->getAmount();
      $this->assertRefundAmount($payment, $amount);

      // Perform the refund request here, throw an exception if it fails.
      // See \Drupal\commerce_payment\Exception for the available exceptions.
      try {
        $body = [
          'transId' => $transaction_id,
          'amount' => [
            'value' => $this->minorUnitsConverter->toMinorUnits($amount),
            'currency' => $amount->getCurrencyCode(),
          ],
        ];
        if ($this->getMode() === 'test_allowed') {
          // Simulate a successful refund.
          $body['simulationMode'] = 'Test:0000';
        }
        elseif ($this->getMode() === 'test_denied') {
          // Simulate a failed refund.
          $body['simulationMode'] = 'Test:0305';
        }
        $response = $this->httpClient->request(
          'POST',
          self::COMPUTOP_BASE_URL . "/api/v2/payments/$remote_id/refunds",
          [
            RequestOptions::HEADERS => [
              'Content-Type' => 'application/json',
              'Authorization' => 'Basic ' . base64_encode($this->getMerchantId() . ':' . $this->getRestApiKey()),
            ],
            RequestOptions::JSON => $body,
          ],
        );
        $status = $response->getStatusCode();
        $content = json_decode($response->getBody()->getContents(), TRUE);
      }
      catch (\Exception $exception) {
        $e = new PaymentGatewayException($exception->getMessage());
        throw $e;
      }
      if ($status >= 200 && $status < 300 && $content['responseCode'] === '00000000') {
        $old_refunded_amount = $payment->getRefundedAmount();
        $new_refunded_amount = $old_refunded_amount->add($amount);
        if ($new_refunded_amount->lessThan($payment->getAmount())) {
          $payment->setState('partially_refunded');
        }
        else {
          $payment->setState('refunded');
        }

        $payment->setRefundedAmount($new_refunded_amount);
        $payment->save();
      }
      else {
        throw new \Exception();
      }
    }
    catch (\Exception $e) {
      throw new PaymentGatewayException('Refund failed: ' . $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->validatePayment($payment, 'new');

  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    $payment_method->delete();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    $this->validatePayment($payment, 'authorization');

    // Get the Nexi transaction ID from the payment.
    $transaction_id = $payment->getRemoteId();
    if (empty($transaction_id)) {
      throw new PaymentGatewayException('Cannot void payment: missing transaction ID');
    }

    try {

    }
    catch (\Exception $e) {
      throw new PaymentGatewayException('Void failed: ' . $e->getMessage());
    }
  }

}
