<?php

namespace Drupal\commerce_nexi\PluginForm\InlineWidget;

use Drupal\commerce_nexi\Plugin\Commerce\PaymentGateway\NexiGateway;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding a payment method.
 */
class PaymentMethodAddForm extends PaymentOffsiteForm implements ContainerInjectionInterface {

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new PaymentMethodAddForm object.
   */
  public function __construct(RouteMatchInterface $route_match, EntityTypeManagerInterface $entity_type_manager, MessengerInterface $messenger, LoggerInterface $logger) {
    $this->routeMatch = $route_match;
    $this->entityTypeManager = $entity_type_manager;
    $this->messenger = $messenger;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_route_match'),
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('logger.channel.commerce_payment'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $this->routeMatch->getParameter('commerce_order');
    if (!$order instanceof OrderInterface) {
      throw new PaymentGatewayException('Could not start payment process.');
    }

    $balance = $order->getBalance();
    if ($balance->isNegative() || $balance->isZero()) {
      // A sanity check to not overbill.
      throw new PaymentGatewayException('Could not start payment process. Please try again or contact customer support if the error persists.');
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $this->entity;
    $gateway = $payment_method->getPaymentGateway();
    if (!$gateway instanceof PaymentGatewayInterface || !$gateway->getPlugin() instanceof NexiGateway) {
      throw new PaymentGatewayException('Could not start payment process.');
    }
    /** @var \Drupal\commerce_nexi\Plugin\Commerce\PaymentGateway\NexiGateway $nexi_gateway */
    $nexi_gateway = $gateway->getPlugin();
    $payment_iframe = $nexi_gateway->getPaymentIframe($order);
    // We have an iframe to show.
    // This is the second load after the first submission.
    $form['payment_iframe'] = [
      '#type' => 'inline_template',
      '#template' => '<iframe src="{{ url }}" width="100%" height="1000px" frameborder="0"></iframe>',
      '#context' => [
        'url' => $payment_iframe,
      ],
    ];

    $form['#attached']['library'][] = 'commerce_nexi/nexi_checkout';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Nothing happens on submit.
  }

}
