<?php

namespace Drupal\commerce_opp_webhooks\Plugin\AdvancedQueue\JobType;

use Drupal\advancedqueue\Job;
use Drupal\advancedqueue\JobResult;
use Drupal\advancedqueue\Plugin\AdvancedQueue\JobType\JobTypeBase;
use Drupal\commerce_opp\BrandRepositoryInterface;
use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\PaymentMethodStorageInterface;
use Drupal\commerce_payment\PaymentStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the job type for processing OPP webhook infos.
 *
 * @AdvancedQueueJobType(
 *   id = "commerce_opp_webhooks",
 *   label = @Translation("Process OPP webhook infos"),
 *   max_retries = 3,
 *   retry_delay = 1800,
 * )
 */
class ProcessOppWebhook extends JobTypeBase implements ContainerFactoryPluginInterface {

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The payment storage.
   *
   * @var \Drupal\commerce_payment\PaymentStorageInterface
   */
  protected PaymentStorageInterface $paymentStorage;

  /**
   * The payment method storage.
   *
   * @var \Drupal\commerce_payment\PaymentMethodStorageInterface
   */
  protected PaymentMethodStorageInterface $paymentMethodStorage;

  /**
   * The brand repository.
   *
   * @var \Drupal\commerce_opp\BrandRepositoryInterface
   */
  protected BrandRepositoryInterface $brandRepository;

  /**
   * Constructs a new ProcessOppWebhook object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_channel_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\commerce_opp\BrandRepositoryInterface $brand_repository
   *   The brand repository.
   */
  public function __construct(array $configuration, string $plugin_id, $plugin_definition, LoggerChannelFactoryInterface $logger_channel_factory, EntityTypeManagerInterface $entity_type_manager, BrandRepositoryInterface $brand_repository) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->logger = $logger_channel_factory->get('commerce_opp_webhooks');
    $this->paymentStorage = $entity_type_manager->getStorage('commerce_payment');
    $this->paymentMethodStorage = $entity_type_manager->getStorage('commerce_payment_method');
    $this->brandRepository = $brand_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('commerce_opp.brand_repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function process(Job $job) {
    $payload = $job->getPayload();
    $payment_id = (int) $payload['payment_id'];
    $remote_id = $payload['remote_id'];
    $remote_state = $payload['remote_state'];
    $transition_id = $payload['transition_id'];
    $brand_id = $payload['brand'];
//    $price = $payload['price'];
//    if (is_array($price)) {
//      $price = Price::fromArray($price);
//    }

    $payment = $this->paymentStorage->load($payment_id);
    assert($payment instanceof PaymentInterface);
    $drupal_payment_state = $payment->getState()->getId();

    if ($payment->isCompleted() || $drupal_payment_state === 'completed') {
      $msg = sprintf('Skip cron processing of already completed payment ID %s', $payment_id);
      return JobResult::success($msg);
    }
    if ($drupal_payment_state === 'authorization' && $transition_id !== 'capture') {
      $msg = sprintf('Skip cron processing of authorized payment ID %s that is in %s state', $payment_id, $drupal_payment_state);
      return JobResult::success($msg);
    }

    if ($drupal_payment_state === 'authorization_voided') {
      if ($payload['from_state'] !== $drupal_payment_state) {
        $msg = sprintf('Skip cron processing of voided payment ID %s, as the state at time of receiving the webhook call was %s', $payment_id, $payload['from_state']);
        return JobResult::success($msg);
      }
      if ($transition_id === 'void') {
        $msg = sprintf('Skip cron processing of payment ID %s, as it is already voided.', $payment_id);
        return JobResult::success($msg);
      }
    }

    if (!$payment->getState()->isTransitionAllowed($transition_id)) {
      $msg = sprintf('Transition %s is not allowed for payment ID %s (current state: %s)', $transition_id, $payment->id(), $payment->getState()->getId());
      return JobResult::failure($msg);
    }

    $order = $payment->getOrder();
    $payment_method = $payment->getPaymentMethod();
    if (empty($payment_method)) {
      // Create a payment method.
      $opp_gateway = $payment->getPaymentGateway()->getPlugin();
      $payment_method = $this->paymentMethodStorage->createForCustomer(
        $opp_gateway->getDefaultPaymentMethodType()->getPluginId(),
        $payment->getPaymentGatewayId(),
        $order->getCustomerId(),
        $order->getBillingProfile()
      );

      if ($payment_method->hasField('card_type')) {
        $brand = !empty($brand_id) ? $this->brandRepository->getBrand($brand_id) : NULL;
        if ($brand) {
          if ($card_type = $brand->getCommerceId()) {
            $payment_method->set('card_type', $card_type);
          }
        }
        if ($card_info = $payload['card_info']) {
          $payment_method->set('card_number', $card_info['last4Digits']);
          $payment_method->set('card_exp_month', $card_info['expiryMonth']);
          $payment_method->set('card_exp_year', $card_info['expiryYear']);
          $expires = CreditCard::calculateExpirationTimestamp($payment_method->card_exp_month->value, $payment_method->card_exp_year->value);
          $payment_method->setExpiresTime($expires);
        }
      }
      if ($payment_method->hasField('paypal_mail')) {
        if (!empty($payload['virtual_account_info']['accountId'])) {
          $payment_method->set('paypal_mail', $payload['virtual_account_info']['accountId']);
        }
      }
    }
    $registration_id = $payload['registration_id'];
    $payment_method->setReusable(!empty($registration_id));
    if ($registration_id) {
      $payment_method->setRemoteId($registration_id);
    }
    if ($payment_method->hasField('opp_brand') && $payment_method->get('opp_brand')->isEmpty()) {
      $payment_method->set('opp_brand', $payment->get('opp_brand')->getString());
    }
    $payment_method->save();
    $payment->set('payment_method', $payment_method->id());

    $payment->getState()->applyTransitionById($transition_id);
    if (empty($payment->getRemoteId())) {
      $payment->setRemoteId($remote_id);
    }
    if (!empty($brand_id) && $payment->hasField('opp_brand') && $payment->get('opp_brand')->isEmpty()) {
      $payment->set('opp_brand', $brand_id);
    }
    if ($transition_id === 'authorize') {
      $payment->setExpiresTime(0);
    }
    $payment->setRemoteState($remote_state);
    $payment->save();

    $messages = [];
    if ($transition_id === 'authorize') {
      $order = $payment->getOrder();
      if ($order->getState()->isTransitionAllowed('place')) {
        $order->getState()->applyTransitionById('place');
        // A placed order should never be locked.
        $order->unlock();
        $order->save();
        $messages[] = 'Placed order ' . $order->id();
      }
      else {
        $msg = sprintf('Cannot place order for payment %s (authorized by webhook), order state transition is not allowed!', $payment->id());
        $this->logger->error($msg);
        $messages[] = $msg;
      }
    }
    $msg = sprintf('Successfully processed transaction status info of payment ID %s', $payment_id);
    $messages[] = $msg;
    return JobResult::success(implode(' | ', $messages));
  }

}
